
-- This an implementation of the canonic labeling of graphs + automorphism group.
-- J.P. Bernardy -- 2003

-- The implementation is based on

-- Brendan D. McKay, PRACTICAL GRAPH ISOMORPHISM,
-- in Congressus Numerantium, Vol. 30 (1981), pp. 45-87.

-- The following source code is distributed under the conditions of the GPL

module Nauty(canonicGraph) where

import Partition
import Data.Graph(Graph, Vertex)
import Data.Array 
import Data.Array.MArray(newArray, writeArray, getElems)
import Data.Array.ST(STArray)
import Data.List(sort, isPrefixOf, intersect)
import Control.Monad.State
import Control.Monad.ST
import Permutation
import Data.STRef
import Nests



firstNoCommon _ [] = Nothing
firstNoCommon [] (v:_) = Just v
firstNoCommon (v1:v1s) (v2:v2s) 
    | v1 == v2 = firstNoCommon v1s v2s
    | otherwise = Just v2


maybeElem Nothing l  = True
maybeElem (Just v) l = v `elem` l

-- tells if l1 is included in l2
included :: Eq a => [a] -> [a] -> Bool
l1 `included` l2 = all (`elem` l2) l1

-- relabels a graph, given a discrete partition
relabel gr partition = applyPerm simplePermutation gr
    where simplePermutation = array bnds [(v, i) 
					  | v <- map head partition
					  | i <- range bnds]
	  bnds = bounds gr


leftMostNode :: Graph -> Partition -> (Partition, [Indicator], [Vertex])
leftMostNode gr pi1
    | ((v1, pi2):_) <- childPartitions gr pi1
		    = let (nu, ls, path) = leftMostNode gr pi2
			  in (nu, lambda gr pi1 : ls, v1 : path)
    | otherwise = (pi1, [lambda gr pi1], [])


-- nu = current node
-- zeta = 1st terminal node
-- rho = best guess at the node leading to canonical labelling
-- Lambda = indicator function for a node (usually written xLambda)
-- theta = orbit partiton of the automorphism group found yet
-- gamma = automorphism found
-- psi = store for automorphisms (gamma) found, in the form of (fix gamma, mcr gamma)

-- returns the graph relabelled, canonically. See McKay for details.


l :: Int = 50

nauty :: Graph -> ST s ([Permutation], Graph)
nauty gr0 =
    do {
       ;let gr = fmap sort $ gr0
       ;let graphBounds = bounds gr
       ;thetaRef <- newSTRef $ (listArray graphBounds (range graphBounds), range graphBounds)
       ;autGRef <- newSTRef []
       ;let root = initialPartition gr
       ;let (zeta, zetaLambda, zetaPath) = leftMostNode gr root
       ;let grZeta = relabel gr zeta
       ;rhoRef <- newSTRef (zeta, zetaLambda, grZeta)
       ;psi <- (newArray (0, l) undefined :: ST s (STArray s Int ([Vertex], [Vertex]))) 
       ;psiSize <- newSTRef 0
       ;let 
	{
--	 exploreNode :: Partition -> [Vertex] -> [Indicator] -> ST s ();
	 exploreNode nu nuPath nuLambda =
	 do {
	    ;let childNodes = childPartitions gr nu
	    ;(rho, rhoLambda, grRho) <- readSTRef rhoRef
	    ;(_, mcrTheta) <- readSTRef thetaRef
	    ;let
	    {foundTerminalNode =
	     do {
		;let grNu = relabel gr nu
		;(if (nuLambda, grNu) == (zetaLambda, grZeta)
	          then foundAutomorphism (permBetween graphBounds zeta nu)
	          else case compare (nuLambda, grNu) (rhoLambda, grRho) of 
	          { 
		   LT -> writeSTRef rhoRef (nu, nuLambda, grNu);
		   EQ -> foundAutomorphism (permBetween graphBounds rho nu);
		   GT -> return ();
		  }
		 )
		};
	     exploreKid (v, pi) = 
	        exploreNode pi (nuPath ++ [v]) (nuLambda ++ [lambda gr pi]);

--	     exploreSubnodes :: Bool -> [(Vertex, Partition)] -> ST s ();
	     exploreSubnodes first [] = return ();
	     exploreSubnodes first (n@(v, _):ns) = 
	     do {
		;automs0 <- getElems psi
		;sz <- readSTRef psiSize
		-- this is explained on pages 60-61.
		;let automs = take sz automs0
		;let test1 = maybeElem (firstNoCommon zetaPath nuPath) mcrTheta
		;let fixingAutomsMcrs = [m | (f,m) <- automs, nuPath `included` f]
		;let maxFan = foldl1 intersect (splittingCell nu : fixingAutomsMcrs)
		;when (first || (test1 && (v `elem` maxFan))) (exploreKid n)
		;when test1 (exploreSubnodes False ns)
		};
	     
             foundAutomorphism gamma =
	     do {
		 -- update psi
		; sz <- readSTRef psiSize
		; when (sz < l)
		(do {writeSTRef psiSize (sz+1)
		    ;writeArray psi sz (fixed gamma, mcr $ orbitsFromPerm $ gamma)
		    });
		 -- update theta
		;(thetaOld, _) <- readSTRef thetaRef
		;let theta = mergePerms gamma thetaOld
		;writeSTRef thetaRef (theta, mcr $ orbitsFromPerm theta)
		;modifySTRef autGRef (gamma:)
		}
	     
	    };
	    ;when (nuLambda <= rhoLambda || (nuLambda `isPrefixOf` zetaLambda))
		      (if null childNodes 
		      then foundTerminalNode
		      else exploreSubnodes True childNodes)
	    };
        };
       ;exploreNode root [] [lambda gr root]
       ;autG <- readSTRef autGRef
       ;(_,_,canonicGraph) <- readSTRef rhoRef
       ;return (autG, canonicGraph)
       }



canonicGraph graph = runST (nauty graph)
		 
		
