
-- This an implementation of the canonic labeling of graphs + automorphism group.
-- J.P. Bernardy -- 2003

-- The implementation is based on

-- Brendan D. McKay, PRACTICAL GRAPH ISOMORPHISM,
-- in Congressus Numerantium, Vol. 30 (1981), pp. 45-87.

-- The following source code is distributed under the conditions of the GPL


module Partition(Cell, Partition, refine, isSingleton, unitPartition, isDiscrete, mcr,
		 Indicator, lambda, fixedInOrbits) where

import Data.Graph
import Data.List
import Data.Array((!), range, bounds)
import Data.Int

type Cell = [Vertex]
-- with the invariant that the list is sorted

type Partition = [Cell]


isSingleton [x] = True
isSingleton _ = False

unitPartition bnds = [range bnds]

isDiscrete p = all isSingleton p

-- refines a Partition wrt to another Partition
-- explained on pages 50-52.
refine :: Graph -> Partition -> Partition -> Partition
refine gr p [] = p
refine gr p (w:ws) = refine gr p' alpha
  where (p', alpha) = refineCells p ws

	refineCells [] p = ([], p)	
	refineCells (c:cs) p = (rc ++ rcs, xxp)
	  where (rc, xp) = refineCell c p
		(rcs, xxp) = refineCells cs xp

	refineCell c alpha 
         | isSingleton xs = ([c], alpha)
	 | otherwise = (xs, alpha' ++ smallXs)   
	  where 
	        xs = refineCellByOneCell c w
		alpha' = replace (c==) largeXt alpha
		(largeXt, smallXs) = extractLargest xs

-- splits a cell in groups of equal degrees with respect to another cell.
	refineCellByOneCell refinedCell referenceCell = 
	  groupSortWith (degreeCellVertex gr referenceCell) refinedCell

replace :: (a->Bool) -> a -> [a] -> [a]
replace f rep [] = [] 
replace f rep (l:ls) 
  | f l = rep:ls
  | otherwise = l:replace f rep ls

extractLargest list = (largest, before ++ after)
  where (before, (largest:after)) = break hasMaxLength list
	hasMaxLength el = length el == maxLength
	maxLength = maximum $ map length $ list 

groupSortWith key list = map (map snd) $ groupBy fstEq $ sortBy fstComp $ [(key v, v) | v <- list]
    where fstComp x y = compare (fst x) (fst y)
	  fstEq x y = fst x == fst y

mcr partition = map head partition

fixedInOrbits :: Partition -> [Vertex]
fixedInOrbits partition = map head $ filter isSingleton $ partition

isNeighbour :: Graph -> Vertex -> Vertex -> Bool
isNeighbour gr n1 n2 = n2 `elem` gr!n1

-- degree of a cell wrt a node
degreeCellVertex :: Graph -> Cell -> Vertex -> Int
degreeCellVertex gr cell vertex = count (isNeighbour gr vertex) cell
    where count p l = length $ filter p $ l



----------------------------------------
-- The indicator function


type Indicator = Int32

-- An order insensitive hash on lists
oih = product . map (+2)  

lambda :: Graph -> Partition -> Indicator
--lambda gr nu = sort [degreeCellVertex gr c v | c <- nu,  v <- range $ bounds $ gr]
--lambda gr nu = sort [sort $ map (degreeCellVertex gr c) (range $ bounds $ gr) | c <- nu]

lambda gr nu 
    = oih [oih $ map fromIntegral $ map (degreeCellVertex gr c) (range $ bounds $ gr) | c <- nu]
-- the above equation assumes that (*) is commutative on Int32, and never overflows. 

