/*
* Copyright (c) 2012 Nathanael Hübbe
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* 1. Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
* 2. Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
* SUCH DAMAGE.
*/

#define _GNU_SOURCE

#include <dirent.h>
#include <wordexp.h>
#include <stdio.h>
#include <dlfcn.h>

#include "H5private.h"		/* Generic Functions			*/
#include "H5externalModuleLoader.h"
#include "H5Eprivate.h"

//The list of paths that are searched for external modules.
static const char* const kModuleSearchPaths = "/etc/h5modules/ /usr/etc/h5modules/ /usr/local/etc/h5modules/ ~/etc/h5modules/ $H5_EXTERNAL_MODULES_PATH";

//Types for the functions the external modules must provide.
typedef const char* (*H5_XM_get_module_name_t)(void);
typedef herr_t (*H5_XM_get_module_version_t)(int* majorVersion, int* minorVersion, int* releaseNumber);
typedef herr_t (*H5_XM_init_t)(void** privateData);
typedef herr_t (*H5_XM_deinit_t)(void* privateData);

//The object to hold the information associated with one loaded external module.
typedef struct ExternalModule ExternalModule;
struct ExternalModule {
	H5_XM_get_module_name_t getModuleName;
	H5_XM_get_module_version_t getModuleVersion;
	H5_XM_init_t init;
	H5_XM_deinit_t deinit;
	void* dllHandle;
	void* privateData;
	ExternalModule* nextModule;
};

//The global linked list of loaded modules.
static ExternalModule* gModules = NULL;

/*--------------------------------------------------------------------------
 * NAME
 *   H5_load_external_modules -- Search for external modules and load them.
 *
 * USAGE
 *    herr_t H5_load_external_modules()
 *
 * RETURNS
 *    Non-negative on success/Negative on failure
 *
 * DESCRIPTION
 *    Searches for external modules.
 *    See header file for the definition of what an external module is and how they are found.
 *    FIXME: Currently not reentrant. But it's not meant to be called more than once anyway.
 *
 *--------------------------------------------------------------------------
 */
herr_t
H5_load_external_modules(void)
{
    herr_t ret_value = SUCCEED;
	wordexp_t expandedPaths;
	size_t i;
    FUNC_ENTER_NOAPI(H5_load_external_modules, FAIL);

	//Separate the paths and do the tilde/variable expansion.
	if(wordexp(kModuleSearchPaths, &expandedPaths, WRDE_NOCMD))
		HGOTO_ERROR(H5E_FUNC, H5E_CANTINIT, FAIL, "can't expand search paths for external modules")
	//Iterate over the search paths.
	for(i = 0; i < expandedPaths.we_wordc; i++) {
		//Try to open the directory at the current search path.
		const char* const curPath = expandedPaths.we_wordv[i];
		DIR* const moduleDir = opendir(curPath);
		if(moduleDir) {	//Failure to open a module directory is not an error, it just means that we don't have any modules here.
			union {	//Workaround for dirent64 not being guaranteed to be big enough :-(
				struct dirent64 d;
				char b[offsetof (struct dirent64, d_name) + NAME_MAX + 1];
			} u;
			//Iterate over the files in the current directory.
			struct dirent64 *curEntry;
			while(1) {
				//Compute the full path to the current file.
				if(readdir64_r (moduleDir, &u.d, &curEntry)) break;
				if(!curEntry) break;
				if(curEntry->d_name[0] != '.') {	//Ignore invisible files.
					char* curModulePath;
					const char* const modulePathTemplate = (curPath[strlen(curPath)-1] == '/') ? "%s%s" : "%s/%s";
					ExternalModule* moduleStore = (ExternalModule*)malloc(sizeof(ExternalModule));
					if(moduleStore && asprintf(&curModulePath, modulePathTemplate, curPath, curEntry->d_name) >= 0) {
						//Try to open the current file as a dll.
						moduleStore->dllHandle = dlopen(curModulePath, RTLD_LAZY | RTLD_LOCAL);
						if(moduleStore->dllHandle) {	//Failure of dlopen is not an error, it just means that the file is definitely not a module.
							int isModule = 1;
							//Check if this a legal hdf5 module.
							dlerror();	//Reset the internal error variable.
							moduleStore->getModuleName = (H5_XM_get_module_name_t)dlsym(moduleStore->dllHandle, "H5_XM_get_module_name");
							moduleStore->getModuleVersion = (H5_XM_get_module_version_t)dlsym(moduleStore->dllHandle, "H5_XM_get_module_version");
							moduleStore->init = (H5_XM_init_t)dlsym(moduleStore->dllHandle, "H5_XM_init");
							moduleStore->deinit = (H5_XM_deinit_t)dlsym(moduleStore->dllHandle, "H5_XM_deinit");
							if(!dlerror() &&
									moduleStore->getModuleName &&
									moduleStore->getModuleVersion &&
									moduleStore->init &&
									moduleStore->deinit
							) {
								//Init the module.
								if((*moduleStore->init)(&moduleStore->privateData) < 0) {
									isModule = 0;
									fprintf(stderr, "\tError during external module initialization.\n");
								}
							} else isModule = 0;
							if(isModule) {
								//Store the module in a global structure.
								moduleStore->nextModule = gModules;
								gModules = moduleStore;
								moduleStore = NULL;	//So that it won't be freed.
							} else {
								//If the dll is not a valid hdf5 module then close it and forget about it.
								dlclose(moduleStore->dllHandle);
							}
						}
						free(curModulePath);
					}
					if(moduleStore) free(moduleStore);	//Cleanup iff something went wrong.
				}
			}
			closedir(moduleDir);
		}
	}
	wordfree(&expandedPaths);
	{
		int v1, v2, v3;
		if(H5get_external_module_version("Test module", &v1, &v2, &v3) >= 0) {
			printf("Version of the test module is %d.%d.%d\n", v1, v2, v3);
		}
	}

done:
    FUNC_LEAVE_NOAPI(ret_value);
} /* end H5_load_external_modules() */

/*--------------------------------------------------------------------------
 * NAME
 *   H5_unload_external_modules -- Tell all loaded modules to destruct and release all resources associated with them.
 *
 * USAGE
 *    herr_t H5_unload_external_modules()
 *
 * RETURNS
 *    Non-negative on success/Negative on failure
 *
 * DESCRIPTION
 *    Tell all loaded modules to destruct and release all resources associated with them.
 *    FIXME: Currently not reentrant. But it's not meant to be called more than once anyway.
 *
 *--------------------------------------------------------------------------
 */
herr_t
H5_unload_external_modules(void)
{
    herr_t ret_value = SUCCEED;
	ExternalModule* curModule, *temp;
    FUNC_ENTER_NOAPI(H5_unload_external_modules, FAIL);

	curModule = gModules;
	gModules = NULL;
	while(curModule) {
		//Destruct the module.
		(*curModule->deinit)(curModule->privateData);
		//Unload the dll.
		if(dlclose(curModule->dllHandle)) ret_value = FAIL;
		//Destroy the describing object and switch to the next module.
		temp = curModule->nextModule;
		free(curModule);
		curModule = temp;
	}

done:
    FUNC_LEAVE_NOAPI(ret_value);
} /* end H5_unload_external_modules() */

/*--------------------------------------------------------------------------
 * NAME
 *   H5get_loaded_external_modules -- List all currently loaded external modules.
 *
 * USAGE
 *    herr_t H5get_loaded_external_modules(names, count)
 *
 * Parameters:
 *  const char*** names;  OUT: The adress of a string array. The caller is responsible to free only *names, not the individual strings (they are constant).
 *  size_t* count;  OUT: *count will receive the number of elements in the *names array.
 *
 * RETURNS
 *    Non-negative on success/Negative on failure
 *
 * DESCRIPTION
 *    Asks all loaded modules for their names and builds an array of them.
 *    Is as reentrant as the H5_XM_get_module_name() functions provided by the external modules.
 *
 *--------------------------------------------------------------------------
 */
//The reentrancy stems from the facts, that gModules is only accessed once and that there is no function that will ever change an ExternalModule object once it's been added to the linked list. The only thing that could happen theoretically, is that H5_unload_external_modules() is called in parallel, freeing the objects as they are worked upon. But this should not happen, since it's only called atexit.
herr_t
H5get_loaded_external_modules(const char*** names, size_t* count)
{
    herr_t ret_value = SUCCEED;
	ExternalModule* startModule = gModules, *curModule;
	size_t i = 0;

    FUNC_ENTER_API(H5get_loaded_external_modules, FAIL)
	//Count the loaded modules.
	curModule = startModule;
	for(; curModule; curModule = curModule->nextModule) i++;
	//Allocate some mem.
	*count = i;
	if(!(*names = (const char**)malloc(sizeof(char*)*i)))
		HGOTO_ERROR(H5E_STORAGE, H5E_CANTALLOC, FAIL, "malloc failed.")
	//Collect the names.
	curModule = startModule;
	for(; i > 0 && curModule; i--, curModule = curModule->nextModule) {
		(*names)[i-1] = (*curModule->getModuleName)();
	}
done:
    FUNC_LEAVE_API(ret_value)
}   /* end H5get_loaded_external_modules() */

/*--------------------------------------------------------------------------
 * NAME
 *   H5get_external_module_version -- Get the version of a specific module.
 *
 * USAGE
 *    herr_t H5get_external_module_version(name, majorVersion, minorVersion, releaseNumber)
 *
 * Parameters:
 *  const char* name; IN: The name of the module to query.
 *  int* majorVersion; OUT: The major version number.
 *  int* minorVersion; OUT: The minor version number.
 *  int* releaseNumber; OUT: The release number.
 *
 * RETURNS
 *    Non-negative on success/Negative on failure
 *
 * DESCRIPTION
 *    Tell all loaded modules to destruct and release all resources associated with them.
 *    Is as reentrant as the H5_XM_get_module_name() functions provided by the external modules.
 *
 *--------------------------------------------------------------------------
 */
//The reentrancy stems from the facts, that gModules is only accessed once and that there is no function that will ever change an ExternalModule object once it's been added to the linked list. The only thing that could happen theoretically, is that H5_unload_external_modules() is called in parallel, freeing the objects as they are worked upon. But this should not happen, since it's only called atexit.
herr_t
H5get_external_module_version(const char* name, int* majorVersion, int* minorVersion, int* releaseNumber)
{
    herr_t ret_value = SUCCEED;
	ExternalModule* curModule = gModules;

    FUNC_ENTER_API(H5get_loaded_external_modules, FAIL)
	//Iterate through the loaded modules until the correct one is found.
	for(; curModule; curModule = curModule->nextModule) {
		const char* curName = (*curModule->getModuleName)();
		if(curName == name) break;
		if(!strcmp(curName, name)) break;
	}
	//Was it found?
	if(curModule) {
		ret_value = (*curModule->getModuleVersion)(majorVersion, minorVersion, releaseNumber);
	} else {
		ret_value = FAIL;
	}
done:
    FUNC_LEAVE_API(ret_value)
}   /* end H5get_loaded_external_modules() */
