/*
 * Copyright (c) 2012,13 A. Mosca
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - The name of the author may not be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/** @addtogroup pls
 * @{
 */

/**
 * @file	pls.c
 * @brief	Personal ls command.
 */

#include <stdio.h>
#include <getopt.h>
#include <sys/stat.h>
#include <dirent.h>
#include <stdlib.h>
#include <io/console.h>
#include <errno.h>
#include <io/color.h>
#include <io/style.h>
#include <vfs/vfs.h>
#include "pls.h"


static const char *cmdname = "pls";

static uint32_t stat_optmask;  /* for fields that need a stat call */
static uint8_t nostat_optmask; /* for fields that do not need a stat call */
static uint8_t format_optmask;
static uint16_t console_width;
static console_ctrl_t *console;
static uint8_t spacing;
static uint32_t count_dirs, count_files;

static struct option const long_options[] = {
	{ "dsort", no_argument, 0, 'D' },
	{ "expanse", required_argument, 0, 'E' },
	{ "hread", no_argument, 0, 'H' },
	{ "isort", no_argument, 0, 'I' },
	{ "color", no_argument, 0, 'K' },
	{ "nosort", no_argument, 0, 'N' },
	{ "append", no_argument, 0, 'P' },
	{ "quote", no_argument, 0, 'Q' },
	{ "recursive", no_argument, 0, 'R' },
	{ "ssort", no_argument, 0, 'S' },
	{ "width", required_argument, 0, 'W' },
	{ "brief", no_argument, 0, 'b' },
	{ "servid", no_argument, 0, 'd' },
	{ "fshandle", no_argument, 0, 'f' },
	{ "help", no_argument, 0, 'h' },
	{ "index", no_argument, 0, 'i' },
	{ "justcount", no_argument, 0, 'j' },
	{ "lnkcnt", no_argument, 0, 'k' },
	{ "oneline", no_argument, 0, 'n' },
	{ "rsort", no_argument, 0, 'r' },
	{ "size", no_argument, 0, 's' },
	{ "version", no_argument, 0, 'v' },
	{ "line", no_argument, 0, 'x' },
/*	{ "asort", no_argument, 0, 'A' }, */
/* 	{ "csort", no_argument, 0, 'C' }, */
/* 	{ "gsort", no_argument, 0, 'G' }, */
/* 	{ "msort", no_argument, 0, 'M' }, */
/* 	{ "osort", no_argument, 0, 'O' }, */
/* 	{ "timeformat", required_argument, 0, 'T' }, */
/* 	{ "usename", no_argument, 0, 'U' }, */
/* 	{ "atime", no_argument, 0, 'a' }, */
/* 	{ "ctime", no_argument, 0, 'c' }, */
/* 	{ "group", no_argument, 0, 'g' }, */
/* 	{ "mtime", no_argument, 0, 'm' }, */
/* 	{ "owner", no_argument, 0, 'o' }, */
/* 	{ "perm", no_argument, 0, 'p' }, */
	{ 0, 0, 0, 0 }
};


int main(int argc, char **argv)
{
	int c, opt_ind;
	int rc, ret_code;
	char name[MAX_PATH_LEN + 1];

	format_optmask = FMTOPT_COLUMN;
	stat_optmask = SOPT_NAME; /* == 0 */
	nostat_optmask = 0;
	console = NULL;
	console_width = DEFAULT_COLUMNS;
	spacing = DEFAULT_SPACING;
	count_dirs = 0;
	count_files = 0;
	rc =  CMD_SUCCESS;
	ret_code = CMD_SUCCESS;

	for (c = 0, optind = 0, opt_ind = 0; c != -1;) {
		c = getopt_long(argc, argv, "DE:HIKNPQRSW:bdfhijknrsvx",
		    long_options, &opt_ind);

		switch (c) {
		case 'D':
			stat_optmask &= ~SOPT_EXCL;
			stat_optmask |= SOPT_GROUPDIRS;
			break;
		case 'E':
			nostat_optmask |= EOPT_SPACING;
			str_uint8_t(optarg, NULL, 10, true, &spacing);
			break;
		case 'H':
			stat_optmask |= AOPT_READABLE;
			break;
		case 'I':
			stat_optmask &= ~SOPT_EXCL;
			stat_optmask |= (SOPT_INDEX | FOPT_INDEX);
			break;
		case 'K':
			stat_optmask |= AOPT_COLORS;
			break;
		case 'N':
			nostat_optmask |= SOPT_NONE;
			break;
		case 'P':
			stat_optmask |= AOPT_APPENDS;
			break;
		case 'Q':
			nostat_optmask |= AOPT_QUOTING;
			break;
		case 'R':
			stat_optmask |= AOPT_RECURSIVE;
			break;
		case 'S':
			stat_optmask &= ~SOPT_EXCL;
			stat_optmask |= (SOPT_SIZE | FOPT_SIZE);
			break;
		case 'W':
			nostat_optmask |= EOPT_AUTOWIDTH;
			str_uint16_t(optarg, NULL, 10, true, &console_width);
			break;
		case 'b':
			format_optmask = FMTOPT_BRIEF;
			break;
		case 'd':
			stat_optmask |= FOPT_SERVICE_ID;
			break;
		case 'f':
			stat_optmask |= FOPT_FSHANDLE;
			break;
		case 'h':
			help_cmd_pls();
			return CMD_SUCCESS;
		case 'i':
			stat_optmask |= FOPT_INDEX;
			break;
		case 'j':
			format_optmask = FMTOPT_COUNT;
			stat_optmask |= EOPT_COUNT;
			break;
		case 'k':
			stat_optmask |= FOPT_LNKCNT;
			break;
		case 'n':
			format_optmask = FMTOPT_MULTI;
			break;
		case 'r':
			nostat_optmask |= SOPT_REVERSE;
			break;
		case 's':
			stat_optmask |= FOPT_SIZE;
			break;
		case 'v':
			printf("%s: version %s\n", cmdname, PLS_VERSION);
			return CMD_SUCCESS;
		case 'x':
			format_optmask = FMTOPT_LINE;
			break;
		}
	}
	argc -= optind;
	argv += optind;

	sanitize_options();
	get_console();

	if (argc == 0) {

		getcwd(name, MAX_PATH_LEN + 1);
		if (name == NULL) {
			fprintf(stderr, "Can not get cwd");
			drop_console(console);
			return CMD_CRITICAL_FAILURE;
		}

		ret_code = scan_dir(name, false);

	} else {
		struct stat st;

		do {
			str_cpy(name, MAX_PATH_LEN + 1, *argv);

			argv++;

			if (stat(name, &st)) {
				fprintf(stderr, "Can not stat: %s\n", name);
				ret_code = CMD_MINOR_FAILURE;
				continue;
			}

			if (st.is_directory) {
				/*
				 *  If just one arg do not print dirname with
				 * colon.
				 */
				rc = scan_dir(name, argc > 1 ? true : false);
			} else if (st.is_file) {
				rc = scan_file(name);
			}
			ret_code = MAX(ret_code, rc);

		} while (*argv);
	}

	drop_console(console);

	if (format_optmask & FMTOPT_COUNT)
		printf("%" PRIu32 " dir%c    %" PRIu32 " file%c\n",
		    count_dirs, count_dirs == 1 ? ' ' : 's',
		    count_files, count_files == 1 ? ' ' : 's');

	return ret_code;
}


static void help_cmd_pls(void)
{
	static char helpfmt[] =
		"Usage:  %s [OPTION]... [FILE]...\n"
		"  -h, --help       A short option summary\n"
		"  -v, --version    Print version number\n"
		"Format options (default = column format):\n"
		"  -b, --brief      Comma separated entries (overrides -E)\n"
		"  -j, --justcount  Only count files and directories\n"
		"  -n, --oneline    One line per entry format\n"
		"  -x, --line       Line listing format\n"
		"Stat options   (default = name, fields printed in oneline format):\n"
		"  -d, --servid     Print service id\n"
		"  -f, --fshandle   Print file system handle\n"
		"  -i, --index      Print inode number\n"
		"  -k, --lnkcnt     Print number of hard links\n"
		"  -s, --size       Print file size\n"
/*		"  -o, --owner      Print file owner\n" */
/*		"  -g, --group      Print file group\n" */
/*		"  -p, --perm       Print file permissions\n" */
/*		"  -a, --atime      Print last access time\n" */
/*		"  -m, --mtime      Print last modification time\n" */
/*		"  -c, --ctime      Print last status change time\n" */
		"Sort options   (default = name, fields printed in oneline format):\n"
		"  -D, --dsort      Directories first then files\n"
		"  -I, --isort      Sort by inode number\n"
		"  -N, --nosort     Do not sort (overrides all but -r)\n"
		"  -S, --ssort      Sort by file size\n"
		"  -r, --rsort      Reverse sorting order\n"
/*		"  -O, --osort      Sort by file owner\n" */
/*		"  -G, --gsort      Sort by file group\n" */
/*		"  -A, --asort      Sort by last access time\n" */
/*		"  -M, --msort      Sort by last modification time\n" */
/*		"  -C, --csort      Sort by last status change time\n" */
		"Adorn options:\n"
		"  -E, --expanse    Modify spacing between names (1..12)\n"
		"  -H, --hread      Human readable size format (B,KiB,..,EiB)\n"
		"  -K, --color      Use colors to diplay names\n"
		"  -P, --append     Append directory indicator\n"
		"  -Q, --quote      Quote entries\n"
		"  -R, --recursive  List subdirectories recursively\n"
		"  -W, --width      Set terminal width (40..144)\n";
/*	    	"  -T, --timeformat Specify time format\n" */
/*	    	"  -U, --usename    Use names instead of uid/gid\n" */
	printf("`%s' lists files and directories.\n", cmdname);
	printf(helpfmt, cmdname);
}

/** Save the process from incautious users */
static void sanitize_options(void)
{
	/* TODO: check and signal when incompatible options were given */

	if ((console_width < MIN_COLUMNS) || (console_width > MAX_COLUMNS))
		console_width = DEFAULT_COLUMNS;
	if ((spacing < MIN_SPACING) || (spacing > MAX_SPACING))
		spacing = DEFAULT_SPACING;
}

/** Scan (and print) the given dir entry.
 *
 * @param name		Name of the dir to scan.
 * @param type  	True if i have to type entry name (more items, recursive).
 *
 * @return		CMD_SUCCESS on success,
 *			CMD_CRITICAL_FAILURE on opendir or stat failures.
 *
 * This is the main processing cycle.
 *
 */
static int scan_dir(const char *name, bool type)
{
	DIR *dirp;
	struct dirent *dp;
	stat_node_t *head = NULL;
	struct stat st;
	char path[MAX_PATH_LEN + 1];
	int rc = CMD_SUCCESS;

	dirp = opendir(name);
	if (!dirp) {
		fprintf(stderr, "Can not open dir: %s\n", name);
		return CMD_CRITICAL_FAILURE;
	}

	while ((dp = readdir(dirp)) != NULL) {
		snprintf(path, MAX_PATH_LEN, "%s/%s", name, dp->d_name);
		/* Call stat() only if needed */
		if (stat_optmask) {

			if (stat(path, &st)) {
				fprintf(stderr, "Can not stat: %s\n", path);
				closedir(dirp);
				return CMD_CRITICAL_FAILURE;
			}
		}
		head = add_node(head, dp->d_name, &st);
	}
	closedir(dirp);

	/*
	 * At least 2 entries or recursive option: print dir entry with colon.
	 * Skip it if the 'count' option was given.
	 */
	if ((type == true)  && !(format_optmask & FMTOPT_COUNT)) {
		char *cpath;

		cpath = absolutize((char *)name, NULL);
		printf("\n%s:\n", cpath);
	}

	/* Reverse entries if option was given */
	if (nostat_optmask & SOPT_REVERSE)
		head = reverse_nodes(head);

	/* Got all this dir's data, print them and free nodes */
	print_nodes(head);
	free_nodes(head);

	/*  Reprocess current dir if 'recursive' option was given */
	if (stat_optmask & AOPT_RECURSIVE)
		rc = scan_recur_dir(name);

	return rc;
}

/** Quick file scan (and print).
 *
 * Only used for user typed files listing.
 * Do not care to put it first when mixing with user typed dirs, do what
 * a user asked for.
 */
static int scan_file(const char *name)
{
	stat_node_t *head = NULL;
	struct stat st;

	if (stat(name, &st)) {
		fprintf(stderr, "Can not stat: %s\n", name);
		return CMD_CRITICAL_FAILURE;
	}
	head = add_node(head, (char *)name, &st);

	print_nodes(head);
	free_nodes(head);

	return CMD_SUCCESS;
}

/** Add a linked list entry with sorting.
 *
 *
 *
 */
static stat_node_t *add_node(stat_node_t *np, char *name, struct stat *pst)
{
	if (np == NULL) {
		np = calloc (1, sizeof(stat_node_t));
		if (np == NULL) {
			fprintf(stderr, "%s: memory allocation failed: %s:%d\n",
			    cmdname, __func__, __LINE__);
			safe_exit(np, NULL, ENOMEM);
		}
		np->length = str_length(name);
		str_cpy(np->name, str_size(name) + 1, name);
		fill_node(np, pst);
		np->next = NULL;

	} else if (nostat_optmask & SOPT_NONE) {
		/* 'do not sort' option: add a node as it comes */
		np->next = add_node (np->next, name, pst);

	} else { /* which sorting ? */
		bool swap_node = true;

		switch (stat_optmask & SOPT_EXCL) {
		case SOPT_NAME:
			if ((str_cmp(name, np->name)) > 0) {
				np->next = add_node (np->next, name, pst);
				swap_node = false;
			}
			break;
		case SOPT_SIZE:
			if (pst->size < np->size) {
				np->next = add_node (np->next, name, pst);
				swap_node = false;
			} else if (pst->size == np->size) {
				if ( (str_cmp(name, np->name)) > 0) {
					np->next = add_node (np->next, name, pst);
					swap_node = false;
				}
			}
			break;
		case SOPT_INDEX:
			if (pst->index < np->index) {
				np->next = add_node (np->next, name, pst);
				swap_node = false;
			}
			break;
		case SOPT_GROUPDIRS:
			if (pst->is_file && np->is_dir) {
				np->next = add_node (np->next, name, pst);
				swap_node = false;
			} else if ((pst->is_directory && np->is_dir) ||
				   (pst->is_file && !(np->is_dir))) {
				if ( (str_cmp(name, np->name)) > 0) {
					np->next = add_node (np->next, name, pst);
					swap_node = false;
				}
			}
			break;
/*
		case SOPT_OWNER:
			break;
		case SOPT_GROUP:
			break;
		case SOPT_ATIME:
			break;
		case SOPT_MTIME:
			break;
		case SOPT_CTIME:
			break;
*/
		}

		if (swap_node == true) {
			stat_node_t *tmp;

			tmp = np;
			np = calloc (1, sizeof(stat_node_t));
			if (np == NULL) {
				fprintf(stderr, "%s: memory allocation failed: %s:%d\n",
				   cmdname, __func__, __LINE__);
				safe_exit(tmp, NULL, ENOMEM);
			}
			np->length = str_length(name);
			str_cpy(np->name, str_size(name) + 1, name);
			fill_node(np, pst);
			np->next = tmp;
		}
	}
	return np;
}

/** Fill a linked list entry with stat data.
 *
 * @param np		Node pointer.
 * @param pst		Stat structure pointer.
 *
 */
static void fill_node(stat_node_t *np, struct stat *pst)
{
	if (stat_optmask == SOPT_NAME)
		return;

	if (stat_optmask & FOPT_SIZE)
		np->size = pst->size;
	if (stat_optmask & FOPT_INDEX)
		np->index = pst->index;
	if (stat_optmask & FOPT_LNKCNT)
		np->lnkcnt = pst->lnkcnt;
	if (stat_optmask & FOPT_FSHANDLE)
		np->fs_handle = pst->fs_handle;
	if (stat_optmask & FOPT_SERVICE_ID)
		np->service_id = pst->service_id;
	if (stat_optmask & AOPT_APPENDS) {
		np->is_dir = pst->is_directory;
		/* TODO: modify as file types get ready */
		if (pst->is_directory == true)
			np->append = '/';
		else
			np->append = '\0';
	}
	if (stat_optmask & AOPT_COLORS)
		/* TODO: modify as file types get ready */
		np->is_dir = pst->is_directory;
	if (stat_optmask & EOPT_COUNT)
		np->is_dir = pst->is_directory;
	if (stat_optmask & SOPT_GROUPDIRS)
		np->is_dir = pst->is_directory;

/*
	if (stat_optmask & FOPT_OWNER)
	if (stat_optmask & FOPT_GROUP)
	if (stat_optmask & FOPT_PERMS)
	if (stat_optmask & FOPT_ATIME)
	if (stat_optmask & FOPT_MTIME)
	if (stat_optmask & FOPT_CTIME)
*/
}

/** Print linked list entries.
 *
 * @param np		Starting node pointer.
 *
 */
static void print_nodes(stat_node_t *np)
{
	if (np == NULL)
		return;

	if (format_optmask & FMTOPT_COLUMN) {
		int row, rows;
		stat_node_t *head = np;

		rows = calculate_rows(np);
		update_col_view(np, rows);

		while (recalculate_col_view(np, rows)) {
			++rows;
			update_col_view(np, rows);
		}
		for (row = 1; row <= rows; ++row) {
			np = head;
			for (; np != NULL; np = np->next) {
				if (np->row != row)
					continue;
				print_entry(np);
			}
			printf("\n");
		}

	} else if (format_optmask & FMTOPT_BRIEF) {
		int chars;

		for (chars = 0; np != NULL; np = np->next) {
			chars += np->length + extra_chars(np);

			if (chars > console_width) {
				printf("\n");
				chars = np->length + extra_chars(np);
			}
			/* Mark last entry */
			if (np->next == NULL)
				np->column = 1;
			print_entry(np);
		}
		printf("\n");

	} else if (format_optmask & FMTOPT_LINE) {
		int  rows, columns;

		columns = calculate_columns(np);
		update_line_view(np, columns);

		while (recalculate_line_view(np)) {
			--columns;
			update_line_view(np, columns);
		}
		for (rows = 1; np != NULL; np = np->next) {
			/* Row ended, add a newline */
			if (np->row != rows)
				printf("\n");

			print_entry(np);
				rows = np->row;
		}
			printf("\n");

	} else if (format_optmask & FMTOPT_MULTI) {
		uint8_t lind = 0, lsiz = 0;

		if (stat_optmask & FOPT_INDEX)
			lind = longest_index(np);
		if (stat_optmask & FOPT_SIZE && !(stat_optmask & AOPT_READABLE))
			lsiz = longest_size(np);

		for (; np != NULL; np = np->next) {

			if (stat_optmask & FOPT_SERVICE_ID)
				printf("%3zu  ", np->service_id);
			if (stat_optmask & FOPT_FSHANDLE)
				printf("%3" PRId16 "  ", np->fs_handle);
			if (stat_optmask & FOPT_INDEX)
				printf("%*" PRIu32 "  ", lind + 1, np->index);
			if (stat_optmask & FOPT_LNKCNT)
				printf("%3u  ", np->lnkcnt);
			if (stat_optmask & FOPT_SIZE) {
				if (stat_optmask & AOPT_READABLE) {
					char hr_string[MAX_SIZE_STRING] = "";

					readable_size(np->size, hr_string);
					printf("%s  ", hr_string);
				} else {
					printf("%*" PRIu64 " ", lsiz + 1, np->size);
				}
			}
/*
			if (stat_optmask & FOPT_OWNER)
			if (stat_optmask & FOPT_GROUP)
			if (stat_optmask & FOPT_PERMS)
			if (stat_optmask & FOPT_ATIME)
			if (stat_optmask & FOPT_MTIME)
			if (stat_optmask & FOPT_CTIME)
*/
			print_entry(np);
			printf("\n");
		}

	} else if (format_optmask & FMTOPT_COUNT) {
		for (; np != NULL; np = np->next) {
			if (np->is_dir)
				++count_dirs;
			else
				++count_files;
		}
	}
}

/** Free singly linked list entries recursively.
 *
 */
static void free_nodes(stat_node_t *np)
{
	if (np != NULL) {
		free_nodes(np->next);
		free(np);
	}
}

static void free_names(stat_name_t *np)
{
	if (np != NULL) {
		free_names(np->next);
		free(np);
	}
}

/** Get console data if needed.
 *
 */
static void get_console(void)
{
	console = console_init(stdin, stdout);

	/* multiline and autowidth are not set: get console width */
	if (!(format_optmask & FMTOPT_MULTI) &&
	       !(nostat_optmask & EOPT_AUTOWIDTH)) {

		if (console == NULL) {
			fprintf(stderr, "Can not get console (assume %d cols)\n",
			    console_width);
			return;
		}

		int rc;
		sysarg_t rows, cols;

		rc = console_get_size(console, &cols, &rows);
		if (rc != EOK) {
			fprintf(stderr, "Can not get console size (assume %d cols)\n",
			    console_width);
		} else {
			console_width = (uint16_t)cols;
		}
	}
	/* colorize */
	if (stat_optmask & AOPT_COLORS) {
		/* TODO: Save current colors and avoid setting the same fg/bg */

		sysarg_t ccap;
		int rc = console_get_color_cap(console, &ccap);

		if (rc != EOK) {
			fprintf(stderr, "Can not get ccap. disabling colors\n");
			stat_optmask &= ~AOPT_COLORS;
		}
		if (console == NULL) {
			fprintf(stderr, "Can not get console. disabling colors\n");
			stat_optmask &= ~AOPT_COLORS;
		}
	}
}

static void drop_console(console_ctrl_t *con)
{
	if (con != NULL)
		console_done(con);
}

/** Reverse linked list entries.
 *
 */
static stat_node_t *reverse_nodes(stat_node_t *np)
{
	stat_node_t *next, *cur, *res = NULL;

	cur = np;

	while(cur != NULL ) {
		next = cur->next;
		cur->next = res;
		res = cur;
		cur = next;
	}

	return res;
}


/** When fitting lines take into account these too */
static int extra_chars(stat_node_t *np)
{
	int chars = 0;

	if (nostat_optmask & AOPT_QUOTING)
		chars += 2; /* quotes */
	if (stat_optmask & AOPT_APPENDS)
		chars += np->append != '\0';

	if (format_optmask & FMTOPT_BRIEF)
		chars += 2; /* comma + space */
	else
		chars += spacing;

	return chars;
}

/** Format a single entry.
 *
 */
static void print_entry(stat_node_t *np)
{
	if (nostat_optmask & AOPT_QUOTING) {
		printf("\"");
	}
	if (stat_optmask & AOPT_COLORS) {
		/* TODO: modify as file types get ready */
		if (np->is_dir == true) {
			console_flush(console);
			console_set_color(console, COLOR_WHITE, DIR_COLOR, 0);
			printf("%s",  np->name);
			console_flush(console);
			/* TODO Revert to original colors */
			console_set_style(console, STYLE_NORMAL);
		} else {
			printf("%s",  np->name);
		}
	} else { /* No colors, just print name */
		printf("%s",  np->name);
	}
	if (nostat_optmask & AOPT_QUOTING) {
		printf("\"");
	}
	if (stat_optmask & AOPT_APPENDS) {
		/* TODO: modify as file types get ready */
		if (np->append != '\0')
			printf("%c",  np->append);
	}
	if (format_optmask & FMTOPT_BRIEF) {
		/* If it has to be changed modify extra_chars and help */
		if (np->column == 0)
			printf(", ");
	}
	if (np->tab) {
		printf("%*s", np->tab, " ");
	}
}


/*
 * Proceed this way in order to fit lines in column and line format view:
 *
 * Have a fuzzy calculation as a starting point (calculate_{rows,columns})
 * Save data in the node structure (update_{col,line}_view)
 * Loop recalculate_{col,line}_view until no boundaries are crossed
 * updating nodes (update_{col,line}_view) to the next attempt
 *
 *
 */

/** Update row, column members in nodes struct */
static void update_col_view(stat_node_t *np, int rows)
{
	int row, column;

	row = column = 1;

	for (; np != NULL; np = np->next, ++row) {
		if (row > rows) {
			++column;
			row = 1;
		}
		np->row = row;
		np->column = column;
	}
}

/** Find if current column view needs recalculation to fit the screen
 *
 * @return		true if it needs
 * 			false otherwise
 */
static bool recalculate_col_view(stat_node_t *np, int rows)
{
	int chars, longest, max_col, row;
	stat_node_t *head = np;

	max_col = max_columns(np);

	for (row = 1; row <= rows; ++row) {
		np = head;

		for (chars = 0; np != NULL; np = np->next) {
			if (np->row == row) {
				longest = longest_col(head, np->column);
				chars += longest + extra_chars(np);

				/* Last row entry has no tabbing */
				if (np->column == max_col)
					np->tab = 0;
				else
					np->tab = longest - np->length -
					    (np->append != 0) + spacing;
			}
			/* Verify boundary */
			if (chars > (int)console_width)
				return true;
		}
	}

	return false;
}

static int max_columns(stat_node_t *np)
{
	int cols;

	for (cols = 1; np != NULL; np = np->next) {
		if (np->column > cols)
			cols = np->column;
	}

	return cols;
}

static int longest_col(stat_node_t *np, int column)
{
	int chars, size;

	for (chars = 0; np != NULL; np = np->next) {
		if (np->column == column) {
			size = np->length + (np->append != 0);
			if (size > chars)
				chars = size;
		}
	}

	return chars;
}

/** Update row, column members in nodes struct */
static void update_line_view(stat_node_t *np, int cols)
{
	int row, column, names;

	row = column = names = 1;

	for (; np != NULL; np = np->next, ++names, ++column) {
		if (names > cols) {
			++row;
			column = names = 1;
		}
		np->row = row;
		np->column = column;
	}
}

/** Find if current line view needs recalculation to fit the screen
 *
 * @return		true if it needs
 * 			false otherwise
 */
static bool recalculate_line_view(stat_node_t *np)
{
	int chars, longest, max_col, row = 1;
	stat_node_t *head = np;

	max_col = max_columns(np);

	for (chars = 0; np != NULL; np = np->next) {
		longest = longest_col(head, np->column);
		chars +=  longest + extra_chars(np);

		/* verify boundary */
		if ((chars > (int)console_width) && (np->row == row)) {
			return true;
		} else if (np->row != row) {
			chars = 0;
		}

		/* Last line entry has no tabbing */
		if (np->column == max_col)
			np->tab = 0;
		else
			np->tab = longest - np->length -
			    (np->append != 0) + spacing;

		row = np->row;
	}
	return false;
}


/** Underestimated, precise measurements will
 * come later.
 */
static int calculate_rows(stat_node_t *np)
{
	int chars;

	for (chars = 0; np != NULL; np = np->next)
		chars += np->length + extra_chars(np);

	return chars / console_width + (chars % console_width != 0);
}

/** Overestimated, precise measurements will
 * come later.
 */
static int calculate_columns(stat_node_t *np)
{
	int rows, nodes;

	rows = calculate_rows(np);
	nodes = count_nodes(np);

	return nodes / rows + (nodes % rows != 0);
}
/** Count the sll entries */
int count_nodes(stat_node_t *np)
{
	if (np == NULL)
		return 0;
	else
		return (1 + count_nodes(np->next));
}
/** Format size in a human readable way */
static void readable_size(aoff64_t val, char *string)
{
	const aoff64_t unit = 1024LLU;

	if (val < unit)
		snprintf(string, MAX_SIZE_STRING, "%8d B", (int)val);
	else {
		aoff64_t r;
		int cnt = -1;
		char unit_string[7] = "KMGTPE";
		const int rounder = 103; /* (unit / 10) + 1 */

		do {
			r = (val % unit) / rounder;
			val  >>= 10;
			++cnt;
		} while (val >= unit);

		snprintf(string, MAX_SIZE_STRING, "%4d%c%1d %ciB", (int)val,
		    NUMERIC_SEPARATOR, (int)r , unit_string[cnt]);
	}
}

/** Get the longest file size.
 *
 * printf size fields will be set on this.
 */
static uint8_t longest_size(stat_node_t *np)
{
	aoff64_t size;
	char *s;

	for (size = 0; np != NULL; np = np->next)
		size = MAX(size, np->index);

	asprintf(&s, "%" PRIu64 , size);

	return str_length(s);
}

static uint8_t longest_index(stat_node_t *np)
{
	uint32_t ind;
	char *s;

	for (ind = 0; np != NULL; np = np->next)
		ind = MAX(ind, np->index);

	asprintf(&s, "%" PRIu32, ind);

	return str_length(s);
}

/** Scan already printed dir for recursion.
 *
 * Go back to scan_dir when knowing if 'no sort' or 'name sort' has to be taken.
 */
static int scan_recur_dir(const char *name)
{
	DIR *dirp;
	struct dirent *dp;
	stat_name_t *head = NULL;
	struct stat st;
	char path[MAX_PATH_LEN + 1];

	dirp = opendir(name);
	if (!dirp) {
		fprintf(stderr, "Can not open dir: %s\n", name);
		return CMD_CRITICAL_FAILURE;
	}

	while ((dp = readdir(dirp)) != NULL) {
		snprintf(path, MAX_PATH_LEN, "%s/%s", name, dp->d_name);

		if (stat(path, &st)) {
			fprintf(stderr, "Can not stat: %s\n", path);
			return CMD_CRITICAL_FAILURE;
		}

		if (st.is_directory) {
			/*
			 * 'Do not sort' option was chosen: do not sort names,
			 * just process them as they come.
			 */
			if (nostat_optmask & SOPT_NONE)
				scan_dir(path, true);
			else
				head = get_subdir_name(head, dp->d_name);
		}
	}
	closedir(dirp);

	if (!(nostat_optmask & SOPT_NONE)) {

		while (head != NULL) {
			snprintf(path, MAX_PATH_LEN, "%s/%s", name, head->name);
			scan_dir(path, true);
			head = head->next;
		}
		free_names(head);
	}

	return CMD_SUCCESS;
}

/** Get subdir names.
 *
 * While recursively scanning subdirs save dir names to process them
 * with name sorting.
 * The 'do not sort' option is treated in the caller.
 */
static stat_name_t *get_subdir_name(stat_name_t *np, char *name)
{
	if (np == NULL) {

		np = calloc (1, sizeof(stat_name_t));
		if (np == NULL) {
			fprintf(stderr, "%s: memory allocation failed: %s:%d\n",
			    cmdname, __func__, __LINE__);
			safe_exit(NULL, np, ENOMEM);
		}

		str_cpy(np->name, str_size(name) + 1,  name);
		np->next = NULL;

	} else {
		/* Sort by name, find your place */
		if ((str_cmp(name, np->name)) > 0) {
			np->next = get_subdir_name(np->next, name);
		} else {
			stat_name_t *tmp;

			tmp = np;

			np = calloc (1, sizeof(stat_name_t));
			if (np == NULL) {
				fprintf(stderr, "%s: memory allocation failed: %s:%d\n",
				    cmdname, __func__, __LINE__);
				safe_exit(NULL, tmp, ENOMEM);
			}

			str_cpy(np->name, str_size(name) + 1,  name);
			np->next = tmp;
		}
	}

	return np;
}

/** Free everything you can and exit with errcode given */
static int safe_exit(stat_node_t *nodep,stat_name_t *namep, int errcode)
{
	free_nodes(nodep);
	free_names(namep);
	drop_console(console);

	exit(errcode);
}

/**
 * @}
 */
