#!/usr/bin/perl
# cf_generate.pl
#
# cf_generate: Build the files cf_byprog.html, cf_byname.html and
#           attrs.html from the informations found 
#           in ../htcommon/defaults.cc. 
#           attrs.html : attrs_head.html + generation + attrs_tail.html
#           cf_byprog.html : cf_byprog_head.html + generation + cf_byprog_tail.html
#           cf_byname.html : cf_byname_head.html + generation + cf_byname_tail.html
#              
# Part of the ht://Dig package   <http://www.htdig.org/>
# Copyright (c) 1999-2000 The ht://Dig Group
# For copyright details, see the file COPYING in your distribution
# or the GNU Public License version 2 or later
# <http://www.gnu.org/copyleft/gpl.html>
#
# $Id: cf_generate.pl,v 1.1.2.12 2000/10/10 03:15:39 ghutchis Exp $
#


# This file contains the functions for loading
# attributes from the XML file. The following
# functions are from this file:
#
#  LoadAttributeData
#  GetHTMLDescription
#  GetAttributeNames
#  GetAttributeData
#  IsValidAttributeName

require( "../htcommon/manage_attributes.pl" );


$ATTRS_HEAD="attrs_head.html";
$ATTRS_TAIL="attrs_tail.html";
$ATTRS_OUT="attrs.html";

$BYPROG_HEAD="cf_byprog_head.html";
$BYPROG_TAIL="cf_byprog_tail.html";
$BYPROG_OUT="cf_byprog.html";

$BYNAME_HEAD="cf_byname_head.html";
$BYNAME_TAIL="cf_byname_tail.html";
$BYNAME_OUT="cf_byname.html";

# Exit and cleanup function
sub MyDie
{
   local( $msg ) = @_;

   unlink( $ATTRS_OUT ) if -r $ATTRS_OUT;
   unlink( $BYPROG_OUT ) if -r $BYPROG_OUT;
   unlink( $BYNAME_OUT ) if -r $BYNAME_OUT;

   die $msg;
}

# Takes the file $fname and writes it directly to the file $fh
sub PassFile
{
   local( $fh, $fname ) = @_;

   open( PASSFP, "<$fname" ) || &MyDie( "Unable to open $fname for reading " );
   while( <PASSFP> )
   {
     print $fh $_;
   }
   close(PASSFP);
}
    
#################################################
# Initial tests - don't proceed unless these 
# succeed

# Make sure that all the header and footer files
# exist

foreach $f ( $ATTRS_HEAD,  $ATTRS_TAIL, 
             $BYPROG_HEAD, $BYPROG_TAIL,
             $BYNAME_HEAD, $BYNAME_TAIL  )
{
  ( -r $f ) || 
     &MyDie("Unable to read $f");
}

# ---------------------------------------------------------------------
# Load all attribute data, and failed
# if that produces any errors

&LoadAttributeData( 0 ) ||
   &MyDie("Errors in Attribute Data Set");

# ---------------------------------------------------------------------
# Open the output file handles

open( ATTRS_FP,  ">$ATTRS_OUT" )  || &MyDie("Unable to open $ATTRS_OUT for writing");
open( BYPROG_FP, ">$BYPROG_OUT" ) || &MyDie("Unable to open $BYPROG_OUT for writing");
open( BYNAME_FP, ">$BYNAME_OUT" ) || &MyDie("Unable to open $BYNAME_OUT for writing");

# ---------------------------------------------------------------------
# Output the headers 

&PassFile( \*ATTRS_FP,  $ATTRS_HEAD );
&PassFile( \*BYPROG_FP, $BYPROG_HEAD );
&PassFile( \*BYNAME_FP, $BYNAME_HEAD );

# ---------------------------------------------------------------------
# Generate bodies

$letter = "";
$bullet = "<img src=\"dot.gif\" alt=\"*\" width=9 height=9>";
%byprogtable = ();

# This function takes a string of text and 
# transforms <>& into entities

sub HTMLEscape
{
    local( $text ) = @_;
    
    $text =~ s/&/&amp;/g;
    $text =~ s/</&lt;/g;
    $text =~ s/>/&gt;/g;
    
    return $text;
}

# This function will be used to generate reference links
#
# Parameters:
#  1) The type of link - should be one of "attr", "program" or "faq"
#  2) The name of the link - this will be an attribute name, program name or FAQ number
#  3) The name of the "source" frame. Optional - defaults to "body". This
#     parameter allows this function to be used when generate the "TOC" pages
#
# This function is used in this script wherever links are required
# to these types of items. It is also passed to the "GetHTMLDescription" function
# 

sub RefLink
{
    local( $type, $name, $source ) = @_;
    $source = "body" if ( ! $source );
    
    local( $targetAttr ) = "";
    $targetAttr=" target=\"body\"" if ( $source ne "body" );
    
    if ( $type eq "attr" )
    {
        if ( ! IsValidAttributeName($name) )
        {
            &MyDie( "Attribute Reference requested for unknown attribute '$name'");
        }
        return "<a$targetAttr href=\"attrs.html#$name\">$name</a>";
    }
    elsif ( $type eq "program" )
    {
        $targetAttr = " target=\"_top\"" if ( $name eq "htsearch" );
        return "<a$targetAttr href=\"$name.html\">$name</a>";
    }
    elsif ( $type eq "faq" )
    {
        return "<a$targetAttr href=\"FAQ.html#q$name\">$name</a>";
    }
    else # This option shouldn't happen.....
    {
        &MyDie( "Reference requested for unknown data type '$type' ");
    }
}

# This function is passed to GetHTMLDescription function
# in order to render codeblock elements

sub RenderCodeBlock
{
    local($data) = @_;
   
    $data =~ s/\n/<br>\n/g; 
    return "<blockquote><code>$data</code></blockquote>\n";
}

# This function creates an example text

sub MakeExampleEntry
{
    local( $name, $attrdata ) = @_;

    local( $egtext ) = "";
    local( $eg );
    
    foreach $eg ( @{$attrdata->{"examples"}} )
    {
        # Remove leading and trailing white space
        $eg =~ s/^\s*//g; 
        $eg =~ s/\s*$//g;
        
        $eg = &HTMLEscape( $eg );
        
        $eg =~ s/\n/<br>\n/g;
        
        $egtext .= <<EOF;
\t\t\t<tr>
\t\t\t <td valign="top">$name:</td>
\t\t\t <td nowrap>$eg</td>
\t\t\t</tr>
EOF
        
    }    
    
    if ( ! $egtext )
    {
        return "<em>No example provided</em>";
    }
    else
    {
        return "<table border=0>\n$egtext\t\t </table>";
    }
}
    

#Helper function for WriteMainAttrEntry        
sub WriteAttrField
{
    local( $name, $data ) = @_;
    print ATTRS_FP "\t\t <dt>$name:</dt><dd>$data</dd>\n";
}

sub WriteMainAttrEntry
{
    local( $name, $attrdata ) = @_;

    print ATTRS_FP <<EOF;
\t<dl>
\t <dt class=attr><a name="$name">$name</a></dt>
\t <dd>
\t\t<dl>
EOF

    # The "programs" is returned as a reference to a list of
    # of program names - this is transformed into a space delimited
    # list of links to each program page

    local( $usedby) = join( " ", map { &RefLink( "program", $_ ) } @{$attrdata->{"programs"}} );

    # The defaults is simply HTML escaped, with a message being used
    # for no default

    local( $default ) = &HTMLEscape($attrdata->{"default"});
    $default = "<em>No default</em>" if ( $default eq "" );

    # The $version string 

    local( $version ) = $attrdata->{"version"};
    $version = "$version or later" if ( $version ne "all" );

    # The internal version of the type has "_" characters for 
    # the sake of the XML

    local( $type ) = $attrdata->{"type"};
    $type =~ s/_/ /g;

    &WriteAttrField( "type",        $type );
    &WriteAttrField( "used by",     $usedby );
    &WriteAttrField( "default",     $default );
    &WriteAttrField( "block",       $attrdata->{"block"} );
    &WriteAttrField( "version",     $version );
    &WriteAttrField( "description", &GetHTMLDescription( $name, "RefLink", "RenderCodeBlock" ) );
    &WriteAttrField( "example",     &MakeExampleEntry( $name, $attrdata) );
          
    print ATTRS_FP <<EOF;
\t\t</dl>
\t </dd>
\t</dl>
\t<hr>
EOF
    
}


#-------------
# MAIN LOOP
#-------------

# For each attribute name
foreach $varname ( sort &GetAttributeNames() )
{

   # Get the data associated with that attribute
   $attrdata = &GetAttributeData( $varname );
  
   # Skip this attribute if it is a hidden attribute 
   # that isn't publicly documented
   next if $attrdata->{"nodocs"};

   
   #--------------------------------------------------
   # MAIN Page

   &WriteMainAttrEntry( $varname, $attrdata );
   
   #--------------------------------------------------
   # TOC Page - Attributes sorted by letter
   
   if( $letter ne uc(substr($varname, 0, 1)))
   {
       # If this isn't the first letter, finish of the
       # previous letter
       print BYNAME_FP "\t</font> <br>\n"
           if($letter);
      
       $letter = uc(substr($varname, 0, 1));
       print BYNAME_FP "\t<strong>$letter</strong> <font face=\"helvetica,arial\" size=\"2\"><br>\n";
   }

   print BYNAME_FP "\t $bullet " . &RefLink("attr",$varname, "contents" ) . "<br>\n";
   
   
   #--------------------------------------------------
   # TOC Page - Attributes sorted by program
   # This section just collects the attributes for writing later
   
   foreach $prog ( @{$attrdata->{"programs"}} )
   {
       if ( ! $byprogtable{ $prog }  )
       {
          $byprogtable{ $prog } = [];
       }
       
       push( @{$byprogtable{ $prog }}, $varname );
   }
}

# Write the date to Main File

($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst ) = localtime();  
if ( $year < 70  ){ $year += 2000; }
if ( $year < 200 ){ $year += 1900; }
sub pad2{ local($n) =@_; if ( $n < 10 ){ return "0$n"; } return $n; }

print ATTRS_FP "Page Generated $year/" . &pad2($mon) . "/" . pad2($mday) . " " .
                pad2($hour) . ":" . pad2($min) . ":" . pad2($sec) . "\n";

#--------------------------------------------------
# TOC Page - Attributes sorted by program
# build the page based on previously collected data

foreach $prog ( sort keys %byprogtable )
{
    print BYPROG_FP "\t<br><strong>" . &RefLink( "program", $prog , "contents") . "</strong> <font face=\"helvetica,arial\" size=\"2\"><br>\n";
    foreach $varname ( @{$byprogtable{ $prog }} )
    {
        print BYPROG_FP "\t $bullet " . &RefLink( "attr", $varname, "contents") . "<br>\n";
    }
}


# ---------------------------------------------------------------------
# Output the footers

&PassFile( \*ATTRS_FP,  $ATTRS_TAIL );
&PassFile( \*BYPROG_FP, $BYPROG_TAIL );
&PassFile( \*BYNAME_FP, $BYNAME_TAIL );

# ---------------------------------------------------------------------
# Close the output files

close( ATTRS_FP );
close( BYPROG_FP );
close( BYNAME_FP );

