//
// htsearch.cc
//
// htsearch: The main search CGI. Parses the CGI input, reads the config files
//           and calls the necessary code to put together the result lists
//           and the final display.
//
// Part of the ht://Dig package   <http://www.htdig.org/>
// Copyright (c) 1995-2003 The ht://Dig Group
// For copyright details, see the file COPYING in your distribution
// or the GNU Library General Public License (LGPL) version 2 or later
// <http://www.gnu.org/copyleft/lgpl.html>
//
// $Id: htsearch.cc,v 1.71 2004/02/20 23:20:40 lha Exp $
//

/*
To remake Makefile.in from Makefile.am, type automake in top level directory.
To remake Makefile from Makefile.in type, ./config.status in top level directory.
*/

#define CHECKPOINT \
cerr << "At checkpoint. Linenumber: " << __LINE__ << " file: " << __FILE__ << " function: " << __FUNCTION__ << endl

#ifdef HAVE_CONFIG_H
#include "htconfig.h"
#endif /* HAVE_CONFIG_H */

#include "ourSearchProgram.h"
#include "HtWordList.h"
#include "StringList.h"
#include "HtURLCodec.h"
#include "HtURLRewriter.h"
#include "WordContext.h"
#include "Collection.h"

#include <time.h>
#include <ctype.h>
#include <signal.h>


// If we have this, we probably want it.
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#elif HAVE_GETOPT_LOCAL
#include <getopt_local.h>
#endif

void reportError(char *);
String checkDatabaseFile(String configName, String readableName, HtConfiguration* config);
void readConfigFile(String filename, HtConfiguration* config);
Collection* getCollection(HtConfiguration* config, String configFile);
void useSearchRewriteRules(HtConfiguration* config);
void checkURLPartsForErrors();

int debug = 0;

int
main(int argc, char **argv)
{
    StringList collectionList; // List of databases to search on
    collectionList.Add(""); // use default config
    HtConfiguration* config= HtConfiguration::config();
    
    argc--; argv++;
    if (argc != 2)
	    reportError("Wrong number of parameters. Expected two. ourSearchProgram <config file> <search word>");
    String configFilename = argv[0];
    String searchWord = argv[1];
    
    config->Defaults(&defaults[0]); // Compiled default config
    readConfigFile(configFilename, config); // Override with values from config file
    
    // Initialize htword library (key description + wordtype...)
    WordContext::Initialize(*config);
    
    checkURLPartsForErrors();
    
    useSearchRewriteRules(config);
    
    Collection* collection = getCollection(config, configFilename);
    
    HtWordList words(*config);
    
    words.Open(collection->getWordFile(), O_RDONLY);
    
    List* wordList = words[searchWord]; // searching for term 'computer'
    if (wordList == 0)
	    cerr << "No wordlist" << endl;
    else {
	    cerr << "wordList->Count(): " << wordList->Count() << endl;
	    
	    wordList->Start_Get();
	    HtWordReference *wr;
	    while ((wr = (HtWordReference *) wordList->Get_Next())) {
		    cerr << "Flags " << wr->Flags() << ", docID: " << wr->DocID() << endl;
	    }
    }
}



//*****************************************************************************
// Report an error.  Since we don' know if we are running as a CGI or not,
// we will assume this is the first thing returned by a CGI program.
//
void
reportError(char *msg)
{
	cerr << msg << endl;
	exit(1);
}

String
checkDatabaseFile(String configName, String readableName, HtConfiguration* config)
{
	const String filename = config->Find(configName);
	if (access(filename, R_OK) < 0)
		reportError(form("Unable to read %s file '%s'\nDid you run htdig?",
				 readableName.get(), filename.get()));
	return filename;
}

void readConfigFile(String filename, HtConfiguration* config)
{
    if (access((char*)filename, R_OK) < 0)
	    reportError(form("Unable to read configuration file '%s'",
			     filename.get()));
    config->Read(filename);
}

Collection*
getCollection(HtConfiguration* config, String configFile) {
        String word_db = checkDatabaseFile("word_db", "word database", config);
	String doc_index = checkDatabaseFile("doc_index", "document index", config);
	String doc_db = checkDatabaseFile("doc_db", "document database", config);
	String doc_excerpt = checkDatabaseFile("doc_excerpt", "document excerpts", config);
	return new Collection((char*)configFile,
			      word_db.get(), doc_index.get(), doc_db.get(), doc_excerpt.get());
}

// for htsearch, use search_rewrite_rules attribute for HtURLRewriter.
void useSearchRewriteRules(HtConfiguration* config) {
    config->AddParsed("url_rewrite_rules", "${search_rewrite_rules}");
    String url_part_errors = HtURLRewriter::instance()->ErrMsg();
    if (url_part_errors.length() != 0)
	    reportError(form("Invalid url_rewrite_rules: %s",
			     url_part_errors.get()));
}

// Check url_part_aliases and common_url_parts for errors.
void
checkURLPartsForErrors() {
    String url_part_errors = HtURLCodec::instance()->ErrMsg();
    
    if (url_part_errors.length() != 0)
	    reportError(form("Invalid url_part_aliases or common_url_parts: %s",
			     url_part_errors.get()));
}

