#include "Query.h"
#include "WordSearcher.h"

WordSearcher *
ExactWordQuery::searcher = 0;
//
//

Query::~Query()
{
	if(instances > 0)
	{
		--instances;
		if(!instances)
		{
			ClearCache();
		}
	}
}


void
Query::ClearCache()
{
cout << "CLEAR: entries=" << cache.Count() << endl;
#if 0
	cache.Start_Get();
	ResultList *kill = (ResultList *)cache.Get_NextElement();
	while(kill)
	{
		if(kill && kill != empty)
		{
			delete kill;
		}
		kill = (ResultList *)cache.Get_NextElement();
	}
#endif
	cache.Release();
}

//
// get the query results
// by default
// evaluation is performed once, results are kept
//
ResultList *
Query::GetResults()
{
	String signature = GetSignature();
	ResultList *result = (ResultList *)cache[signature];

	if(!result)
	{
		cout << "EVAL: " << signature << endl;
		result = Evaluate();

		if(!result)
		{
			result = empty;
		}
		cache.Add(signature, result);
	}

	if(result == empty)
	{
		result = 0;
	}
	else
	{
		AdjustWeight(*result);
	}
	return result;
}

//
//
//

Dictionary
Query::cache;

int
Query::instances = 0;



ResultList * const
Query::empty = new ResultList;

//
// creates a query with a OrQuery with all the
// distinct fuzzy results
//
// additionally, sets fuzzy scores for used words
//
Query *
OrFuzzyExpander::MakeQuery(const String &word)
{
	Query *result = 0;
	Dictionary exacts;
	filters.Start_Get();
	Fuzzy *fuzzy = (Fuzzy *)filters.Get_Next();
	while(fuzzy)
	{
		List words;
		String nonconst = word;
		fuzzy->getWords(nonconst, words);
		words.Start_Get();
		String *w = (String *)words.Get_Next();
		while(w)
		{
cout << word << ":::" << *w << endl;
			ExactWordQuery *exact = (ExactWordQuery *)exacts[*w];
			if(!exact)
			{
				exact = new ExactWordQuery(*w);
				exact->SetWeight(fuzzy->getWeight());
				exacts.Add(*w, exact);
			}
			else
			{
				exact->SetWeight(
						exact->GetWeight() +
						fuzzy->getWeight());
			}
			w = (String *)words.Get_Next();
		}
		fuzzy = (Fuzzy *)filters.Get_Next();
	}
	
	exacts.Start_Get();
	Query *exact = (Query *)exacts.Get_NextElement();
	if(exact)
	{
		result = exact;
		exact = (Query *)exacts.Get_NextElement();
	}
	if(exact)
	{
		Query *tmp = result;
		result = new OrQuery;
		result->Add(tmp);
		while(exact)
		{
			result->Add(exact);
			exact = (Query *)exacts.Get_NextElement();
		}
	}
	exacts.Release();

	return result;
}

void
ExactWordQuery::AdjustWeight(ResultList &results)
{
	results.SetWeight(weight);
}

ResultList *
ExactWordQuery::Evaluate()
{
	ResultList *result = 0;
	if(searcher)
	{
		result = searcher->Search(word);
	}
	if(result && !result->Count())
	{
		delete result;
		result = 0;
	}
	return result;
}

//
//	l	r	and
//	----------------------
//	0	0	0
//	0	b	0
//	0	x	0
//	a	0	0
//	a	b	intersect(a,b)
//	a	x	a
//	x	0	0
//	x	b	b
//	x	x	x
//
//	i.e. some 0 => 0
//	ignores can be left out of intersection
// the shorter of the result lists is put apart for intersection
// this optimises the intersection process
//

ResultList *
AndQuery::Evaluate()
{
	ResultList *result = 0;
	ResultList *shorter = 0;

	operands.Start_Get();
	Query *operand = (Query *) operands.Get_Next();
	while(operand && !shorter)
	{
		result = operand->GetResults();
		if(!result)
		{
			break;
		}
		if(!result->IsIgnore())
		{
			shorter = result;
		}
		operand = (Query *) operands.Get_Next();
	}
	if(shorter)
	{
		List longer;
		while(operand && result)
		{
			result = operand->GetResults();
			if(result && !result->IsIgnore())
			{
				if(result->Count() < shorter->Count())
				{
					longer.Add(shorter);
					shorter = result;
				}
				else
				{
					longer.Add(result);
				}
			}
			operand = (Query *) operands.Get_Next();
		}
		if(longer.Count())
		{
			result = Intersection(*shorter, longer);
			longer.Release();
		}
		else
		{
			result = new ResultList(*shorter);
		}
	}
	return result;
}

//
// return a result list containing only the matches common to
// all input parameters.
//
// l is iterated, matches from l are searched in all elements of rs
//
ResultList *
AndQuery::Intersection(ResultList &shorter, List &lists)
{
	ResultList *result = 0;
	HtVector *elements = shorter.elements();
	for(int i = 0; i < elements->Count(); i++)
	{
		List confirms;
		DocMatch *confirm = 0;
		DocMatch *match = (DocMatch *)(*elements)[i];

		lists.Start_Get();
		for(int i = 0; i < lists.Count(); i++)
		{
			ResultList *list = (ResultList *)lists.Get_Next();
			confirm = list->find(match->GetId());
			if(confirm)
			{
				confirms.Add(confirm);
			}
		}
		if(confirms.Count() == lists.Count())
		{
			if(!result)
			{
				result = new ResultList;
			}
			DocMatch *copy = new DocMatch(*match);
			confirms.Start_Get();
			confirm = (DocMatch *)confirms.Get_Next();
			while(confirm)
			{
				copy->Merge(*confirm);
				confirm = (DocMatch *)confirms.Get_Next();
			}
			result->add(copy);
		}
		confirms.Release();
	}
	elements->Release();
	delete elements;
	return result;
}

String
OperatorQuery::GetLogicalWords() const
{
	ListCursor c;
	String out;
	out << "(";
	if(operands.Count())
	{
		operands.Start_Get(c);
		out << ((Query *) operands.Get_Next(c))->GetLogicalWords();
		Query *next = (Query *) operands.Get_Next(c);
		while(next)
		{
			out << " " << OperatorString() << " ";
			if(next)
			{
				out << next->GetLogicalWords();
			}
			else
			{
				out << "*nothing*";
			}
			next = (Query *) operands.Get_Next(c);
		}
	}
	out << ")";
	return out;
}


//	l	r	or
//	---------------------
//	0	0	0
//	0	b	b
//	0	x	x
//	a	0	a
//	a	b	union(a,b)
//	a	x	a
//	x	0	x
//	x	b	b
//	x	x	x
//
// i.e. nulls and ignored are left out union
//

ResultList *
OrQuery::Evaluate()
{
	ResultList *result = 0;
	ResultList *longer = 0;
	List  shorter;
	int ignores = 0;
	operands.Start_Get();
	Query *operand = (Query *) operands.Get_Next();
	while(operand)
	{
		ResultList *next = operand->GetResults();
		if(next)
		{
			if(!next->IsIgnore())
			{
				if(!longer || longer->Count() < next->Count())
				{
					if(longer)
					{
						shorter.Add(longer);
					}
					longer = next;
				}
				else
				{
					shorter.Add(next);
				}
			}
			else
			{
				ignores++;
			}
		}
		operand = (Query *) operands.Get_Next();
	}
	if(longer)
	{
		result = Union(*longer, shorter);
		shorter.Release();
	}
	else if(ignores == operands.Count())
	{
		result = new ResultList;
		result->Ignore();
	}
	return result;
}

//
// add to l the results in r with doc-id absent from l
//
ResultList *
OrQuery::Union(ResultList &longer, List &lists)
{
	ResultList *result = new ResultList(longer);
	
	lists.Start_Get();
	ResultList *current = (ResultList *) lists.Get_Next();
	while(current)
	{
		DictionaryCursor c;
		current->Start_Get(c);
		DocMatch *match = (DocMatch *) current->Get_NextElement(c);
		while(match)
		{
			DocMatch *previous = result->find(match->GetId());
			if(previous)
			{
				previous->Merge(*match);
			}
			else
			{
				DocMatch *copy = new DocMatch(*match);
				result->add(copy);
			}
			match = (DocMatch *) current->Get_NextElement(c);
		}
		current = (ResultList *) lists.Get_Next();
	}
	return result;
}


//
//	l	r	not
//	-------------------------
//	0	0	0
//	0	b	0
//	0	x	0
//	a	0	a
//	a	b	diff(a,b)
//	a	x	a
//	x	0	x
//	x	b	x
//	x	x	x
// 
// note l is the first operand, r is the rest
// i.e. l = 0 => not = 0
//      l = x => not = x
//      r = 0 => not = l
//      r = x => not = l
//      subtract otherwise
//
ResultList *
NotQuery::Evaluate()
{
	operands.Start_Get();
	Query *operand = (Query *) operands.Get_Next();
	ResultList *result = 0;
	ResultList *positive = operand->GetResults();
	if(positive)
	{
		List negative;
		if(!positive->IsIgnore())
		{
			operand = (Query *) operands.Get_Next();
			while(operand)
			{
				ResultList *next = operand->GetResults();
				if(next && !next->IsIgnore())
				{
					negative.Add(next);
				}
				operand = (Query *) operands.Get_Next();
			}
		}
		if(negative.Count())
		{
			result = Subtract(*positive, negative);
			negative.Release();
		}
		else
		{
			result = new ResultList(*positive);
		}
	}
	return result;
}

//
// remove from l the results with doc-id present in any of rs
//
ResultList *
NotQuery::Subtract(ResultList &positive, List &negatives)
{
	ResultList *result = 0;
	HtVector *elements = positive.elements();
	for(int i = 0; i < elements->Count(); i++)
	{
		DocMatch *match = (DocMatch *)(*elements)[i];

		bool confirm = true;
		negatives.Start_Get();
		ResultList *negative = (ResultList *)negatives.Get_Next();
		while(confirm && negative)
		{
			if(negative->exists(match->GetId()))
			{
				confirm = false;
			}
			negative = (ResultList *)negatives.Get_Next();
		}
		if(confirm)
		{
			if(!result)
			{
				result = new ResultList;
			}
			result->add(new DocMatch(*match));
		}
	}
	elements->Release();
	delete elements;
	return result;
}

String
NearQuery::OperatorString() const
{
	String s;
	s << "near/" << distance;
	return s;
}

//
//	l	r	nextTo
//	-----------------------
//	0	0	0
//	0	b	0
//	0	x	0
//	a	0	0
//	a	b	near(a, b)
//	a	x	a
//	x	0	0
//	x	b	b
//	x	x	x
//
ResultList *
NearQuery::Evaluate()
{
	ResultList *result = 0;
	Query *left = (Query *)operands[0];
	Query *right = (Query *)operands[1];

	if(left && right)
	{
		ResultList *l = left->GetResults();
		if(l)
		{
			ResultList *r = right->GetResults();
			if(r)
			{
				if(l->IsIgnore())
				{
					result = new ResultList(*r);
				}
				else if(r->IsIgnore())
				{
					result = new ResultList(*l);
				}
				else
				{
					result = Near(*l, *r);
				}
			}
		}
	}
	return result;
}

ResultList *
NearQuery::Near(ResultList &l, ResultList &r)
{
	ResultList *result = 0;
	HtVector *elements = l.elements();
	for(int i = 0; i < elements->Count(); i++)
	{
		DocMatch *match = (DocMatch *)(*elements)[i];
		DocMatch *confirm = r.find(match->GetId());
		List *r = new List;
		if(confirm)
		{
			MergeLocations(	*match->GetLocations(),
					*confirm->GetLocations(),
					*r);
		}
		if(r->Count())
		{
			if(!result)
			{
				result = new ResultList;
			}
			DocMatch *copy = new DocMatch(*match);
			copy->SetLocations(r);
			result->add(copy);
		}
		else
		{
			delete r;
		}
	}
	elements->Release();
	delete elements;
	return result;
}

//
//: merge match positions in a 'next' operation
// each position of left operand match is tested against right operand positions
// if two contiguous positions are found, they are merged into a single one
// beginning at the begin of the left operand
// and ending and the end of the right operand
// 
void
PhraseQuery::MergeLocations(const List &p, const List &q, List &r)
{
	ListCursor pc;
	const Location *right = 0;
	const Location *left = 0;
	p.Start_Get(pc);
	for(int j = 0; j < p.Count(); j++)
	{
		left = (const Location *)p.Get_Next(pc);
		ListCursor qc;
		q.Start_Get(qc);
		for(int k = 0; k < q.Count(); k++)
		{
			right = (const Location *)q.Get_Next(qc);
			if(left->to + 1 == right->from)
			{
				double prevsize = left->to - left->from + 1.0;
				double addsize = right->to - right->from + 1.0;
				double weight = 
					((left->weight * prevsize) +
					(right->weight * addsize)) / 
					(right->to - left->from + 1.0);
				r.Add(new Location(
						left->from,
						right->to,
						left->flags & right->flags,
						weight));
				break;
			}
		}
	}
}

//
//: merge match positions in a 'near' operation
// all combinations are tested; the pairs of positions near enough are kept
// 
void
NearQuery::MergeLocations(const List &p, const List &q, List &r)
{
	for(int j = 0; j < p.Count(); j++)
	{
		const Location &left = *(const Location *)p[j];
		for(int k = 0; k < q.Count(); k++)
		{
			const Location &right = *(const Location *)q[k];
			int dist = right.from - left.to;
			if(dist < 1)
			{
				dist = left.from - right.to;
				if(dist < 1)
				{
					dist = 0;
				}
			}
			if(unsigned(dist) <= distance)
			{
				r.Add(new Location(left));
				r.Add(new Location(right));
			}
		}
	}
}

ResultList *
PhraseQuery::Evaluate()
{
	ResultList *result = 0;

	operands.Start_Get();
	Query *next = (Query *)operands.Get_Next();
	if(next)
	{
		result = (ResultList *)next->GetResults();
		next = (Query *)operands.Get_Next();
	}
	if(result)
	{
		result = new ResultList(*result);
	}
	while(result && next)
	{
		ResultList *r = next->GetResults();
		if(r)
		{
			if(result->IsIgnore())
			{
				delete result;
				result = new ResultList(*r);
			}
			else if(!r->IsIgnore())
			{
				ResultList *tmp = result;
				result = Near(*tmp, *r);
				delete tmp;
			}
			next = (Query *)operands.Get_Next();
		}
		else
		{
			delete result;
			result = 0;
		}
	}
	return result;
}

String
PhraseQuery::GetLogicalWords() const
{
	ListCursor c;
	String out;
	out << "\"";
	if(operands.Count())
	{
		operands.Start_Get(c);
		out << ((Query *) operands.Get_Next(c))->GetLogicalWords();
		Query *next = (Query *) operands.Get_Next(c);
		while(next)
		{
			out << " ";
			if(next)
			{
				out << next->GetLogicalWords();
			}
			else
			{
				out << "*nothing*";
			}
			next = (Query *) operands.Get_Next(c);
		}
	}
	out << "\"";
	return out;
}

ResultList *
PhraseQuery::Near(ResultList &l, ResultList &r)
{
	ResultList *result = 0;
	HtVector *elements = l.elements();
	for(int i = 0; i < elements->Count(); i++)
	{
		DocMatch *match = (DocMatch *)(*elements)[i];
		DocMatch *confirm = r.find(match->GetId());
		List *r = new List;
		if(confirm)
		{
			MergeLocations(	*match->GetLocations(),
					*confirm->GetLocations(),
					*r);
		}
		if(r->Count())
		{
			if(!result)
			{
				result = new ResultList;
			}
			DocMatch *copy = new DocMatch(*match);
			copy->SetLocations(r);
			result->add(copy);
		}
		else
		{
			delete r;
		}
	}
	elements->Release();
	delete elements;
	return result;
}
