#ifndef __Query_h__
#define __Query_h__

//
// for details about the basic architectural patterns see the book:
// Design Patterns, by the infamous GoF
// Interpreter pattern
// Factory pattern
// Flyweight pattern
//

#ifdef HAVE_CONFIG_H
#include "htconfig.h"
#endif
#include "Object.h"
#include "htString.h"
#include "List.h"
#include "Dictionary.h"
#include "ResultList.h"
#include "Fuzzy.h"

class ResultList;

//
// a parsed, executable query
// abstract class
//
class Query : public Object
{
public:
	virtual ~Query();
	virtual void Add(Query *) {}
	virtual String GetLogicalWords() const = 0;
	virtual String GetSignature() const = 0;
	ResultList *GetResults();

protected:
	Query() { ++instances; }
	virtual ResultList *Evaluate() = 0;
	virtual void AdjustWeight(ResultList &) {}

private:
	static void ClearCache();

	static Dictionary cache;
	static ResultList * const empty;
	static int instances;
};

//
// a query that looks up for an exact word
//

class WordSearcher;

class ExactWordQuery : public Query
{
public:
	ExactWordQuery(const String &w) :
		word(w), weight(1.0) {}
	~ExactWordQuery() {}

	String GetLogicalWords() const { return word; }

	static void SetSearcher(WordSearcher *c) { searcher = c; }
	
	void SetWeight(double x) { weight = x; }
	double GetWeight() const { return weight; }
	String GetSignature() const
		{ return String("Exact:")+GetLogicalWords(); }

private:
	ResultList *Evaluate();
	void AdjustWeight(ResultList &);
	String word;
	double weight;
	static WordSearcher *searcher;
};

//
// a family of query factories
// they make fuzzy queries for given words
// and store word weights to results
// by using the existing fuzzy algorithms
//
class FuzzyExpander
{
public:
	FuzzyExpander() {}
	virtual ~FuzzyExpander() {}

	virtual Query *MakeQuery(const String &word) = 0;
};

//
// makes a Or query with all the fuzzy expansions
//
class OrFuzzyExpander : public FuzzyExpander
{
public:
	OrFuzzyExpander() {}
	void Add(Fuzzy *filter) { filters.Add(filter); }
	virtual ~OrFuzzyExpander() {}

private:
	Query *MakeQuery(const String &word);
	List filters;
};

//
// operator query
// abstract class
// a query that combines and scores the result lists
// returned by other queries stored in a operand list
//
class OperatorQuery : public Query
{
public:
	virtual ~OperatorQuery()
	{
		operands.Destroy();
	}

	void Add(Query *operand)
	{
		operands.Add(operand);
	}

	virtual String GetLogicalWords() const;

	String GetSignature() const
		{ return String("Compound:")+GetLogicalWords(); }

protected:
	OperatorQuery() {}

	virtual ResultList *Evaluate() = 0;
	virtual String OperatorString() const = 0;

	List operands; // List<Query *>
};

//
// and query
// an operator query that does 'and' combination
// 
class AndQuery : public OperatorQuery
{
public:

private:
	ResultList *Evaluate();
	ResultList *Intersection(ResultList &shorter, List &longer);
	String OperatorString() const { return String("and"); }
};

//
// and query
// an operator query that does 'or' combination
//
class OrQuery : public OperatorQuery
{
public:

private:
	ResultList *Evaluate();
	ResultList *Union(ResultList &longer, List &shorter);
	String OperatorString() const { return String("or"); }
};

//
// not operator (n-ary not!)
// not(a, b, c, d...) == a except (b or c or d or...)
//
class NotQuery : public OperatorQuery
{
public:

private:
	ResultList *Evaluate();
	ResultList *Subtract(ResultList &, List &);
	String OperatorString() const { return String("not"); }
};


//
// near query
// an operator that performs combination by match proximity
//
class NearQuery : public OperatorQuery
{
public:
	NearQuery(Query *left, Query *right, unsigned int dist) :
		distance(dist)
		{ Add(left); Add(right); }

	NearQuery(unsigned int dist = 10) :
		distance(dist) {}

private:
	ResultList *Evaluate();
	ResultList *Near(ResultList &, ResultList &);
	String OperatorString() const;
	unsigned int distance;
	void MergeLocations(const List &, const List &, List &);
};

//
// phrase query
// an operator performing sequence matching
//
class PhraseQuery : public OperatorQuery
{
public:
	PhraseQuery() {}
	~PhraseQuery() {}
	String GetLogicalWords() const;

private:
	ResultList *Evaluate();
	ResultList *Near(ResultList &, ResultList &);
	void MergeLocations(const List &, const List &, List &);
	String OperatorString() const { return ""; }

};

#endif
