package HTML::FormFu::Element::DateTime;

use strict;
use base 'HTML::FormFu::Element::_Field', 'HTML::FormFu::Element::Multi';
use Class::C3;

use HTML::FormFu::Attribute qw/ mk_attrs /;
use HTML::FormFu::Util qw/ _filter_components _parse_args /;
use DateTime;
use DateTime::Format::Builder;
use DateTime::Format::Natural;
use DateTime::Locale;
use Scalar::Util qw/ blessed /;
use Carp qw/ croak /;

__PACKAGE__->mk_attrs(
    qw/
        minute hour day month year
        /
);

__PACKAGE__->mk_accessors(
    qw/
        strftime auto_inflate default_natural time_zone
        /
);

# build get_Xs methods
for my $method (
    qw/
    deflator filter constraint inflator validator transformer /
    )
{
    my $sub = sub {
        my $self       = shift;
        my %args       = _parse_args(@_);
        my $get_method = "get_${method}s";

        my $accessor = "_${method}s";
        my @x        = @{ $self->$accessor };
        push @x, map { @{ $_->$get_method(@_) } } @{ $self->_elements };

        return _filter_components( \%args, \@x );
    };

    my $name = __PACKAGE__ . "::get_${method}s";

    no strict 'refs';

    *{$name} = $sub;
}

sub new {
    my $self = shift->next::method(@_);

    $self->is_field(0);
    $self->strftime("%d-%m-%Y %H:%M");
    $self->minute( {
            type   => '_DateSelect',
            prefix => [],
        } );
    $self->hour( {
            type   => '_DateSelect',
            prefix => [],
        } );
    $self->day( {
            type   => '_DateSelect',
            prefix => [],
        } );
    $self->month( {
            type   => '_DateSelect',
            prefix => [],

        } );
    $self->year( {
            type   => '_DateSelect',
            prefix => [],
            less   => 0,
            plus   => 10,
        } );

    return $self;
}

sub get_fields {
    my $self = shift;
    my %args = _parse_args(@_);

    my $f = $self->HTML::FormFu::Element::Block::get_fields;

    unshift @$f, $self;

    return _get_elements( \%args, $f );
}

sub _add_elements {
    my $self = shift;

    $self->_elements( [] );

    $self->_date_defaults;

    $self->_add_part_elem( $_ ) for qw /day month year hour minute/;

    if ( $self->auto_inflate
        && !@{ $self->get_inflators( { type => "DateTime" } ) } )
    {
        _add_inflator($self);
    }

    return;
}

sub _date_defaults {
    my $self = shift;

    my $default;
    if( defined( $default = $self->default_natural ) ) {
        my $parser = DateTime::Format::Natural->new;
        $default = $parser->parse_datetime( $default );
    }
    elsif( defined( $default = $self->default ) ) {
        my $is_blessed = blessed( $default );

        if ( !$is_blessed || ( $is_blessed && !$default->isa('DateTime') ) ) {
            my $builder = DateTime::Format::Builder->new;
            $builder->parser( { strptime => $self->strftime } );

            $default = $builder->parse_datetime($default);
        }
    }

    if ( defined $default ) {
        if( defined $self->time_zone ){
            $default->set_time_zone('UTC')->set_time_zone( $self->time_zone );
        }
        $self->minute->{default}   = $default->minute;
        $self->hour->{default}   = $default->hour;
        $self->day->{default}   = $default->day;
        $self->month->{default} = $default->month;
        $self->year->{default}  = $default->year;
    }

    return;
}

sub _add_part_elem {
    my ($self, $part) = @_;

    my $config = $self->$part;

    my $name = _build_name($self, $part);

    my @prefix = map { [ '', $_ ] }
        ref $config->{prefix} ? @{ $config->{prefix} } : $config->{prefix};
    my @options = $self->_get_part_options( $part, $config );

    $self->element( {
            type    => $config->{type},
            name    => $name,
            options => [ @prefix, @options ],
            defined $config->{default}
            ? ( default => $config->{default} )
            : (),
        } );

    return;
}


sub _get_part_options {
    my ($self, $part, $config) = @_;
    if( $part eq 'minute' ) {
        return ( map { [ $_, $_ ] } 0 .. 59 );
    }
    elsif( $part eq 'hour' ){
        return ( map { [ $_, $_ ] } 0 .. 23 );
    }
    elsif( $part eq 'day' ){
        return ( map { [ $_, $_ ] } 1 .. 31 );
    }
    elsif( $part eq 'month' ){
        my @months = _build_month_list($self);
        return ( map { [ $_ + 1, $months[$_] ] } 0 .. 11 );
    }
    elsif( $part eq 'year' ){
        my $year_ref = defined $config->{reference} ? $config->{reference} : ( localtime(time) )[5] + 1900;

        my @years = defined $config->{list} 
        ? @{ $config->{list} } 
        : ( $year_ref - $config->{less} ) .. ( $year_ref + $config->{plus} );
        return ( map { [ $_, $_ ] } @years );
    }
}

sub _build_month_list {
    my ($self) = @_;

    my $month = $self->month;
    my @months;

    if ( defined $month->{names} ) {
        @months = @{ $month->{names} };
    }
    else {
        for my $lang ( @{ $self->form->languages } ) {
            my $loc;
            eval { $loc = DateTime::Locale->load($lang); };
            if ( !$@ ) {
                @months
                    = map {ucfirst}
                    $month->{short_names}
                    ? @{ $loc->month_abbreviations }
                    : @{ $loc->month_names };

                last;
            }
        }
    }

    return @months;
}

sub _build_name {
    my ($self, $part) = @_;

    my $name
        = defined $self->$part->{name}
        ? $self->$part->{name}
        : $self->name . "_$part";

    return $name;
}

sub _add_inflator {
    my ($self) = @_;

    $self->inflator( {
            type     => "DateTime",
            parser   => { strptime => $self->strftime, },
            strptime => $self->strftime,
        } );

    return;
}

sub process {
    my $self = shift;

    $self->_add_elements;

    return $self->next::method(@_);;
}

sub get_part_value {
    my( $self, $input, $part) = @_;
    my $name = _build_name($self, $part);
    my $nested_name = $self->get_element( { name => $name } )->nested_name;
    return $self->get_nested_hash_value( $input, $nested_name );
}

sub process_input {
    my ( $self, $input ) = @_;
    my %values;
    for my $part ( qw/ minute hour day month year/ ){
        $values{$part} = $self->get_part_value( $input, $part );
    }


    if (   defined $values{day}
        && length $values{day}
        && defined $values{month}
        && length $values{month}
        && defined $values{year}
        && length $values{year} )
    {
        my $dt;

        eval {
            $dt = DateTime->new( %values );
        };

        my $value;

        if ($@) {
            $value = $self->strftime;
        }
        else {
            $value = $dt->strftime( $self->strftime );
        }

        $self->set_nested_hash_value( $input, $self->nested_name, $value );
    }

    return $self->next::method($input);
}

sub render_data {
    return shift->render_data_non_recursive(@_);
}

sub render_data_non_recursive {
    my $self = shift;

    my @elements = map { $_->render_data } @{ $self->_elements };
    my $render = $self->next::method( {
            elements => [ @elements ],
            @_ ? %{ $_[0] } : () } );

    return $render;
}

sub string {
    my ( $self, $args ) = @_;

    $args ||= {};

    my $render
        = exists $args->{render_data}
        ? $args->{render_data}
        : $self->render_data_non_recursive;

    my @elements = @{ $self->get_elements };
    my $html = $self->my_string_elements( $render, @elements[ 0, 1, 2 ] );
    $render->{label} = $render->{time_label} || 'Time';
    $html .= $self->my_string_elements( $render, @elements[ 3, 4 ] );
    return $html;
}

sub my_string_elements {
    my ( $self, $render, @elements ) = @_;
    my $html = $self->_string_field_start($render);
    $html .= sprintf "<span%s>\n", process_attrs( $render->{attributes} );
    for my $elem ( @elements ) {
        my $render = $elem->render_data;
        next if !defined $render;
        if ( $elem->reverse_multi ) {
            $html .= $elem->_string_field($render);

            if ( defined $elem->label ) {
                $html .= "\n" . $elem->_string_label($render);
            }
        }
        else {
            if ( defined $elem->label ) {
                $html .= $elem->_string_label($render) . "\n";
            }
            $html .= $elem->_string_field($render);
        }

        $html .= "\n";
    }
    $html .= "</span>";
    # field wrapper template - end
    $html .= $self->_string_field_end($render);
    return "$html\n";
}


1;

__END__

=head1 NAME

HTML::FormFu::Element::Date - 3 select menu multi-field

=head1 SYNOPSIS

    ---
    elements:
      - type: Date
        name: birthdate
        label: 'Birthdate:'
        day:
          prefix: "- Day -"
        month:
          prefix: "- Month -"
        year:
          prefix: "- Year -"
          less: 70
          plus: 0
        auto_inflate: 1


=head1 DESCRIPTION

Creates a L<multi|HTML::FormFu::Element::Multi> element containing 3 select 
menus for the day, month and year.

A date element named C<foo> would result in 3 select menus with the names 
C<foo_day>, C<foo_month> and C<foo_year>. The names can instead be 
overridden by the C<name> value in L</day>, L</month> and L</year>.

This element automatically merges the input parameters from the select 
menu into a single date parameter (and doesn't delete the individual menu's 
parameters).

=head1 METHODS

=head2 default

Arguments: DateTime object

Arguments: $date_string

Accepts either a L<DateTime> object, or a string containing a date, matching 
the L</strftime> format. Overwrites any default value set in L</day>, 
L</month> or L</year>.

=head2 strftime

Default Value: "%d-%m-%Y"

The format of the date as returned by L<HTML::FormFu/params>, if 
L</auto_inflate> is not set.

If L</auto_inflate> is used, this is still the format that the parameter 
will be in prior to the DateTime inflator being run; which is 
what any L<Filters|HTML::FormFu::Filter> and 
L<Constraints|HTML::FormFu::Constraint> will receive.

=head2 day

Arguments: \%setting

Set values effecting the C<day> select menu. Known keys are:

=head3 name

Override the auto-generated name of the select menu.

=head3 default

Set the default value of the select menu

=head3 prefix

Arguments: $value

Arguments: \@values

A string or arrayref of strings to be inserted into the start of the select 
menu.

Each value is only used as the label for a select item - the value for each 
of these items is always the empty string C<''>.

=head2 month

Arguments: \%setting

Set values effecting the C<month> select menu. Known keys are:

=head3 name

Override the auto-generated name of the select menu.

=head3 default

Set the default value of the select menu

=head3 prefix

Arguments: $value

Arguments: \@values

A string or arrayref of strings to be inserted into the start of the select 
menu.

Each value is only used as the label for a select item - the value for each 
of these items is always the empty string C<''>.

=head3 names

Arguments: \@months

A list of month names used for the month menu.

If not set, the list of month names is obtained from L<DateTime::Locale> 
using the locale set in L<HTML::FormFu/languages>.

=head3 short_names

Argument: bool

If true (and C<months> is not set) the list of abbreviated month names is 
obtained from L<DateTime::Locale> using the locale set in 
L<HTML::FormFu/languages>.

=head2 year

Arguments: \%setting

Set values effecting the C<year> select menu. Known keys are:

=head3 name

Override the auto-generated name of the select menu.

=head3 default

Set the default value of the select menu

=head3 prefix

Arguments: $value

Arguments: \@values

A string or arrayref of strings to be inserted into the start of the select 
menu.

Each value is only used as the label for a select item - the value for each 
of these items is always the empty string C<''>.

=head3 list

Arguments: \@years

A list of years used for the year menu.

If this is set, C<reference>, C<less> and C<plus> are ignored.

=head3 reference

Arguments: $year

Default Value: the current year, calculated from L<time()|perlfunc/time()>

If C<list> is not set, the list is created from the range of 
C<reference - year_less> to C<reference + year_plus>.

=head3 less

Arguments: $count

Default Value: 0

=head3 plus

Arguments: $count

Default Value: 10

=head2 auto_inflate

If true, a L<DateTime Inflator|HTML::FormFu::Inflator::DateTime> will 
automatically be added to the element, and it will be given a formatter so 
that stringification will result in the format specified in L</strftime>.

If you require the DateTime Inflator to have a different stringification 
format to the format used internally by your Filters and Constraints, then 
you must explicitly add your own DateTime Inflator, rather than using 
L</auto_inflate>.

=head1 CAVEATS

Although this element inherits from L<HTML::FormFu::Element::Block>, it's 
behaviour for the methods 
L<filter/filters|HTML::FormFu/filters>, 
L<constraint/constraints|HTML::FormFu/constraints>, 
L<inflator/inflators|HTML::FormFu/inflators>, 
L<validator/validators|HTML::FormFu/validators> and 
L<transformer/transformers|HTML::FormFu/transformers> is more like that of 
a L<field element|HTML::FormFu::Element::_Field>, meaning all processors are 
added directly to the date element, not to it's select-menu child elements.

This element's L<get_elements|HTML::FormFu/get_elements> and 
L<get_all_elements|HTML::FormFu/get_all_elements> are inherited from 
L<HTML::FormFu::Element::Block>, and so have the same behaviour. However, it 
overrides the C<get_fields> method, such that it returns both itself and 
it's child elements.

=head1 SEE ALSO

Is a sub-class of, and inherits methods from 
L<HTML::FormFu::Element::_Field>, 
L<HTML::FormFu::Element::Multi>, 
L<HTML::FormFu::Element::Block>, 
L<HTML::FormFu::Element>

L<HTML::FormFu::FormFu>

=head1 AUTHOR

Carl Franks, C<cfranks.org>

=head1 LICENSE

This library is free software, you can redistribute it and/or modify it under
the same terms as Perl itself.
