-------------------------------------------------------------------------------
-- $Id: Netlist.hs,v 1.22 2000/01/23 03:30:15 satnam Exp $
-------------------------------------------------------------------------------


module Netlist (StateST(..), runSTCirReturningCount, runSTCir, evaluateSTCir, 
                addGate, showNetList, showNetListB, Pointers,
                newBitVar, connectBit, inputBit, outputBit, inputVec,
                outputVec, NetList, globalClock, globalClear, getGlobals,
                showNetListToFile, putBesideNL, putBelowNL, swapTilesNL,
                addUnplacedGate, putOverNL, setGlobals,
                tileNL, addSpaceNL, ST)
where

import LazyST
import Array
import Gates
import TypeExpr
import Tile
import IOExts
import Utils

-------------------------------------------------------------------------------
-- This module provides support for efficiently building up large netlists
-- which represent circuits. A netlist is represented as a mutable
-- array using state-threads (from the ST module) which allows the
-- circuit graph to be built using update-in-place calcultions.
-- Once the circuit netlist is complete it can be returned as a frozen
-- normal Haskell array for further processing which only requires
-- the netlist to be read and not written.
-- There is nothing specific about any particular circuit design
-- library in this module: it just provides the building blocks for
-- managing the creation of netlists. The VirtexNetlist module shows
-- how this module is used to provide a Virtex FPGA non-standard 
-- interpretation for Lava circuits.
-------------------------------------------------------------------------------


type NetList = (Int, Int, [Port], Globals, Array Int InstanceTuple)

-------------------------------------------------------------------------------

-- The state that is encapsulated by the netlist interpretation
-- holds three values: a mutable count of the number of instances
-- of basic library components, a mutable count of the number of
-- nets generated and an array of instances. The information required
-- to represent an instance is described in the Tile module. The
-- unique name of an instance can be derived from its position in the
-- netlist array.


type Pointers s = (STRef s Int,                   -- Instance count
                   STRef s Int,                   -- Net count
                   STRef s [Port],                -- Input/Output ports
                   STRef s Globals,               -- Global Nets
                   STRef s TileStack,             -- Tile stack
                   STArray s Int InstanceTuple)   -- Array of instances

type Globals = (SignalA String,                   -- Clock
                SignalA String,                   -- Clear
                SignalA String,			  -- Global zero
                SignalA String)                   -- Global one
                
type TileStack = [[((Int,Int),[Int])]]

-------------------------------------------------------------------------------

-- The type StateST encapsulates calculations that map the tuple
-- of pointers to state-thread calculation in some state 's' returning
-- a value of type 'a'.

data StateST s a = StateST (Pointers s -> ST s a)

-------------------------------------------------------------------------------

-- An instance of the Monad type is declared for the (StateST s) type
-- which allows us to combine in sequence netlist building operations.

instance Monad (StateST s) where
  (>>=) = bindStateST
  return = returnStateST   

-------------------------------------------------------------------------------

-- bindStateST is used to define the bind operation for the (StateST s)
-- type. It takes one circuit netlist builder (circuit1) and applies
-- it to the state pointers returning the state reference variable(s)
-- r1. These are used to calculate circuit 'c2' which is then applied
-- to the state pointers to yield the resulting state references. 

bindStateST :: StateST s a -> (a -> StateST s b) -> StateST s b
bindStateST (StateST circuit1) circuit2
  = StateST (\pointers -> do r1 <- circuit1 pointers
                             let (StateST c2) = circuit2 r1
                             r2 <- c2 pointers
                             return r2
            )

-------------------------------------------------------------------------------

-- returnStateST takes a value of some type and lifts it into the
-- Monad for the (StateST s) type. It works simply by returning the
-- constructor for the (StateST s) Monad which encapsulates a calculation
-- which takes the pointer state and then simply returns the value
-- being lifted. 

returnStateST :: a -> StateST s a
returnStateST a = StateST (\pointers -> return a)   

-------------------------------------------------------------------------------

-- incrementST is a useful operation for incrementing the value
-- of a STRef of a Int value. It reads the referenced integer, returns
-- this value as the result and also increments the references integer.

incrementST :: STRef s Int -> ST s Int
incrementST countRef
  = do count <- readSTRef countRef
       writeSTRef countRef (count+1)
       return count 

-------------------------------------------------------------------------------

addSpaceNL :: Int -> Int -> StateST state () 
addSpaceNL width height
  = StateST (\pointers -> addSpaceNL' pointers width height)
   
-------------------------------------------------------------------------------

addSpaceNL' :: Pointers state -> Int -> Int -> ST state ()
addSpaceNL' (instCountRef, netCountRef, intf, globals, tileStackRef, arrayRef) 
         width height 
  = do inst <- incrementST instCountRef
       writeSTArray arrayRef inst (Unused, Position (width-1,height-1), [], [])
       insertTile tileStackRef ((width,height), [inst])

-------------------------------------------------------------------------------

addGate :: Cell -> [(String, BitSignal)] -> String -> StateST state BitSignal 
addGate cell inputs output
  = StateST (\pointers -> addGate' pointers cell inputs output)
   
-------------------------------------------------------------------------------

addGate' :: Pointers state -> Cell -> [(String, BitSignal)] -> String -> 
            ST state BitSignal
addGate' (instCountRef, netCountRef, intf, globals, tileStackRef, arrayRef) 
         gate inputs output 
  = do inst <- incrementST instCountRef
       count <- incrementST netCountRef
       let o = LavaSig count
       writeSTArray arrayRef inst (gate, Position (0,0), cellProperties gate,
         [PortMap Input  i0arg i0 | (i0arg, i0) <- inputs] ++
         [PortMap Output output o ])
       insertTile tileStackRef ((1,1), [inst])
       return o 

-------------------------------------------------------------------------------

addUnplacedGate :: Cell -> [(String, BitSignal)] -> String -> StateST state BitSignal 
addUnplacedGate cell inputs output
  = StateST (\pointers -> addUnplacedGate' pointers cell inputs output)

-------------------------------------------------------------------------------

addUnplacedGate' :: Pointers state -> Cell -> [(String, BitSignal)] -> String -> 
            ST state BitSignal
addUnplacedGate' (instCountRef, netCountRef, intf, globals, tileStackRef, arrayRef) 
         gate inputs output 
  = do inst <- incrementST instCountRef
       count <- incrementST netCountRef
       let o = LavaSig count
       writeSTArray arrayRef inst (gate, Unplaced, cellProperties gate,
         [PortMap Input  i0arg i0 | (i0arg, i0) <- inputs] ++
         [PortMap Output output o ])
       return o
-------------------------------------------------------------------------------


tileNL :: (a -> StateST state b) -> a -> StateST state (b, Tile)
tileNL circuit input
  = do t <- begin_tileNL
       r <- circuit input
       circuit_tile <- end_tileNL t
       return (r, circuit_tile) 

-------------------------------------------------------------------------------

begin_tileNL = StateST (\pointers -> begin_tileNL' pointers) 

-------------------------------------------------------------------------------

begin_tileNL' (instCountRef, netCountRef, intfRef, globals, tileStackRef, 
               arrayRef)
  = do instCount <- readSTRef instCountRef
       return instCount
                 
-------------------------------------------------------------------------------

end_tileNL :: Int -> StateST state Tile
end_tileNL start = StateST (\pointers -> end_tileNL' pointers start) 

-------------------------------------------------------------------------------

end_tileNL' :: Pointers state -> Int -> ST state Tile
end_tileNL' (instCountRef, netCountRef, intfRef, globals, tileStackRef, 
             arrayRef) start
 = do endPoint <- readSTRef instCountRef
      let insts = [start..endPoint-1]
      tiles <- sequence [readSTArray arrayRef i | i <- insts]
      let locs = [((x,y), i) 
                   | (i, (cell, Position (x,y), prop, portmaps)) <- 
                     zip insts tiles]
      let xs = 0 : map (fst . fst) locs
      let ys = 0 : map (snd . fst) locs
      let insts = map snd locs
      return ((maximum xs - minimum xs + 1,
               maximum ys - minimum ys + 1), insts)  
                 
-------------------------------------------------------------------------------


insertTile tileStackRef tile
  = do tileStack <- readSTRef tileStackRef
       (if tileStack == [] then
          return ()
        else
          writeSTRef tileStackRef (insertFirst tile tileStack))

-------------------------------------------------------------------------------

pushTileNL
  = StateST (\pointers -> pushTileNL' pointers) 

-------------------------------------------------------------------------------

pushTileNL' (instCountRef, netCountRef, intfRef, globals, tileStackRef, 
             arrayRef)
  = do tileStack <- readSTRef tileStackRef
       writeSTRef tileStackRef (if length tileStack > 1 then
                                 if head tileStack == [] then
                                   tileStack
                                 else
                                  []:tileStack
                                else
                                  []:tileStack)

-------------------------------------------------------------------------------

addNewTile (instCountRef, netCountRef, intfRef, globals, tileStackRef, 
             arrayRef) tile
  = do tileStack <- readSTRef tileStackRef
       writeSTRef tileStackRef ([tile]:tileStack)
       
-------------------------------------------------------------------------------


insertFirst x xs = (x: head xs) : (tail xs)

-------------------------------------------------------------------------------

popTileNL
  = StateST (\pointers -> popTileNL' pointers) 

-------------------------------------------------------------------------------

popTileNL' (instCountRef, netCountRef, intfRef, globals, tileStackRef, 
             arrayRef)
  = do tileStack <- readSTRef tileStackRef
       checkStack tileStack
       writeSTRef tileStackRef (tail tileStack)
       return (concatTiles (head tileStack))
       
       
checkStack [] = error ("popTile NL: empty stack.\n")
checkStack _ = return ()       

-------------------------------------------------------------------------------

swapTilesNL :: StateST state () 
swapTilesNL
  = StateST (\pointers -> swapTiles' pointers)

-------------------------------------------------------------------------------

swapTiles' :: Pointers state -> ST state ()
swapTiles' ptr@(instCountRef, netCountRef, intfRef, globals, tileStackRef, 
              arrayRef)
  = do t1 <- popTileNL' ptr
       t2 <- popTileNL' ptr  
       addNewTile ptr t1
       addNewTile ptr t2

-------------------------------------------------------------------------------

putBesideNL :: Tile -> Tile -> StateST state () 
putBesideNL t1 t2
  = StateST (\pointers -> putBesideNL' pointers t1 t2)

-------------------------------------------------------------------------------

putBesideNL' :: Pointers state -> Tile -> Tile -> ST state ()
putBesideNL' ptr@(instCountRef, netCountRef, intfRef, globals, tileStackRef, 
              arrayRef) t1@((w1,h1),inst1) t2@((w2,h2),inst2) 
  = notrace ("beside: " ++ show t1 ++ " to " ++ show t2 ++ "\n")
       (sequence_ [translateInstance arrayRef i w1 0 | i <- inst2])

-------------------------------------------------------------------------------

putBelowNL :: Tile -> Tile -> StateST state () 
putBelowNL t1 t2
  = StateST (\pointers -> putBelowNL' pointers t1 t2)

-------------------------------------------------------------------------------

putBelowNL' :: Pointers state -> Tile -> Tile -> ST state ()
putBelowNL' ptr@(instCountRef, netCountRef, intfRef, globals, tileStackRef, 
              arrayRef) ((w1,h1),inst1) ((w2,h2),inst2) 
  = do sequence_ [translateInstance arrayRef i 0 h1 | i <- inst2]

-------------------------------------------------------------------------------

putOverNL :: Tile -> Tile -> StateST state () 
putOverNL t1 t2
  = StateST (\pointers -> putOverNL' pointers t1 t2)

-------------------------------------------------------------------------------

putOverNL' :: Pointers state -> Tile -> Tile -> ST state ()
putOverNL' ptr@(instCountRef, netCountRef, intfRef, globals, tileStackRef, 
              arrayRef)  ((w1,h1),inst1) ((w2,h2),inst2)
  = do return ()

-------------------------------------------------------------------------------


translateInstance :: STArray state Int InstanceTuple -> Int -> Int -> Int ->
                     ST state ()
translateInstance arrayRef i dx dy
  = do (cell, pos, prop, portmaps) <- readSTArray arrayRef i
       writeSTArray arrayRef i (cell, translatePos (dx,dy) pos, prop, portmaps)

       
 
-------------------------------------------------------------------------------

getGlobals ::  StateST state Globals
getGlobals 
  = StateST (\pointers -> getGlobals' pointers)

-------------------------------------------------------------------------------

getGlobals' :: Pointers state -> ST state Globals
getGlobals' pointers@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef)
   = readSTRef globals

-------------------------------------------------------------------------------

setGlobals ::  Globals -> StateST state ()
setGlobals globals
  = StateST (\pointers -> setGlobals' pointers globals)

-------------------------------------------------------------------------------

setGlobals' :: Pointers state -> Globals -> ST state ()
setGlobals' pointers@(instCountRef, netCountRef, intfRef, globalsRef, 
                      tileStack, arrayRef) globals
   = writeSTRef globalsRef globals



-------------------------------------------------------------------------------


globalClock ::  String -> StateST state BitSignal
globalClock name 
  = StateST (\pointers -> globalClock' pointers name)

 
-------------------------------------------------------------------------------

globalClock' :: Pointers state -> String -> ST state BitSignal
globalClock' pointers@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) name
   = do intf <- readSTRef intfRef
        writeSTRef intfRef ((Port Input name BitType):intf)
        i <- addUnplacedGate' pointers Bufgp [("i", Var name)] "o"
        (clk, clr, zero, one) <- readSTRef globals
        writeSTRef globals (i, clr, zero, one)
        return i 

-------------------------------------------------------------------------------

globalClear ::  String -> StateST state BitSignal
globalClear name 
  = StateST (\pointers -> globalClear' pointers name)

 
-------------------------------------------------------------------------------

globalClear' :: Pointers state -> String -> ST state BitSignal
globalClear' pointers@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) name
   = do intf <- readSTRef intfRef
        writeSTRef intfRef ((Port Input name BitType):intf)
        i <- addUnplacedGate' pointers Ibuf [("i", Var name)] "o"
        (clk, clr, zero, one) <- readSTRef globals
        writeSTRef globals (clk, i, zero, one)
        return i 


-------------------------------------------------------------------------------

inputBit ::  String -> TypeExpr -> StateST state BitSignal
inputBit name BitType 
  = StateST (\pointers -> inputBit' pointers name)

 
-------------------------------------------------------------------------------

inputBit' :: Pointers state -> String -> ST state BitSignal
inputBit' pointers@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) name
   = do intf <- readSTRef intfRef
        writeSTRef intfRef ((Port Input name BitType):intf)
        i <- addUnplacedGate' pointers Ibuf [("i", Var name)] "o"
        return i 

-------------------------------------------------------------------------------

inputVec ::  String -> TypeExpr -> StateST state [BitSignal]
inputVec name (BitVector a dir b) 
  = StateST (\pointers -> inputVec' pointers name a dir b)

 
-------------------------------------------------------------------------------

inputVec' :: Pointers state -> String -> Int -> Direction -> Int ->
             ST state [BitSignal]
inputVec' pointers@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) name a dir b
   = do intf <- readSTRef intfRef
        writeSTRef intfRef ((Port Input name (BitVector a dir b)):intf)
        let input = [ArrayElement typ (Var name) i | i <- [min a b .. max a b]]
        sequence [addUnplacedGate' pointers Ibuf [("i", i)] "o" | i <- input]
     where
     typ = BitVector a dir b
         

-------------------------------------------------------------------------------

outputBit :: BitSignal -> String -> StateST state ()
outputBit driver out
  = StateST (\pointers -> outputBit' pointers driver out)


-------------------------------------------------------------------------------

outputBit' p@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) driver out
   = do intf <- readSTRef intfRef
        connectOutputBit p driver (Var out)  
        writeSTRef intfRef ((Port Output out BitType):intf)
        
-------------------------------------------------------------------------------

outputVec :: [BitSignal] -> String -> TypeExpr -> StateST state ()
outputVec driver out typ
  = StateST (\pointers -> outputVec' pointers driver out typ)


-------------------------------------------------------------------------------

outputVec' p@(instCountRef, netCountRef, intfRef, globals, tileStack, arrayRef) drivers out typ
   = do intf <- readSTRef intfRef
        sequence_ [connectOutputBit p driver o | (driver,o) <- zip drivers outs]  
        writeSTRef intfRef ((Port Output out typ):intf)
     where
     BitVector a dir b  = typ
     outs = [ArrayElement typ (Var out) i | i <- [min a b .. max a b]]
        
-------------------------------------------------------------------------------

outputBitNoPad :: BitSignal -> String -> StateST state ()
outputBitNoPad driver out
  = StateST (\pointers -> outputBitNoPad' pointers driver out)

-------------------------------------------------------------------------------

outputBitNoPad' p@(instCountRef, netCountRef, intfRef, globals, tileStack, 
                   arrayRef) driver out
   = do intf <- readSTRef intfRef
        connectBit p driver (Var out)  
        writeSTRef intfRef ((Port Output out BitType):intf)

-------------------------------------------------------------------------------

connectBit :: Pointers state -> BitSignal -> BitSignal -> ST state BitSignal
connectBit (instCountRef, netCountRef, intf, globals, tileStack, arrayRef) fromBit toBit
  = do inst <- incrementST instCountRef
       writeSTArray arrayRef inst (Buf, Unplaced, [], 
         [PortMap Input "i" fromBit, PortMap Output "o" toBit])
       return toBit   

-------------------------------------------------------------------------------

connectOutputBit :: Pointers state -> BitSignal -> BitSignal -> 
                    ST state BitSignal
connectOutputBit (instCountRef, netCountRef, intf, globals, tileStack, arrayRef) fromBit toBit
  = do inst <- incrementST instCountRef
       netCount <- incrementST netCountRef
       writeSTArray arrayRef inst (Obuf_S_2, Unplaced, [], 
         [PortMap Input "i" fromBit, PortMap Output "o" toBit])
       return toBit  
       
-------------------------------------------------------------------------------

connectInputBit (instCountRef, netCountRef, intf, globals, tileStack, arrayRef) fromBit toBit
  = do inst <- incrementST instCountRef
       writeSTArray arrayRef inst (Ibuf, Unplaced, [], 
         [PortMap Input "i" fromBit, PortMap Output "o" toBit])
       return toBit  

  
-------------------------------------------------------------------------------

newBitVar (instCountRef, netCountRef, intf, globals, tileStack, arrayRef)
  = do netNr <- incrementST netCountRef
       return (LavaSig netNr)

-------------------------------------------------------------------------------

runSTCirReturningCount :: StateST s a -> 
                          ST s (Int, Int, [Port], Globals, 
                                Array Int InstanceTuple)
runSTCirReturningCount (StateST cir)
  = do instCounRef <- newSTRef 0
       netCountRef <- newSTRef 0
       intfRef <- newSTRef []
       globalsRef <- newSTRef (UndefinedVar, UndefinedVar, 
                               UndefinedVar, UndefinedVar)
       tileStackRef <- newSTRef []
       arrayRef <- newSTArray (0,30000) (Unused, Unplaced, [], [])
       r <- cir (instCounRef, netCountRef, intfRef, globalsRef, tileStackRef, 
                 arrayRef)
       a <- freezeSTArray arrayRef
       instCount <- readSTRef instCounRef
       netCount <- readSTRef netCountRef
       intf <- readSTRef intfRef
       globals <- readSTRef globalsRef
       tileStack <- readSTRef tileStackRef
       checkTileStack tileStack
       return (instCount, netCount, intf, globals, a)

-------------------------------------------------------------------------------


checkTileStack stack
  = if stack == [] then
      return ()
    else
      trace ("Warning: tile stack = " ++ show stack ++ "\n") (return ())
       
-------------------------------------------------------------------------------

runSTCir :: StateST s a -> ST s (Array Int InstanceTuple)
runSTCir cir
  = do (instCount, netCount, intf, globals, a) <- runSTCirReturningCount cir
       return a
       
-------------------------------------------------------------------------------

evaluateSTCir :: (forall s . StateST s a) -> 
                 (Int, Int, [Port], Globals, Array Int InstanceTuple) 
evaluateSTCir cir = runST (runSTCirReturningCount cir)

-------------------------------------------------------------------------------

showNetList :: (forall s . StateST s a) -> IO ()
showNetList cir
  = do putStrLn "Writing netlist."
       putStr (showInstanceTupleArray (instCount-1) nl)
       putStrLn "Done."
    where
    (instCount, _, _, _, nl) = runST (runSTCirReturningCount cir)

-------------------------------------------------------------------------------

showNetListToFile :: String -> (forall s . StateST s a) -> IO ()
showNetListToFile filename cir
  = do putStr "Writing netlist..."
       writeFile (filename ++ ".nl") result
       putStrLn "Done."
    where
    result = showInstanceTupleArray (instCount-1) nl
    (instCount, _, _, _, nl) = runST (runSTCirReturningCount cir)

-------------------------------------------------------------------------------
  
showNetListB :: (forall s . StateST s a) -> IO ()
showNetListB cir
  = do putStrLn "Writing netlist."
       putStr (showInstanceTupleArray 7 nl)
       putStrLn "Done."
    where
    (instCount, _, _, _, nl) = runST (runSTCirReturningCount cir)

-------------------------------------------------------------------------------
  
       
