-------------------------------------------------------------------------------
-- $Id: Symbolic.hs,v 1.12 2000/01/20 18:24:32 satnam Exp $
-------------------------------------------------------------------------------

module Symbolic
where
import LUTs
import LavaClasses
import BitVec
import IOExts

data Symbol a
  = Variable a
  | BoolConstant Bool
  | Previous (Symbol a)
  | Gate1 Int (Symbol a)
  | Gate2 Int (Symbol a) (Symbol a)
  | Gate3 Int (Symbol a) (Symbol a) (Symbol a)
  | Gate4 Int (Symbol a) (Symbol a) (Symbol a) (Symbol a)
  | Inv (Symbol a) 
  | And (Symbol a) (Symbol a)
  | Or (Symbol a) (Symbol a)
  | Exor (Symbol a) (Symbol a)
  | Add [Symbol a] [Symbol a]
  | XorCY (Symbol a) (Symbol a)
  | MuxCY (Symbol a) (Symbol a) (Symbol a)
  | Let String (Symbol a) (Symbol a)
    deriving (Eq, Show)
    
deInit v@(Variable _) = v
deInit (Gate1 1 i) = Inv (deInit i)
deInit (Gate1 init i) = Gate1 init (deInit i)
deInit (Gate2 8 i0 i1) = And (deInit i0) (deInit i1)
deInit (Gate2 14 i0 i1) = Or (deInit i0) (deInit i1)
deInit (Gate2 6 i0 i1) = Exor (deInit i0) (deInit i1)

deInit (Gate1 init i) = Gate1 init (deInit i)
deInit (Gate2 init i0 i1) = Gate2 init (deInit i0) (deInit i1)
deInit (Previous i) = Previous (deInit i) 
deInit (XorCY i0 i1) = XorCY (deInit i0) (deInit i1)
deInit (MuxCY s d0 d1) = MuxCY (deInit s) (deInit d0) (deInit d1)
deInit other = error ("deInit: " ++ show other)

-------------------------------------------------------------------------------

data SymbolicEval a = SymbolicEval a
                      deriving (Eq, Show)  

-------------------------------------------------------------------------------

instance Monad SymbolicEval where
  (>>=) (SymbolicEval a) f
    = f a
  return a = SymbolicEval a   

-------------------------------------------------------------------------------


instance Stream (Symbol String) where
  bitvec2ints v = []
  to_bitvec bv = []

instance Spacers SymbolicEval 

instance Combinational SymbolicEval (Symbol String) where
  gate1 f i = return (Gate1 (lut1_init f) i)
  gate2 f (i0, i1) = return (Gate2 (lut2_init f) i0 i1)
  xorcy (i0, i1) = return (XorCY i0 i1)
  muxcy (s, (d0,d1)) = return (MuxCY s d0 d1)
  delayBit _ i = return (Previous i)
  
-------------------------------------------------------------------------------

instance Loop SymbolicEval Sym  where
  loop = loopBitSym
  loopList = loopBusSym 
  
  
type Sym = Symbol String
       

loopBitSym :: ((a, Sym) -> SymbolicEval (b, Sym)) ->
              a -> SymbolicEval b
loopBitSym cir a
  = do (c,d) <- cir (a, Variable "loopBack")
       return (trace ("loopBack = " ++ showSymbolic (deInit d) ++ " in ") c)
       
-------------------------------------------------------------------------------


loopBusSym :: Int -> ((a, [Sym]) -> SymbolicEval (b, [Sym])) ->
              a -> SymbolicEval b
loopBusSym n cir a
  = do (c,d) <- cir (a, [Variable ("loopBack_" ++ show i) | i <- [0..n-1]])
       return c
       
-------------------------------------------------------------------------------

instance Combinators SymbolicEval
instance Circuit SymbolicEval (Symbol String)    

-------------------------------------------------------------------------------

showSymbolic :: Symbol String -> String
showSymbolic (Variable a) = a
showSymbolic (Previous a) = "pre(" ++ showSymbolic a ++ ")"
showSymbolic (Let f fd c) = "let " ++ f ++ " = " ++ showSymbolic fd ++ " in "
                            ++ showSymbolic c
showSymbolic (Gate1 init i) = "gate1 " ++ show init ++ " " ++ showSymbolic i
showSymbolic (Gate2 init i0 i1) 
  = "gate2 " ++ show init ++ " " ++ showSymbolic i0 ++ " " ++ showSymbolic i1
  
showSymbolic (Inv a) = "~" ++ bracket a 1 (showSymbolic a)
showSymbolic (And i0 i1) = showSymbolic i0 ++ " & " ++ showSymbolic i1
showSymbolic (Exor i0 i1) = showSymbolic i0 ++ " * " ++ showSymbolic i1
showSymbolic (XorCY i0 i1)
   = "xorcy (" ++ showSymbolic i0 ++ ", " ++ showSymbolic i1 ++ ")"
showSymbolic (MuxCY s d0 d1)
  = "muxcy (" ++ showSymbolic s ++ ", ( " ++ showSymbolic d0 ++ ", " ++
    showSymbolic d1 ++ "))"
showSymbolic other = error ("showSymbolic: " ++ show other)

precd (Variable _) = 0
precd (Inv _) = 1
precd (And _ _) = 2
precd (Or _ _) = 3
precd (Gate1 _ _) = 3
precd (Gate2 _ _ _) = 3

-------------------------------------------------------------------------------

bracket s p str
  = if precd s < p then
      str
    else
      "(" ++ str ++ ")"
       
-------------------------------------------------------------------------------
 
class RunSymbolic t where
  runSymbolic :: SymbolicEval t -> IO ()
 
-------------------------------------------------------------------------------

instance RunSymbolic (Symbol String) where
  runSymbolic = runSymbolic1
 
runSymbolic1 :: SymbolicEval (Symbol String) -> IO ()
runSymbolic1 cir
  = putStrLn (showSymbolic (deInit result))
    where
    SymbolicEval result = cir  

-------------------------------------------------------------------------------

instance RunSymbolic (Symbol String, Symbol String) where
  runSymbolic = runSymbolic2
  
  
runSymbolic2 :: SymbolicEval (Symbol String, Symbol String) -> IO ()
runSymbolic2 cir
  = do putStrLn (showSymbolic (deInit result1))
       putStrLn (showSymbolic (deInit result2))
    where
    SymbolicEval (result1, result2) = cir 
    
-------------------------------------------------------------------------------

instance RunSymbolic [Symbol String] where
  runSymbolic = runSymbolicList
 
runSymbolicList :: SymbolicEval [Symbol String] -> IO ()
runSymbolicList cir
  = sequence_ [putStrLn (showSymbolic (deInit result)) | result <- results]
    where
    SymbolicEval results = cir 
    
-------------------------------------------------------------------------------
              