#define PRINTTIME

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <glob.h>
#include <stdio.h>
#include <errno.h>

#ifdef PRINTTIME
#include <time.h>
#endif

static void free_strarr(char **arr) {
	if (arr) {
		char **tmp = arr;
		while (*tmp) {
			free(*tmp);
			tmp++;	
		}
		free(arr);
	}
}

static unsigned int count_strarr(char **arr) {
	char **tmp = arr;
	while (*tmp != NULL) tmp++;
	return (tmp - arr);
}

/* does NOT allocate new memory !!!!!!!!! */
static char *get_suffix(char *string) {
	/* the suffix is everything after the /etc/rc?.d/??? */
	return string + 14;
}

static unsigned short int strarr_has_suffix(char **arr, const char *suffix) {
	char **tmp = arr;
	while (*tmp) {
		if (strcmp(get_suffix(*tmp), suffix)==0) return 1;
		tmp++;
	}
	return 0;
}

static char ** remove_suffix_from_strarr(char **arr, char *suffix) {
	int i=0,j=0;
	while (arr[i]!=NULL) {
		if (strcmp(get_suffix(arr[i]), suffix)==0) {
			free(arr[i]);
		} else {
			if (i!=j) arr[j] = arr[i];
			j++;
		} 
		i++;
	}
	return arr;
}

/*static unsigned short int strarr_has_string(char **arr, const char *string) {
	char **tmp = arr;
	while (*tmp) {
		if (strcmp(*tmp, string)==0) return 1;
		tmp++;
	}
	return 0;
}*/

/* if max == -1 means unlimited */
static char **duplicate_strarr(char **arr, signed int max) {
	char **newarr, **tmp1=arr, **tmp2;
	unsigned int count;

	count = count_strarr(arr);
	if (max > 0 && max < count) {
		count = max;
	}
	newarr = tmp2 = malloc((count+1)*sizeof(char *));
	while (tmp1 - arr < count) {
/*		printf("duplicated %s\n",*tmp1);*/
		*tmp2 = strdup(*tmp1);
		tmp2++;
		tmp1++;
	}
	*tmp2 = NULL;
	return newarr;
}

static char **return_scripts_sorted(const char *runleveldir, const char *startchar) {
	glob_t globbuf;
	int ret;
	char *pattern = malloc(14 * sizeof(char));
	pattern[0] = '\0';
	pattern = strncat(pattern, runleveldir,14);
	pattern = strncat(pattern, startchar, 14);
	pattern = strncat(pattern, "*", 14);
/*	printf("searching for %s\n", pattern);*/
	ret = glob(pattern, 0, NULL, &globbuf);
	free(pattern);
	if (0 == ret) {
		char **retval = duplicate_strarr(globbuf.gl_pathv, -1);
		globfree(&globbuf);
		return retval;
	}
	return NULL;
}

static unsigned short int dir_exists(const char *path) {
	struct stat sb;
	if (0 == stat(path, &sb) && S_ISDIR(sb.st_mode)) {
		return 1;
	}
	return 0;
}

static void run_batch(char **scripts, char *arg1) {
	unsigned short int scriptcount, i;
	pid_t *children;
	
	scriptcount = count_strarr(scripts);
	children = malloc(scriptcount * sizeof(pid_t));
	
	/* first start all scripts */
	for (i=0;i<scriptcount;i++) {
		pid_t pid;
		
		pid = fork();
		if (0 == pid) {
			char *args[3];
			args[0] = scripts[i];
			args[1] = arg1;
			args[2] = NULL;
			printf("executing %s %s\n",scripts[i],arg1);
			if (execv(scripts[i], args) == -1) {
				printf("error %d occurred starting %s\n",errno,scripts[i]);
				printf(strerror(errno));
			}
			exit(errno);
		}
		children[i] = pid;
	}

	/* now wait for all scripts to finish */
	for (i=0;i<scriptcount;i++) {
		int status;
		printf("waiting for %s with pid=%d\n",scripts[i],children[i]);
		waitpid(children[i], &status, 0);
	}
}

static char* build_runleveldir(char *level) {
	if (strlen(level)==1) {
		char *retval = malloc(12);
		retval[0] = '\0';
		snprintf(retval, 12, "/etc/rc%s.d/", level);
		return retval;
	}
	printf("ERROR: runlevel had strlen() > 1\n !!");
	return NULL;
}

int main(int argc, char **argv) {
	char *cur_runlevel, *prev_runlevel;
	char **Sscripts=NULL, **Kscripts=NULL, **prevSscripts=NULL;
	char *runleveldir;
#ifdef PRINTTIME
	time_t tm;
	tm = time(NULL);
	printf("started at %s",ctime(&tm));
#endif
	cur_runlevel = getenv("RUNLEVEL");
	prev_runlevel = getenv("PREVLEVEL");
	printf("from environment: runlevel=%s, prevlevel=%s\n",cur_runlevel,prev_runlevel);

	/* test if the runlevel was passed as the first argument */
	if (argc >= 2) {
		cur_runlevel = argv[1];
	}

	/* test if runlevel is valid 0-9 or S */
	/* needs to be done */

	/* find the runleveldir */
	runleveldir = build_runleveldir(cur_runlevel);
	if (!dir_exists(runleveldir)) {
		printf("ERROR: runleveldir %s does not exist!\n",runleveldir);
	}

	/* find scripts, sort by sequence number */
	if (prev_runlevel != NULL) {
		/* if there was no previous level, there is nothing to kill */
		Kscripts = return_scripts_sorted(runleveldir, "K");
	}

	/* run batches of scripts with the same sequence number parallel */
	if (Kscripts) {
		unsigned short int start=0,len=1,i;
		char **batch;
		for (i=1;Kscripts[i]!=NULL;i++) {
			printf("comparing %s and %s\n",Kscripts[start], Kscripts[i]);
			if (strncmp(Kscripts[start], Kscripts[i], 14)==0) {
				len++;
				printf("the same! len=%d\n",len);
			} else {
				batch = duplicate_strarr(&Kscripts[start], len);
				run_batch(batch, "stop");
				free_strarr(batch);
				start = i;
				len = 1;
			}
		}
		batch = duplicate_strarr(&Kscripts[start], len);
		run_batch(batch, "stop");
		free_strarr(batch);
	}

	Sscripts = return_scripts_sorted(runleveldir, "S");
	if (prev_runlevel && strcmp(prev_runlevel,"S")!=0) {
		/* if this suffix is already started in the previous runlevel, and not 
		stopped in this runlevel, we don't need to start it anymore */
		char **tmp;
		char *prevrunleveldir = build_runleveldir(prev_runlevel);
		prevSscripts = return_scripts_sorted(prevrunleveldir, "S");
		tmp = prevSscripts;
		while (*tmp) {
			char *suffix = get_suffix(*tmp);
			if (strarr_has_suffix(Sscripts, suffix) && !strarr_has_suffix(Kscripts, suffix)) {
				Sscripts = remove_suffix_from_strarr(Sscripts, suffix);
			}
			tmp++;
		}
	}

	/* run batches of scripts with the same sequence number parallel */
	if (Sscripts) {
		unsigned short int start=0,len=1,i;
		char **batch;
		for (i=1;Sscripts[i]!=NULL;i++) {
/*			printf("comparing %s and %s\n",scripts[start], scripts[i]);*/
			if (strncmp(Sscripts[start], Sscripts[i], 14)==0) {
				len++;
/*				printf("the same! len=%d\n",len);*/
			} else {
				batch = duplicate_strarr(&Sscripts[start], len);
				run_batch(batch, "start");
				free_strarr(batch);
				start = i;
				len = 1;
			}
		}
		batch = duplicate_strarr(&Sscripts[start], len);
		run_batch(batch, "start");
		free_strarr(batch);
	}
	free_strarr(Sscripts);
	free_strarr(Kscripts);
	free_strarr(prevSscripts);
	free(runleveldir);
#ifdef PRINTTIME
	tm = time(NULL);
	printf("finished at %s",ctime(&tm));
#endif
	exit(0);
}
