package com.unet.burrows.utils.viewers;
/*
 *
 *  Based on an idea and code from David R. Nadeau, 1998
 *  Copyright (c) A L Burrows 5 June 1999 onwards
 *   
 *  In the screen X, Y:
 *    Button 1 is for translation, X, Z and Y with shift button and Y
 *    Button 3 (right button) is for rotation about Y, X and Z with shift button + X
 *    Button 2 is unused
 *
 *  Possible modifications:
 *    Add cursor key navigation for translation
 *    Add cursor key + ALT for rotation
 *    Add CTRL with mouse navigation to limit to major direction move only
 *    Add crosshair start and line to current position to show movement direction
 *    Add animation effects on left or right mouse button using crosshair
 *
 * @version 1.1, 6 June 99
 */

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.media.j3d.*;
import javax.vecmath.*;
import com.unet.burrows.utils.viewers.*;

/**
 *  MouseWalker is a Walk Examine mouse behaviour.
 *
 *  The behavior wakes up on mouse presses, releases, and mouse drags. 
 *  It modifies transforms in a way that enables the user to walk through a scene,  
 *  moving and turning as if they are in the scene. It is intended to operate on a 
 *  ViewPlatform transform, hence the direction moved by the mouse movement.
 *  <P>
 *  The behavior maps mouse drags to different transforms depending on the mouse button held
 *  down and the state of the Shift key:
 *  <DL>
 *  <DT> Button 1 (left)
 *     <DD> Horizontal movement --> X-axis translation (Right moves scene left)
 *     <DD> Vertical movement   --> Z-axis translation (Up zooms IN)
 *     <DD> Vertical + Shift    --> Y-axis translation (Up moves scene down)
 *
 *  <DT> Button 2 (middle) or any + ALT key
 *     <DD> nothing at this time
 *
 *  <DT> Button 3 (right)
 *     <DD> Horizontal movement --> Y-axis rotation
 *     <DD> Vertical movement   --> X-axis rotation
 *     <DD> Horizontal + Shift   --> Z-axis rotation
 *  </DL>
 *
 *  To support systems with 2 mouse buttons, the following mappings are supported 
 *  while dragging with any mouse button held down and 0 or more keyboard modifiers down:
 *  <UL>
 *   <LI>No modifiers = Button 1
 *   <LI>ALT = Button 2
 *   <LI>Meta = Button 3
 *   <LI>Control = Button 3
 *  </UL>
 *  The behavior automatically modifies a TransformGroup provided to the constructor.  The 
 *  TransformGroup's transform can be set at any time by the application or other behaviors to 
 *  cause the walk rotation and translation to be reset.
 *  <P>
 *  While a mouse button is down, the behavior auto changes the cursor in a given parent 
 *  AWT Component.  If no parent Component is given, no cursor changes are attempted.
 *
 *  @version     1.10 98/06/04
 *  @author      Tony L Burrows, Liverpool John Moores University, SCMS
 */
public class MouseWalker
	extends ViewerBehavior
{
	protected int oldX = 0;                         // Previous and initial cursor locations
	protected int oldY = 0;
	protected int startX = 0;
	protected int startY = 0;

	// Deadzone size (offset from initial XY for which no translate/ rotate action is taken)
	protected static final int epsilon_X = 10;
	protected static final int epsilon_Y = 10;

	protected Cursor savedCursor = null;            // Saved standard cursor

	/**
	 *  Constructs a new MouseWalker that converts mouse actions into rotations and 
    *  translations. These are put into a TransformGroup which MUST be set using the
    *  setTransformGroup method.  The cursor is changed during mouse actions if the parent
    *  frame is set using the setParentComponent method. 
    *  Variants include or exclude target transform and parent frame being set via the 
    *  constructor.
	 *
	 *  @param   parent  a parent AWT Component within which the cursor will change 
	 *                   during mouse drags
	 *           transformGroup   a TransformGroup whose transform is read and 
	 *                            written by the behavior
	 *  @return          a new MouseWalker that may need its TransformGroup 
	 *                   and parent Component set
	 */
	public MouseWalker()
	{
		super();
	}

	public MouseWalker(Component parent)
	{
		super(parent);
	}

	public MouseWalker(TransformGroup transformGroup)
	{
		super();
		subjectTransformGroup = transformGroup;
	}

	public MouseWalker(TransformGroup transformGroup, Component parent)
	{
		super(parent);
		subjectTransformGroup = transformGroup;
	}


	public void initialize()
	{
		super.initialize();                     // from parent class, here for future changes
	}

	public void onElapsedFrames(WakeupOnElapsedFrames timeEvent)
	{                                         // Not needed here, but abstract in parent
   }

	public void onButton1(MouseEvent mev)
		//   positive X mouse delta --> X axis translation
		//   positive Y mouse delta --> Z axis translation
      //   positive Y mouse delta + Shift key --> Y axis translation
		//   +ve X mouse movement is right, +ve Y mouse movement is UP the screen to zoom IN
      //   or DOWN when Y translation, since observer is moving UP
	{
		if (subjectTransformGroup == null) return;                    // nothing to modify

		int x = mev.getX();                                            // Mouse coordinates
		int y = mev.getY();

		if (mev.getID() == MouseEvent.MOUSE_PRESSED)  
		{
			oldX = x;
			oldY = y;
			startX  = x;
			startY  = y;

			if (parentComponent != null)                        // Change to a "move" cursor
			{
				savedCursor = parentComponent.getCursor();
				parentComponent.setCursor(	Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
			}                                             //end of mouse pressed
		} else {
		if (mev.getID() == MouseEvent.MOUSE_RELEASED)
		{
			if (parentComponent != null)                            // Switch cursor back
				parentComponent.setCursor(savedCursor);              // end of mouse released
		} else {                                                   
		int deltaX = x - oldX;                                  // Start mouse drag here 
		int deltaY = y - oldY;
      int deltaZ = 0;

		if (deltaX > UNUSUAL_XDELTA || deltaX < -UNUSUAL_XDELTA ||
			  deltaY > UNUSUAL_YDELTA || deltaY < -UNUSUAL_YDELTA)
		{
			return;                                    // Deltas are too huge to be believable
		}

      if (mev.isShiftDown()) {                      // Switch action depending on Shift key
        deltaZ = 0;
      } else {
        deltaZ = deltaY;
        deltaY = 0;
      };
		double xTranslationDistance = deltaX * XTranslationFactor;
      double yTranslationDistance = -deltaY * YTranslationFactor;
		double zTranslationDistance = deltaZ * ZTranslationFactor;

		translate.set(xTranslationDistance, yTranslationDistance, zTranslationDistance);    
		transform1.set(translate);                            // Build vector and transform

		subjectTransformGroup.getTransform(currentTransform);      // Get current transform
		currentTransform.mul(transform1);                          // Translate as needed
		subjectTransformGroup.setTransform(currentTransform);      // Update transform group
                                                             
		oldX = x;                                                  // Update cursor position
		oldY = y;
	}}}


	public void onButton2(MouseEvent mev)                    // ALT + a button (for middle)
	{
     // nothing to modify since it is currently unused 
     return;                    

	}

	public void onButton3(MouseEvent mev)                     // Right button
	{
		if (subjectTransformGroup == null) return;             // nothing to modify

		int x = mev.getX();
		int y = mev.getY();

		if (mev.getID() == MouseEvent.MOUSE_PRESSED)
		{
			oldX = x;
			oldY = y;
			if (parentComponent != null)                       // Change to a "move" cursor
			{
				savedCursor = parentComponent.getCursor();
				parentComponent.setCursor(	Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
			}                                                    // end of mouse down event
		} else {
  		if (mev.getID() == MouseEvent.MOUSE_RELEASED)
		{
			if (parentComponent != null)                            // Switch the cursor back
				parentComponent.setCursor(savedCursor);              // End mouse release event
		} else {                                                   // Must be a mouse drag                                                       
		//
		// Mouse moved while button down:  Rotation
		//
		// Calculate the change in X and Y from previous position.  Use this to calculate
		// rotation angles:
		//
		//   positive X mouse  --> negative Y-axis rotation
		//   positive Y mouse  --> negative X-axis rotation
		//   positive X mouse + Shift  --> negative Z-axis rotation
		//
		// where +ve X mouse movement is right, +ve Y mouse movement is DOWN the screen
		//
		int deltaX = x - oldX;
		int deltaY = 0;
      int deltaZ = 0;

		if (Math.abs(y-startY) > epsilon_Y)
		{
			deltaY = y - oldY;
		}

		if (deltaX > UNUSUAL_XDELTA || deltaX < -UNUSUAL_XDELTA ||
	       deltaY > UNUSUAL_YDELTA || deltaY < -UNUSUAL_YDELTA)
		{
			return;               // Deltas are too huge to be believable.  Probably a glitch
		}

      if (mev.isShiftDown()) {                      // Switch action depending on Shift key
        deltaZ = deltaX;
        deltaX = 0;
      }

		double xRotationAngle = -deltaY * XRotationFactor;
		double yRotationAngle = -deltaX * YRotationFactor;
      double zRotationAngle = -deltaZ * XRotationFactor;

		transform1.rotX(xRotationAngle);                                  // Build transforms
		transform2.rotY(yRotationAngle);
      transform3.rotZ(zRotationAngle);

		subjectTransformGroup.getTransform(currentTransform);   	    // Get current transform
		currentTransform.get(matrix);                                // and matrix
		translate.set(matrix.m03, matrix.m13, matrix.m23);

		currentTransform.setTranslation(origin);                   // Translate to the origin 
		currentTransform.mul(transform3, currentTransform);        // rotate
		currentTransform.mul(transform2);     
		currentTransform.mul(transform1);
		currentTransform.setTranslation(translate);               // translate back

		subjectTransformGroup.setTransform(currentTransform);     // Update transform group

		oldX = x;
		oldY = y;                                                 // end of mouse drag event
	}}}                                                              
}
