package com.unet.burrows.utils.viewers;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.media.j3d.*;
import javax.vecmath.*;


/**
 * ViewerBehavior
 * Based on code from David R. Nadeau,  @version		1.0, 98/04/16
 * Modified A L Burrows, 5 June 99 onwards
 * @version 1.1, 6 June 99
 **/

/**
 * Wakeup on mouse button presses, releases, and mouse movements and generate transforms for 
 * a transform group.  Subclasses should impose specific semantics, such as "Examine" or 
 * "Walk" similar to the navigation types used by VRML browsers.
 *
 * The following mappings are supported while dragging with any mouse button held down
 *
 *   ALT = Button 2
 *   Control = Button 3
 *
 * The behavior automatically modifies a TransformGroup provided to the constructor. The 
 * TransformGroup's Transform3D can be set at any time by the application or other behaviors
 * to cause the viewer's rotation and translation to be reset.
 *
 * The Viewer sets the range of acceptable wakeup criteria and enables the behaviour. On 
 * getting an event, the Viewer filters out mouse button presses which are glitches due to 
 * Java's event handling system.  Once done, the event is routed to an appropriate event
 * handling mechanism (abstract methods in this class). In addition, basic default scalings
 * are provided and abstract methods for modifying cursor shapes etc.
 */

public abstract class ViewerBehavior
	extends Behavior
{
	// Keep track of the transform group we modify with mouse motion.
	protected TransformGroup subjectTransformGroup = null;

	// Keep a set of wakeup criterion for mouse-generated event types.
	protected WakeupCriterion[] mouseEvents = null;
	protected WakeupOr mouseCriterion = null;

	// Track which button was last pressed
	protected static final int BUTTONNONE = -1;
	protected static final int BUTTON1 = 0;
	protected static final int BUTTON2 = 1;
	protected static final int BUTTON3 = 2;
	protected int buttonPressed = BUTTONNONE;

	// Keep a few Transform3Ds for use during event processing. 
	protected Transform3D currentTransform = new Transform3D( );
	protected Transform3D transform1 = new Transform3D( );
	protected Transform3D transform2 = new Transform3D( );
	protected Transform3D transform3 = new Transform3D( );
	protected Matrix4d matrix = new Matrix4d( );
	protected Vector3d origin = new Vector3d( 0.0, 0.0, 0.0 );
	protected Vector3d translate = new Vector3d( 0.0, 0.0, 0.0 );

	// Track whether we are on or off.
	protected boolean enabled = true;

	// Unusual X and Y delta limits.
	protected static final int UNUSUAL_XDELTA = 400;
	protected static final int UNUSUAL_YDELTA = 400;

	protected Component parentComponent = null;

	/**
	 * Construct a viewer behavior that listens to mouse movement and button presses to 
    * generate rotation and translation transforms written into a transform group and
    * with an AWT parent.  Variants with and without transform and parent.  Transform MUST
    * be set now or later but before use.
    *
	 * @param transformGroup The transform group to be modified by the behavior.
	 * @param parent The AWT Component that contains the area generating mouse events.
  	 */

	public ViewerBehavior( )
	{
		super( );
	}

	public ViewerBehavior( Component parent )
	{
		super( );
		parentComponent = parent;
	}


	public ViewerBehavior( TransformGroup transformGroup )
	{
		super( );
		subjectTransformGroup = transformGroup;
	}

	public ViewerBehavior( TransformGroup transformGroup, Component parent )
	{
		super( );
		subjectTransformGroup = transformGroup;
		parentComponent = parent;
	}


	/**
	 * Set the transform group modified by the viewer behavior. Setting the transform group
	 * to null disables the behavior until the transform group is set to an existing group.
	 *
	 * @param transformGroup The new transform group to be modified by the behavior.
	 */
	public void setTransformGroup( TransformGroup transformGroup )
	{
		subjectTransformGroup = transformGroup;
	}


	public TransformGroup getTransformGroup( )
	{
		return subjectTransformGroup;
	}


	/**
	 *  Sets the parent component who's cursor will be changed during mouse drags. If no 
    *  component set via this method or constructor, no cursor changes will be done.
    *  
	 *
	 *  @param  parent  the AWT Component where cursor changes take place during mouse drags
	 */
	public void setParentComponent( Component parent )
	{
		parentComponent = parent;
	}

	public Component getParentComponent( )
	{
		return parentComponent;
	}


	public void initialize( )
	{
		// Wakeup when mouse is dragged or when mouse button pressed/released.
		mouseEvents = new WakeupCriterion[3];
		mouseEvents[0] = new WakeupOnAWTEvent( MouseEvent.MOUSE_DRAGGED );
		mouseEvents[1] = new WakeupOnAWTEvent( MouseEvent.MOUSE_PRESSED );
		mouseEvents[2] = new WakeupOnAWTEvent( MouseEvent.MOUSE_RELEASED );
		mouseCriterion = new WakeupOr( mouseEvents );
		wakeupOn( mouseCriterion );
	}


	/**
	 * Turn the behavior on or off. This simply sets a flag which causes the behaviour to
    * occur or be skipped.  It does NOT turn off the wakeups, since that seems to cause
    * ab exception to be generated.
	 *
	 * @param state True or false state.
	 */
	public void setEnable( boolean state )
	{
     enabled = state;
	}

	public boolean getEnable( )
	{
		return enabled;
	}


	/**
	 * Process a new wakeup.  Interpret mouse presses, releases, drags.
	 *
	 * @param criteria The wakeup criteria causing the behavior wakeup.
	 */
	public void processStimulus( Enumeration criteria )
	{
		WakeupCriterion wakeup = null;
		AWTEvent[] event = null;
		int whichButton = BUTTONNONE;

		if ( enabled == false )  // Turned off.  Reschedule wakeup and nothing else
		{
		   wakeupOn( mouseCriterion );
			return;
		}

		while( criteria.hasMoreElements( ) )                 // Process all pending wakeups
		{
			wakeup = (WakeupCriterion)criteria.nextElement( );
			if ( wakeup instanceof WakeupOnAWTEvent )
			{
				event = ((WakeupOnAWTEvent)wakeup).getAWTEvent( );

				for ( int i = 0; i < event.length; i++ )       // Process all pending events
				{
					if ( event[i].getID( ) != MouseEvent.MOUSE_PRESSED &&
						event[i].getID( ) != MouseEvent.MOUSE_RELEASED &&
						event[i].getID( ) != MouseEvent.MOUSE_DRAGGED )
						continue;                                // Ignore uninteresting events

					// Java event handling doesn't always catch button bounces (redundant presses
					// and releases), or order events so that the last drag event is delivered
					// before a release. This means we can get stray events that we filter here
					//
					if ( event[i].getID( ) == MouseEvent.MOUSE_PRESSED &&
						buttonPressed != BUTTONNONE )
						continue;                           // Ignore button presses til a release

					if ( event[i].getID( ) == MouseEvent.MOUSE_RELEASED &&
						buttonPressed == BUTTONNONE )
						continue;                          // Ignore button releases until a press

					if ( event[i].getID( ) == MouseEvent.MOUSE_DRAGGED &&
						buttonPressed == BUTTONNONE )
						continue;                         // Ignore drags until a press

					MouseEvent mev = (MouseEvent)event[i];
					int modifiers = mev.getModifiers( );

					//
					// Unfortunately, the underlying event handling doesn't do a "grab" operation
					// when a mouse button is pressed.  This means that once a button  is
					// pressed, if another mouse button or a keyboard modifier key is pressed,
					// the delivered mouse event will show that a different button is being held
					// down.  For instance:
					//
					// Action                           Event
					//  Button 1 press                  Button 1 press
					//  Drag with button 1 down         Button 1 drag
					//  ALT press                       -
					//  Drag with ALT & button 1 down   Button 2 drag
					//  Button 1 release                Button 2 release
					//
					// The upshot is that we can get a button press without a matching release, and 
					// the button associated with a drag can change mid-drag. To fix this, we watch
					// for an initial button press, and from then on consider that button to be the 
					// one held down, even if additional buttons get pressed, and despite what is
					// reported in the event. Only when a button is released do we end such a grab
					//

					if ( buttonPressed == BUTTONNONE )
					{
						// No button pressed yet, figure which button down and how to direct events
						if ( ( (modifiers & InputEvent.BUTTON3_MASK) != 0 ) ||
							( ( (modifiers & InputEvent.BUTTON1_MASK)	!= 0 ) &&
							( (modifiers & InputEvent.CTRL_MASK) == InputEvent.CTRL_MASK ) ) )
						{
							whichButton = BUTTON3;                   // Button 3 activity (CTRL down)
						}
						else if ( (modifiers & InputEvent.BUTTON2_MASK)	!= 0 )
						{
							whichButton = BUTTON2;                   // Button 2 activity (ALT down)
						}
						else
						{
							whichButton = BUTTON1;                   // Button 1 activity (none down)
						}

						// If the event is to press a button, then record that button is now down
						if ( event[i].getID( ) == MouseEvent.MOUSE_PRESSED )
							buttonPressed = whichButton;
					}
					else
					{
						whichButton = buttonPressed;      // button pressed earlier not released yet
					}

					switch( whichButton )                                // Distribute the event
					{
					case BUTTON1:
						onButton1( mev );
						break;
					case BUTTON2:
						onButton2( mev );
						break;
					case BUTTON3:
						onButton3( mev );
						break;
					default:
						break;
					}

					if ( event[i].getID( ) == MouseEvent.MOUSE_RELEASED )
						buttonPressed = BUTTONNONE;            
				}
				continue;
			}

			if ( wakeup instanceof WakeupOnElapsedFrames )
			{
				onElapsedFrames( (WakeupOnElapsedFrames) wakeup );
				continue;
			}
		}

		wakeupOn( mouseCriterion );                         // Reschedule us for another wakeup
	}


	/**
	 * Default X and Y rotation factors, and XYZ translation factors.
	 */
	public static final double DEFAULT_XROTATION_FACTOR = 0.02;
	public static final double DEFAULT_YROTATION_FACTOR = 0.005;
	public static final double DEFAULT_XTRANSLATION_FACTOR = 0.02;
	public static final double DEFAULT_YTRANSLATION_FACTOR = 0.02;
	public static final double DEFAULT_ZTRANSLATION_FACTOR = 0.04;

	protected double XRotationFactor    = DEFAULT_XROTATION_FACTOR;
	protected double YRotationFactor    = DEFAULT_YROTATION_FACTOR;
	protected double XTranslationFactor = DEFAULT_XTRANSLATION_FACTOR;
	protected double YTranslationFactor = DEFAULT_YTRANSLATION_FACTOR;
	protected double ZTranslationFactor = DEFAULT_ZTRANSLATION_FACTOR;


	public void setXRotationFactor( double factor )
	{
		XRotationFactor = factor;
	}

	public double getXRotationFactor( )
	{
		return XRotationFactor;
	}
	public void setYRotationFactor( double factor )
	{
		YRotationFactor = factor;
	}

	public double getYRotationFactor( )
	{
		return YRotationFactor;
	}

	public void setXTranslationFactor( double factor )
	{
		XTranslationFactor = factor;
	}

	public double getXTranslationFactor( )
	{
		return XTranslationFactor;
	}

	public void setYTranslationFactor( double factor )
	{
		YTranslationFactor = factor;
	}

	public double getYTranslationFactor( )
	{
		return YTranslationFactor;
	}

	public void setZTranslationFactor( double factor )
	{
		ZTranslationFactor = factor;
	}

	public double getZTranslationFactor( )
	{
		return ZTranslationFactor;
	}


	/**
	 * Respond to a button1 event (press, release, or drag).
	 *
	 * @param mouseEvent A MouseEvent to respond to.
	 */
	public abstract void onButton1( MouseEvent mouseEvent );


	/**
	 * Respond to a button2 event (press, release, or drag).
	 *
	 * @param mouseEvent A MouseEvent to respond to.
	 */
	public abstract void onButton2( MouseEvent mouseEvent );


	/**
	 * Responed to a button3 event (press, release, or drag).
	 *
	 * @param mouseEvent A MouseEvent to respond to.
	 */
	public abstract void onButton3( MouseEvent mouseEvent );


	/**
	 * Respond to an elapsed frames event (assuming subclass has set up a
	 * wakeup criterion for it).
	 *
	 * @param time A WakeupOnElapsedFrames criterion to respond to.
	 */
	public abstract void onElapsedFrames( WakeupOnElapsedFrames timeEvent );
}
