import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.joda.time.YearMonthDay;

import java.util.ArrayList;
import java.util.List;

/**
 * A range of dates represented by YearMonthDay objects.
 * Dates are inclusive; i.e. the range includes the start date and the end date.
 * <p/>
 * A DateRange can include only one date i.e. the start and end dates are the same.
 * <p/>
 * User: Andreas Mross
 * Date: 14/11/2005
 * Time: 17:23:26
 */
public class DateRange {
    private final YearMonthDay startDate;

    private final YearMonthDay endDate;

    private int size;

    /**
     * Create a new DateRange object.
     *
     * @param startDate
     * @param endDate
     */
    public DateRange(YearMonthDay startDate, YearMonthDay endDate) {
        if (startDate == null) throw new NullPointerException("startDate");
        if (endDate == null) throw new NullPointerException("endDate");
        if (startDate.isAfter(endDate)) throw new IllegalArgumentException("Start date " + startDate + " cannot be after end date " + endDate);
        this.startDate = startDate;
        this.endDate = endDate;
    }

    /**
     * Get a List of all dates in the date range, from start date to end date.
     *
     * @return A List of YearMonthDay objects
     */
    public List getAllDates() {
        List result = new ArrayList();

        for (YearMonthDay current = startDate; !current.isAfter(endDate); current = current.plusDays(1)) {
            result.add(current);
        }
        return result;
    }

    /**
     * Get the common date range between the two date ranges.
     *
     * @param range
     * @return null if the ranges do not overlap
     */
    public DateRange overlap(DateRange range) {
        YearMonthDay start = startDate;
        if (range.startDate.isAfter(start)) {
            start = range.startDate;
        }
        YearMonthDay end = endDate;
        if (range.endDate.isBefore(end)) {
            end = range.endDate;
        }

        if (end.isBefore(start)) return null;
        return new DateRange(start, end);
    }

    public String toString() {
        return "[" + startDate + " - " + endDate + "]";
    }

    /**
     * Two date ranges are equal if they have the same start and end dates.
     *
     * @param o
     * @return
     */
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof DateRange)) return false;

        final DateRange dateRange = (DateRange) o;

        if (!endDate.equals(dateRange.endDate)) return false;
        if (!startDate.equals(dateRange.startDate)) return false;

        return true;
    }

    public int hashCode() {
        int result;
        result = startDate.hashCode();
        result = 29 * result + endDate.hashCode();
        return result;
    }

    /**
     * Get the number of dates in this DateRange.
     * This includes the end date.
     * A DateRange will always be at least one day long.
     *
     * @return
     */
    public int size() {
        // Use lazy initialisation to save object creation.
        // This may not be thread safe?
        // I think ultimately this implementation would still behave correctly even in a
        // multithreaded environment.
        if (size == 0) {
            Period period = new Period(startDate, endDate, PeriodType.days());
            int result = period.getDays() + 1;
            size = result;
            return result;
        }
        return size;
    }

    /**
     * Get the start date
     *
     * @return
     */
    public YearMonthDay getStart() {
        return startDate;
    }

    /**
     * Get the end date.
     *
     * @return
     */
    public YearMonthDay getEnd() {
        return endDate;
    }

    /**
     * Does this date range contain the specified date.
     * Date ranges are inclusive of the start and end date.
     *
     * @param date
     * @return
     */
    public boolean contains(YearMonthDay date) {
        return !(date.isBefore(startDate) || date.isAfter(endDate));
    }

    /**
     * Does this date range contain the specified date range completely.
     * Date ranges are inclusive of the start and end date.
     *
     * @param range
     * @return
     */
    public boolean contains(DateRange range) {
        return contains(range.startDate) && contains(range.endDate);
    }

    /**
     * Is this date range after the specified date.
     * Date ranges are inclusive of the start and end date.
     *
     * @param date
     * @return
     */
    public boolean isAfter(YearMonthDay date) {
        return startDate.isAfter(date);
    }

    /**
     * Is this date range entirely after the specified date range.
     * Date ranges are inclusive of the start and end date.
     *
     * @param range
     * @return
     */
    public boolean isAfter(DateRange range) {
        return startDate.isAfter(range.endDate);
    }

    /**
     * Is this date range before the specified date.
     * Date ranges are inclusive of the start and end date.
     *
     * @param date
     * @return
     */
    public boolean isBefore(YearMonthDay date) {
        return endDate.isBefore(date);
    }

    /**
     * Is this date range entirely before the specified date range.
     * Date ranges are inclusive of the start and end date.
     *
     * @param range
     * @return
     */
    public boolean isBefore(DateRange range) {
        return endDate.isBefore(range.startDate);
    }

    /**
     * Does this date range overlap the specified date range.
     * The date ranges overlap if at least some of the date range is in common. Date ranges are inclusive of the start and end dates.
     *
     * @param range
     * @return
     */
    public boolean overlaps(DateRange range) {
        if (contains(range.startDate) || contains(range.endDate)) return true;
        return range.contains(startDate);
    }

    public boolean containsToday() {
        return contains(new YearMonthDay());
    }

    public boolean isAfterToday() {
        return isAfter(new YearMonthDay());
    }

    public boolean isBeforeToday() {
        return isBefore(new YearMonthDay());
    }

    /**
     * Does this Date Range abut with the Date Range specified.
     * A Date Range abuts if it starts immediately after, or ends immediately before this Date Range without overlap.
     *
     * @param range
     * @return
     */
    public boolean abuts(DateRange range) {
        if (endDate.plusDays(1).equals(range.startDate)) return true;
        if (startDate.minusDays(1).equals(range.endDate)) return true;
        return false;
    }

    /**
     * Gets the gap between this DateRange and that specified.
     * Any two DateRanges can overlap, abut, or have a gap between them. This method returns
     * the amount of the gap only if the DateRanges do actually have a gap between them.
     * If the DateRanges overlap or abut, then null is returned.
     *
     * @param range
     * @return
     */
    public DateRange gap(DateRange range) {
        if (overlaps(range)) return null;
        DateRange lower = this;
        DateRange higher = range;
        if (isAfter(range)) {
            lower = range;
            higher = this;
        }
        YearMonthDay startOfGap = lower.endDate.plusDays(1);
        if (startOfGap.equals(higher.startDate)) return null;
        YearMonthDay endOfGap = higher.startDate.minusDays(1);
        return new DateRange(startOfGap, endOfGap);
    }
}
