import junit.framework.TestCase;
import org.joda.time.YearMonthDay;

import java.util.List;

/**
 * Tests for the DateRange class.
 * 
 * User: Andreas Mross
 * Date: 15/11/2005
 * Time: 09:14:12
 */
public class DateRangeTest extends TestCase {
    public void testGetAllDates() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);
        List dates = range.getAllDates();

        assertEquals("Wrong number of dates returned", dates.size(), 10);
        assertEquals("First date returned is wrong", dates.get(0), start);
        assertEquals("Second date returned is wrong", dates.get(1), new YearMonthDay(2005, 10, 26));
        assertEquals("Last date returned is wrong", dates.get(dates.size() - 1), end);

        DateRange range1 = new DateRange(start, start);
        List dates1 = range1.getAllDates();
        assertEquals("Wrong number of dates returned", dates1.size(), 1);
        assertEquals("First date returned is wrong", dates1.get(0), start);
    }

    public void testConstructor() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);

        DateRange range = new DateRange(start, end);

        range = new DateRange(start, start);

        try {
            DateRange range1 = new DateRange(end, start);
            fail("Start date must be after end date");
        } catch (IllegalArgumentException e) {
            // Expected
        }

        try {
            DateRange range2 = new DateRange(null, end);
            fail("should throw null pointer exception");
        } catch (NullPointerException e) {
            // Cool
        }

        try {
            DateRange range2 = new DateRange(start, null);
            fail("should throw null pointer exception");
        } catch (NullPointerException e) {
            // Cool
        }
    }

    public void testEquals() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);

        DateRange range1 = new DateRange(start, end);
        DateRange range2 = new DateRange(start, end);
        assertEquals(range1, range2);
        assertEquals(range1.hashCode(), range2.hashCode());

        YearMonthDay end2 = new YearMonthDay(2005, 11, 4);
        YearMonthDay start2 = new YearMonthDay(2005, 10, 4);
        DateRange range3 = new DateRange(start2, end2);
        assertFalse(range1.equals(range3));
        assertFalse(range3.equals(range1));
        assertFalse(range3.hashCode() == range1.hashCode());

        DateRange range4 = new DateRange(start2, end);
        assertFalse(range1.equals(range4));
        assertFalse(range4.equals(range1));
        assertFalse(range4.hashCode() == range1.hashCode());

        assertFalse(range1.equals("whatever"));
        assertFalse(range1.equals(null));
        assertTrue(range1.equals(range1));
    }

    public void testOverlap() {
        // Test overlap of equal date ranges
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);

        DateRange range1 = new DateRange(start, end);
        DateRange range2 = new DateRange(start, end);
        assertEquals(range1, range1.overlap(range2));

        // Test overlap of two single date ranges
        DateRange range3 = new DateRange(start, start);
        DateRange range4 = new DateRange(start, start);
        assertEquals(range3, range3.overlap(range4));

        // Test overlap of one day
        YearMonthDay end2 = new YearMonthDay(2005, 11, 8);
        DateRange range5 = new DateRange(end, end2);
        DateRange overlap = range5.overlap(range1);
        assertEquals("Wrong number of dates in overlap", overlap.size(), 1);
        assertEquals(overlap.getStart(), end);
        assertEquals(overlap.getEnd(), end);
        overlap.size();

        DateRange overlap2 = range1.overlap(range5);
        assertEquals("Wrong number of dates in overlap", overlap2.size(), 1);
        assertEquals(overlap2.getStart(), end);
        assertEquals(overlap2.getEnd(), end);

        YearMonthDay start2 = new YearMonthDay(2005, 11, 4);
        DateRange range6 = new DateRange(start2, end2);
        assertNull("Should return null if the ranges don't overlap", range1.overlap(range6));
        assertNull("Should return null if the ranges don't overlap", range6.overlap(range1));
    }

    public void testGetSize() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        assertEquals("Wrong size", 10, range.size());

        DateRange range1 = new DateRange(start, start);
        assertEquals("Should have size of 1", range1.size(), 1);
    }

    public void testGetters() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);
        assertEquals(start, range.getStart());
        assertEquals(end, range.getEnd());
    }

    public void testContainsDate() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);
        assertTrue("Should include start", range.contains(start));
        assertTrue("Should include end", range.contains(end));

        YearMonthDay beforeStart = new YearMonthDay(2005, 10, 24);
        assertFalse("Should not include day before", range.contains(beforeStart));

        YearMonthDay afterEnd = new YearMonthDay(2005, 11, 4);
        assertFalse("Should not include day before", range.contains(afterEnd));
    }

    public void testContainsRange() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        YearMonthDay start1 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2004, 11, 3);
        DateRange before = new DateRange(start1, end1);

        assertFalse("Range is before target", range.contains(before));
        assertFalse("Range is before target", before.contains(range));

        YearMonthDay start2 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end2 = new YearMonthDay(2005, 10, 24);
        DateRange justBefore = new DateRange(start2, end2);

        assertFalse("Range is before target", range.contains(justBefore));
        assertFalse("Range is before target", justBefore.contains(range));

        YearMonthDay start3 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end3 = new YearMonthDay(2005, 10, 25);
        DateRange justIncludes = new DateRange(start3, end3);

        assertFalse("Range is before target", range.contains(justIncludes));
        assertFalse("Range is before target", justIncludes.contains(range));

        assertTrue("Range should contain itself", range.contains(range));

        DateRange oneDay = new DateRange(start, start);
        assertTrue("Range with one date should contain itself", oneDay.contains(oneDay));

        YearMonthDay end4 = new YearMonthDay(2005, 11, 1);
        DateRange startRange = new DateRange(start, end4);
        assertTrue(range.contains(startRange));
        assertFalse(startRange.contains(range));

        YearMonthDay start4 = new YearMonthDay(2005, 10, 28);
        DateRange middleRange = new DateRange(start4, end4);
        assertTrue(range.contains(middleRange));
        assertFalse(middleRange.contains(range));

        DateRange endRange = new DateRange(start4, end);
        assertTrue(range.contains(endRange));
        assertFalse(endRange.contains(range));

        YearMonthDay start5 = new YearMonthDay(2004, 10, 28);
        YearMonthDay end5 = new YearMonthDay(2006, 10, 28);
        DateRange bigger = new DateRange(start5, end5);
        assertFalse(range.contains(bigger));
        assertTrue(bigger.contains(range));
    }

    public void testIsAfterInstant() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        YearMonthDay dayBeforeStart = start.minusDays(1);
        assertTrue("Is after day before", range.isAfter(dayBeforeStart));
        assertFalse("Is not after its own start date", range.isAfter(start));
        assertFalse("Is not after its own end date", range.isAfter(end));

        YearMonthDay inside = new YearMonthDay(2005, 10, 29);
        assertFalse("Is not after a date in the middle of the interval", range.isAfter(inside));

        YearMonthDay after = new YearMonthDay(2006, 1, 5);
        assertFalse("Is not after a date in the future", range.isAfter(after));
    }

    public void testIsAfterRange() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        YearMonthDay start1 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2004, 11, 3);
        DateRange before = new DateRange(start1, end1);

        assertTrue("Range is after target", range.isAfter(before));
        assertFalse("Range is after target", before.isAfter(range));

        YearMonthDay start2 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end2 = new YearMonthDay(2005, 10, 24);
        DateRange justBefore = new DateRange(start2, end2);

        assertTrue("Range is after target", range.isAfter(justBefore));
        assertFalse("Range is after target", justBefore.isAfter(range));

        YearMonthDay start3 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end3 = new YearMonthDay(2005, 10, 25);
        DateRange justIncludes = new DateRange(start3, end3);

        assertFalse("Range includes target", range.isAfter(justIncludes));
        assertFalse("Range includes target", justIncludes.isAfter(range));

        assertFalse("Range should not be after itself", range.isAfter(range));

        DateRange oneDay = new DateRange(start, start);
        assertFalse("Range with one date should not be after itself", oneDay.isAfter(oneDay));

        YearMonthDay start4 = end.plusDays(1);
        YearMonthDay end4 = new YearMonthDay(2006, 11, 1);
        DateRange afterRange = new DateRange(start4, end4);
        assertTrue(afterRange.isAfter(range));
        assertFalse(range.isAfter(afterRange));

        DateRange justNotAfter = new DateRange(end, end4);
        assertFalse(justNotAfter.isAfter(range));
        assertFalse(range.isAfter(justNotAfter));
    }

    public void testIsBeforeInstant() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        YearMonthDay dayAfterEnd = end.plusDays(1);
        assertTrue("Is before day after", range.isBefore(dayAfterEnd));
        assertFalse("Is not before its own start date", range.isBefore(start));
        assertFalse("Is not before its own end date", range.isBefore(end));

        YearMonthDay inside = new YearMonthDay(2005, 10, 29);
        assertFalse("Is not before a date in the middle of the interval", range.isBefore(inside));

        YearMonthDay before = new YearMonthDay(2005, 1, 5);
        assertFalse("Is not before a date in the past", range.isBefore(before));
    }

    public void testIsBeforeRange() {
        YearMonthDay start = new YearMonthDay(2005, 10, 25);
        YearMonthDay end = new YearMonthDay(2005, 11, 3);
        DateRange range = new DateRange(start, end);

        YearMonthDay start1 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2004, 11, 3);
        DateRange before = new DateRange(start1, end1);

        assertFalse("Range is after target", range.isBefore(before));
        assertTrue("Range is after target", before.isBefore(range));

        YearMonthDay start2 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end2 = new YearMonthDay(2005, 10, 24);
        DateRange justBefore = new DateRange(start2, end2);

        assertFalse("Range is after target", range.isBefore(justBefore));
        assertTrue("Range is after target", justBefore.isBefore(range));

        YearMonthDay start3 = new YearMonthDay(2004, 10, 25);
        YearMonthDay end3 = new YearMonthDay(2005, 10, 25);
        DateRange justIncludes = new DateRange(start3, end3);

        assertFalse("Range includes target", range.isBefore(justIncludes));
        assertFalse("Range includes target", justIncludes.isBefore(range));

        assertFalse("Range should not be before itself", range.isBefore(range));

        DateRange oneDay = new DateRange(start, start);
        assertFalse("Range with one date should not be before itself", oneDay.isBefore(oneDay));

        YearMonthDay start4 = end.plusDays(1);
        YearMonthDay end4 = new YearMonthDay(2006, 11, 1);
        DateRange afterRange = new DateRange(start4, end4);
        assertFalse(afterRange.isBefore(range));
        assertTrue(range.isBefore(afterRange));

        YearMonthDay past = new YearMonthDay(2005, 1, 1);
        DateRange justNotBefore = new DateRange(past, start);
        assertFalse(justNotBefore.isBefore(range));
        assertFalse(range.isBefore(justNotBefore));
    }

    public void testOverlaps() {
        YearMonthDay start1 = new YearMonthDay(2005, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2005, 11, 3);
        DateRange range1 = new DateRange(start1, end1);

        assertTrue("Should overlap with itself", range1.overlaps(range1));

        YearMonthDay start2 = end1.plusDays(1);
        YearMonthDay end2 = start2.plusDays(10);
        DateRange abuts1 = new DateRange(start2, end2);
        assertFalse(abuts1.overlaps(range1));
        assertFalse(range1.overlaps(abuts1));

        YearMonthDay start3 = new YearMonthDay(2005, 10, 29);
        DateRange overlaps = new DateRange(start3, end2);
        assertTrue(overlaps.overlaps(range1));
        assertTrue(range1.overlaps(overlaps));

        DateRange justOverlaps = new DateRange(end1, end2);
        assertTrue(justOverlaps.overlaps(range1));
        assertTrue(range1.overlaps(justOverlaps));

        YearMonthDay before = new YearMonthDay(2005, 9, 2);
        DateRange bigger = new DateRange(before, end2);
        assertTrue(bigger.overlaps(range1));
        assertTrue(range1.overlaps(bigger));

        YearMonthDay muchAfter = end2.plusDays(20);
        DateRange after = new DateRange(end2, muchAfter);
        assertFalse(after.overlaps(range1));
        assertFalse(range1.overlaps(after));

        DateRange rangeWithOneDay = new DateRange(start1, start1);
        assertTrue(rangeWithOneDay.overlaps(rangeWithOneDay));
    }

    public void testContainsToday() {
        YearMonthDay today = new YearMonthDay();
        DateRange startsToday = new DateRange(today, today.plusDays(10));
        assertTrue(startsToday.containsToday());
        DateRange endsToday = new DateRange(today.minusDays(10), today);
        assertTrue(endsToday.containsToday());
        DateRange includesToday = new DateRange(today.minusDays(10), today.plusDays(10));
        assertTrue(includesToday.containsToday());
        DateRange justToday = new DateRange(today, today);
        assertTrue(justToday.containsToday());
        DateRange beforeToday = new DateRange(today.minusDays(20), today.minusDays(10));
        assertFalse(beforeToday.containsToday());
        DateRange afterToday = new DateRange(today.plusDays(10), today.plusDays(20));
        assertFalse(afterToday.containsToday());
    }

    public void testIsAfterToday() {
        YearMonthDay today = new YearMonthDay();
        DateRange startsToday = new DateRange(today, today.plusDays(10));
        assertFalse(startsToday.isAfterToday());
        DateRange endsToday = new DateRange(today.minusDays(10), today);
        assertFalse(endsToday.isAfterToday());
        DateRange startsJustAfterToday = new DateRange(today.plusDays(1), today.plusDays(10));
        assertTrue(startsJustAfterToday.isAfterToday());
        DateRange includesToday = new DateRange(today.minusDays(10), today.plusDays(10));
        assertFalse(includesToday.isAfterToday());
        DateRange justToday = new DateRange(today, today);
        assertFalse(justToday.isAfterToday());
        DateRange beforeToday = new DateRange(today.minusDays(20), today.minusDays(10));
        assertFalse(beforeToday.isAfterToday());
        DateRange afterToday = new DateRange(today.plusDays(10), today.plusDays(20));
        assertTrue(afterToday.isAfterToday());
    }

    public void testIsBeforeToday() {
        YearMonthDay today = new YearMonthDay();
        DateRange startsToday = new DateRange(today, today.plusDays(10));
        assertFalse(startsToday.isBeforeToday());
        DateRange endsToday = new DateRange(today.minusDays(10), today);
        assertFalse(endsToday.isBeforeToday());
        DateRange endsJustBeforeToday = new DateRange(today.minusDays(10), today.minusDays(1));
        assertTrue(endsJustBeforeToday.isBeforeToday());
        DateRange includesToday = new DateRange(today.minusDays(10), today.plusDays(10));
        assertFalse(includesToday.isBeforeToday());
        DateRange justToday = new DateRange(today, today);
        assertFalse(justToday.isBeforeToday());
        DateRange beforeToday = new DateRange(today.minusDays(20), today.minusDays(10));
        assertTrue(beforeToday.isBeforeToday());
        DateRange afterToday = new DateRange(today.plusDays(10), today.plusDays(20));
        assertFalse(afterToday.isBeforeToday());
    }

    public void testAbuts() {
        YearMonthDay start1 = new YearMonthDay(2005, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2005, 11, 3);
        DateRange range1 = new DateRange(start1, end1);

        assertFalse("Should not abut with itself", range1.abuts(range1));

        YearMonthDay start2 = end1.plusDays(1);
        YearMonthDay end2 = start2.plusDays(10);
        DateRange abuts1 = new DateRange(start2, end2);
        assertTrue(abuts1.abuts(range1));
        assertTrue(range1.abuts(abuts1));

        YearMonthDay start3 = new YearMonthDay(2005, 10, 29);
        DateRange overlaps = new DateRange(start3, end2);
        assertFalse(overlaps.abuts(range1));
        assertFalse(range1.abuts(overlaps));

        DateRange justOverlaps = new DateRange(end1, end2);
        assertFalse(justOverlaps.abuts(range1));
        assertFalse(range1.abuts(justOverlaps));

        DateRange justDoesntAbut = new DateRange(end1.plusDays(2), end1.plusDays(10));
        assertFalse(justDoesntAbut.abuts(range1));
        assertFalse(range1.abuts(justDoesntAbut));
    }

    public void testGap() {
        YearMonthDay start1 = new YearMonthDay(2005, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2005, 11, 3);
        DateRange range1 = new DateRange(start1, end1);

        assertNull("Should have no gap with itself", range1.gap(range1));

        YearMonthDay start2 = end1.plusDays(1);
        YearMonthDay end2 = start2.plusDays(10);
        DateRange abuts1 = new DateRange(start2, end2);
        assertNull(abuts1.gap(range1));
        assertNull(range1.gap(abuts1));

        YearMonthDay start3 = new YearMonthDay(2005, 10, 29);
        DateRange overlaps = new DateRange(start3, end2);
        assertNull(overlaps.gap(range1));
        assertNull(range1.gap(overlaps));

        DateRange justOverlaps = new DateRange(end1, end2);
        assertNull(justOverlaps.gap(range1));
        assertNull(range1.gap(justOverlaps));

        YearMonthDay before = new YearMonthDay(2005, 9, 2);
        DateRange bigger = new DateRange(before, end2);
        assertNull(bigger.gap(range1));
        assertNull(range1.gap(bigger));

        YearMonthDay start4 = new YearMonthDay(2005, 12, 10);
        DateRange after = new DateRange(start4, start4.plusDays(10));
        DateRange gap = after.gap(range1);
        assertEquals(gap.getStart(), end1.plusDays(1));
        assertEquals(gap.getEnd(), start4.minusDays(1));
        assertEquals(gap, range1.gap(after));

        assertTrue(gap.abuts(range1));
        assertTrue(gap.abuts(after));
    }

    public void testToString() {
        YearMonthDay start1 = new YearMonthDay(2005, 10, 25);
        YearMonthDay end1 = new YearMonthDay(2005, 11, 3);
        DateRange range1 = new DateRange(start1, end1);
        assertEquals(range1.toString(), "[" + start1 + " - " + end1 + "]");
    }
}
