/*
 * Kernel Debugger Architecture Dependent Breakpoint Handling
 *
 * Copyright (C) 1999 Silicon Graphics, Inc.
 * Copyright (C) Scott Lurndal (slurn@engr.sgi.com)
 * Copyright (C) Scott Foehner (sfoehner@engr.sgi.com)
 * Copyright (C) Srinivasa Thirumalachar (sprasad@engr.sgi.com)
 * Copyright (C) Ethan Solomita (ethan@cs.columbia.edu)
 *
 * Written March 1999 by Scott Lurndal at Silicon Graphics, Inc.
 *
 * Modifications from:
 *      Richard Bass                    1999/07/20
 *              Many bug fixes and enhancements.
 *      Scott Foehner
 *              Port to ia64
 *	Scott Lurndal			1999/12/12
 *		v1.0 restructuring.
 *	Keith Owens			2000/05/23
 *		KDB v1.2
 *	Ethan Solomita			2001/11/27
 *		Port to sparc64
 *	Tom Duffy			2002/02/01
 *		update to KDB v2.1
 */

#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/smp.h>
#include <linux/ptrace.h>
#include <linux/kdb.h>
#include <linux/kdbprivate.h>
#include <asm/pgalloc.h>
#include <asm/system.h>


/*
 * kdba_db_trap
 *
 *	Got called by a single-step breakpoint.
 *
 * Parameters:
 *	ef	Exception frame containing machine register state
 *	error	Error number passed to kdb.
 * Outputs:
 *	None.
 * Locking:
 *	None.
 * Remarks:
 *	None.
 */

static unsigned long sstep1_va, sstep2_va; /* Addrs of next sstep bkpt */
static unsigned int  sstep1_insn, sstep2_insn; /* Orig insn for sstep bkpts */
static unsigned long sstep_tstate; /* original value of tstate */
static struct task_struct *sstep_current; /* "current" when ss[b] issued */

/*
 * kdba_db_trap: Received a single-step breakpoint. DOING_SS should be
 * set, plus possibly DOING_SSB and DOING_SSBPT.
 */

kdb_dbtrap_t
kdba_db_trap(kdb_eframe_t ef, int error_unused)
{
	if (KDB_DEBUG(BP))
		kdb_printf("kdb: kdba_db_trap\n");

	if (ef->tpc != sstep1_va && ef->tpc != sstep2_va)
		kdb_printf("Warning: single step failed to stop at "
			   "expected instruction (pc 0x%lx)\n", ef->tpc);

	if (!KDB_STATE(DOING_SS))
		goto unknown;
	
	KDB_STATE_CLEAR(DOING_SS);
	
	if (!KDB_STATE(DOING_SSBPT))
		goto not_ssbpt;
	
	KDB_STATE_CLEAR(DOING_SSBPT);
	
	/*
	 * SSBPT -- we just went past a breakpointed instruction, and so
	 * we single-stepped in order to go back and re-breakpoint it.
	 */

	if (KDB_DEBUG(BP))
		kdb_printf("ssbpt\n");
	return(KDB_DB_SSBPT);
	
not_ssbpt:
	if (KDB_STATE(DOING_SSB)) {
		kdb_printf("SSB trap at ");
		KDB_STATE_CLEAR(DOING_SSB);
	} else {
		kdb_printf("SS trap at ");
	}
	kdb_symbol_print(ef->tpc, NULL, KDB_SP_DEFAULT|KDB_SP_NEWLINE);
	kdb_id1(ef->tpc);
	return KDB_DB_SS;
	
unknown:
	kdb_printf("kdb: debug trap, but not single stepping???\n");
	return KDB_DB_NOBPT;
}

/*
 * kdba_bp_trap
 *
 * 	Perform breakpoint processing upon entry to the
 *	processor breakpoint instruction fault.   Determine and print
 *	the active breakpoint.
 *
 * Parameters:
 *	ef	Exception frame containing machine register state
 *	error	Error number passed to kdb.
 * Outputs:
 *	None.
 * Returns:
 *	0	Standard instruction or data breakpoint encountered
 *	1	Single Step fault ('ss' command)
 *	2	Single Step fault, caller should continue ('ssb' command)
 *	3	No existing kdb breakpoint matches this debug exception
 * Locking:
 *	None.
 * Remarks:
 *
 * 	If multiple processors receive debug exceptions simultaneously,
 *	one may be waiting at the kdb fence in kdb() while the user
 *	issues a 'bc' command to clear the breakpoint the processor which
 * 	is waiting has already encountered.   If this is the case, the
 *	debug registers will no longer match any entry in the breakpoint
 *	table, and we'll return the value '3'.  This can cause a panic
 *	in die_if_kernel().  It is safer to disable the breakpoint (bd),
 *	'go' until all processors are past the breakpoint then clear the
 *	breakpoint (bc).  This code recognises a breakpoint even when
 *	disabled but not when it has been cleared.
 *
 *	WARNING: This routine resets the ip.  It should be called
 *		 once per breakpoint and the result cached.
 */

kdb_dbtrap_t
kdba_bp_trap(kdb_eframe_t ef, int error_unused)
{
	int i;
	kdb_dbtrap_t rv;
	kdb_bp_t *bp;

	/*
	 * Determine which breakpoint was encountered.
	 */
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_bp_trap: tpc=0x%lx (not adjusted) "
			   "tstate=0x%lx ef=0x%p fp=0x%lx\n",
			   ef->tpc, ef->tstate, ef, ef->u_regs[UREG_FP]);

	if (KDB_STATE(DOING_SS))
		kdb_printf("kdba_bp_trap: ??? should be single stepping!\n");
	
	rv = KDB_DB_NOBPT;	/* Cause kdb() to return */
	
	for(i=0, bp=kdb_breakpoints; i<KDB_MAXBPT; i++, bp++) {
		if (bp->bp_free || !bp->bp_enabled)
			continue;
		if (!bp->bp_global && bp->bp_cpu != smp_processor_id())
			continue;
		 if ((void *)bp->bp_addr == (void *)ef->tpc) {
			/* Hit this breakpoint.  */
			kdb_printf("Instruction(i) breakpoint #%d at 0x%lx (adjusted)\n",
				  i, ef->tpc);
			kdb_id1(ef->tpc);
			rv = KDB_DB_BPT;
			KDB_STATE_SET(NEED_SSBPT);
			break;
		}
	}

	if (KDB_DEBUG(BP))
		kdb_printf("kdba_bp_trap: returning %d\n",rv);
	return rv;
}

/*
 * kdba_bptype
 *
 *	Return a string describing type of breakpoint.
 *
 * Parameters:
 *	bph	Pointer to hardware breakpoint description
 * Outputs:
 *	None.
 * Returns:
 *	Character string.
 * Locking:
 *	None.
 * Remarks:
 */

char *
kdba_bptype(kdbhard_bp_t *bph)
{
	return "ERROR-kdba_bptype";
}

/*
 * kdba_printbpreg
 *
 *	Print register name assigned to breakpoint
 *
 * Parameters:
 *	bph	Pointer hardware breakpoint structure
 * Outputs:
 *	None.
 * Returns:
 *	None.
 * Locking:
 *	None.
 * Remarks:
 */

void
kdba_printbpreg(kdbhard_bp_t *bph)
{
	kdb_printf(" ERROR-kdba_printbpreg");
}

/*
 * kdba_printbp
 *
 *	Print string describing hardware breakpoint.
 *
 * Parameters:
 *	bph	Pointer to hardware breakpoint description
 * Outputs:
 *	None.
 * Returns:
 *	None.
 * Locking:
 *	None.
 * Remarks:
 */

void
kdba_printbp(kdb_bp_t *bp)
{
	kdb_printf("\n    is enabled");
}

/*
 * kdba_parsebp
 *
 *	Parse architecture dependent portion of the
 *	breakpoint command.
 *
 * Parameters:
 *	None.
 * Outputs:
 *	None.
 * Returns:
 *	Zero for success, a kdb diagnostic for failure
 * Locking:
 *	None.
 * Remarks:
 */

int
kdba_parsebp(int argc, const char **argv, int *nextargp, kdb_bp_t *bp)
{
	int		nextarg = *nextargp;
	kdbhard_bp_t 	*bph = &bp->bp_template;

	if ((argc + 1) != nextarg)
		return KDB_ARGCOUNT;

	if (bp->bp_forcehw)
		return KDB_NOTIMP;
	
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_parsebp\n");

	/*
	 * Indicate to architecture dependent level that
	 * the instruction replacement breakpoint technique
	 * should be used for this breakpoint.
	 */
	bph->bph_free = 1;
	bp->bp_adjust = 0;	/* software, break is fault, not trap */

	return 0;
}

/*
 * kdba_allocbp
 *
 *	Associate a hardware register with a breakpoint.
 *
 * Parameters:
 *	None.
 * Outputs:
 *	None.
 * Returns:
 *	A pointer to the allocated register kdbhard_bp_t structure for
 *	success, Null and a non-zero diagnostic for failure.
 * Locking:
 *	None.
 * Remarks:
 */

kdbhard_bp_t *
kdba_allocbp(kdbhard_bp_t *bph, int *diagp)
{
	*diagp = KDB_TOOMANYDBREGS;
	return NULL;
}

/*
 * kdba_freebp
 *
 *	Deallocate a hardware breakpoint
 *
 * Parameters:
 *	None.
 * Outputs:
 *	None.
 * Returns:
 *	Zero for success, a kdb diagnostic for failure
 * Locking:
 *	None.
 * Remarks:
 */

void
kdba_freebp(kdbhard_bp_t *bph)
{
	bph->bph_free = 1;
}

/*
 * kdba_initbp
 *
 *	Initialize the breakpoint table for the hardware breakpoint
 *	register.
 *
 * Parameters:
 *	None.
 * Outputs:
 *	None.
 * Returns:
 *	Zero for success, a kdb diagnostic for failure
 * Locking:
 *	None.
 * Remarks:
 *
 *	No hardware registers, no problem.
 */

void
kdba_initbp(void)
{
	return;
}

/*
 * kdba_installbp
 *
 *	Install a breakpoint
 *
 * Parameters:
 *	ef	Exception frame
 *	bp	Breakpoint structure for the breakpoint to be installed
 * Outputs:
 *	None.
 * Returns:
 *	None.
 * Locking:
 *	None.
 * Remarks:
 *	For hardware breakpoints, a debug register is allocated
 *	and assigned to the breakpoint.  If no debug register is
 *	available, a warning message is printed and the breakpoint
 *	is disabled.
 *
 *	For instruction replacement breakpoints, we must single-step
 *	over the replaced instruction at this point so we can re-install
 *	the breakpoint instruction after the single-step.
 */

int
kdba_installbp(kdb_eframe_t ef, kdb_bp_t *bp)
{
	/*
	 * Install the breakpoint, if it is not already installed.
	 */

	if (KDB_DEBUG(BP)) {
		kdb_printf("kdba_installbp bp_installed %d\n", bp->bp_installed);
	}
	if (!bp->bp_installed) {
		if (kdb_getarea_size(&(bp->bp_inst), bp->bp_addr, sizeof(kdb_machinst_t)) ||
		    kdb_putword(bp->bp_addr, SPARC64_BP_INSN, sizeof(kdb_machinst_t))) {
			kdb_printf("kdba_installbp failed to set software"
				   " breakpoint at 0x%lx\n", bp->bp_addr);
			return(1);
		}
		flushi(bp->bp_addr);
		bp->bp_installed = 1;
		if (KDB_DEBUG(BP)) { 
			kdb_printf("kdba_installbp: instr 0x%x at " kdb_bfd_vma_fmt0 "\n",
				   SPARC64_BP_INSN, bp->bp_addr);
			kdb_printf("kdba_installbp:       ; old instr 0x%x\n",
				   bp->bp_inst);
		}
	}
	return(0);
}

/*
 * kdba_removebp
 *
 *	Make a breakpoint ineffective.
 *
 * Parameters:
 *	None.
 * Outputs:
 *	None.
 * Returns:
 *	None.
 * Locking:
 *	None.
 * Remarks:
 */

int
kdba_removebp(kdb_bp_t *bp)
{
	/*
	 * for software breakpoints, restore the instruction stream.
	 */
	if (KDB_DEBUG(BP)) {
		kdb_printf("kdba_removebp bp_installed %d\n", bp->bp_installed);
	}
	if (bp->bp_installed) {
		if (KDB_DEBUG(BP))
			kdb_printf("kdb: restoring instruction 0x%x at " kdb_bfd_vma_fmt "\n",
				   bp->bp_inst, bp->bp_addr);
		if (kdb_putword(bp->bp_addr, bp->bp_inst, sizeof(kdb_machinst_t)))
			return(1);
		flushi(bp->bp_addr);
		bp->bp_installed = 0;
	}
	return(0);
}


/* Helpful code from GDB: */

/* Macros to extract fields from sparc instructions.  */
#define X_OP(i) (((i) >> 30) & 0x3)
#define X_RD(i) (((i) >> 25) & 0x1f)
#define X_A(i) (((i) >> 29) & 1)
#define X_COND(i) (((i) >> 25) & 0xf)
#define X_OP2(i) (((i) >> 22) & 0x7)
#define X_IMM22(i) ((i) & 0x3fffff)
#define X_OP3(i) (((i) >> 19) & 0x3f)
/* Sign extension macros.  */
#define X_DISP22(i) ((X_IMM22 (i) ^ 0x200000) - 0x200000)
#define X_DISP19(i) ((((i) & 0x7ffff) ^ 0x40000) - 0x40000)
#define X_DISP16(i) ((((((i) >> 6) && 0xc000) | ((i) & 0x3fff)) ^ 0x8000) - 0x8000)
#define X_FCN(i) (((i) >> 25) & 0x1f)

static void
getnpcs(kdb_eframe_t ef, unsigned long *nPC1, unsigned long *nPC2)
{
	unsigned int insn;
	long offset = 0;	/* Must be signed for sign-extend.  */
	
	if (kdb_getarea(insn, ef->tpc)) {
		kdb_printf("getnpcs: couldn't read instruction\n");
		*nPC1 = 0;
		*nPC2 = 0;
		return;
	}

	/* Are we a branch? */

	if (X_OP(insn) == 0 &&
	    (X_OP2(insn) == 2    /* Bicc */
	     || X_OP2(insn) == 6 /* FBfcc */
	     || X_OP2(insn) == 1 /* BPcc */
	     || X_OP2(insn) == 3 /* BPr */
	     || X_OP2(insn) == 5 /* FBPfcc */
	     || X_OP2(insn) == 7))	/* ??? */
	{
		/* If we're not annulled, we'll always go to ef->tnpc next */
		   
		if (X_A(insn) == 0)
			goto npc;
		
		/* Are we a conditional annulled branch? */

		if (X_COND(insn) & 0x7) {
			*nPC1 = ef->tnpc;
			*nPC2 = ef->tnpc + 4;
			return;
		}
		
		/* Are we a branch never annulled branch? */

		if (X_COND(insn) == 0) {
			*nPC1 = ef->tnpc + 4;
			*nPC2 = 0;
			return;
		}

		/* We are a branch always annulled branch */

		switch (X_OP2(insn))
		{
		case 7:
		case 2:
		case 6:
			offset = X_DISP22(insn);
			break;
		case 1:
		case 5:
			offset = X_DISP19(insn);
			break;
		case 3:
			offset = X_DISP16(insn);
			break;
		}
		*nPC1 = ef->tpc + offset;
		*nPC2 = 0;
		return;
	}
	
	/* With done/retry, read either tnpc or tpc direct from registers */
	
	if (X_OP (insn) == 2 && X_OP3 (insn) == 62)
	{
		unsigned long tnpc;
		
		if (X_FCN(insn) == 0)		/* done */
			__asm__ __volatile__("rdpr %%tnpc, %0" : "=r" (tnpc));
		else if (X_FCN(insn) == 1)	/* retry */
			__asm__ __volatile__("rdpr %%tpc, %0" : "=r" (tnpc));
		else
			tnpc = 0;
		
		*nPC1 = tnpc;
		*nPC2 = 0;
		return;
	}
	
	/* All other instructions go to tnpc */
npc:
	*nPC1 = ef->tnpc;
	*nPC2 = 0;
}

static int
is_branch(unsigned long pc)
{
	unsigned long insn;
	
	if (kdb_getword(&insn, pc, 4)) {
		kdb_printf("is_branch: couldn't read instruction\n");
		return 1;	/* saying "yes" is safest */
	}
	
	/* Are we a branch? */

	if ((X_OP(insn) == 0 && X_OP2(insn) != 4) || /* branch or illtrap*/
	    (X_OP(insn) == 1) ||		     /* call */
	    (X_OP(insn) == 2 &&
	     (X_OP3(insn) == 0x30 ||		     /* sir */
	      X_OP3(insn) == 0x38 ||		     /* jmpl */
	      X_OP3(insn) == 0x39 ||		     /* return */
	      X_OP3(insn) == 0x3a ||		     /* trap */
	      X_OP3(insn) == 0x3c ||		     /* save */
	      X_OP3(insn) == 0x3d ||		     /* restore */
	      X_OP3(insn) == 0x3e)))		     /* done/retry */
	{
		return 1;
	}
	return 0;
}

void
kdba_setsinglestep(kdb_eframe_t ef)
{
	if (sstep1_va || sstep2_va) {
		kdb_printf("kdba_setsinglestep: called with uncleared ssteps\n");
		kdba_clearsinglestep(ef);
	}
	if (KDB_STATE(DOING_SSB) && !is_branch(ef->tpc)) {
		sstep1_va = ef->tnpc;
		sstep2_va = 0;
		while (!is_branch(sstep1_va))
			sstep1_va += 4;
	} else
		getnpcs(ef, &sstep1_va, &sstep2_va);

	if (sstep1_va == 0) {
		kdb_printf("kdba_setsinglestep: can't single step!\n");
		return;
	}
	
	kdb_getarea(sstep1_insn, sstep1_va);
	kdb_putword(sstep1_va, SPARC64_DB_INSN, 4);
	flushi(sstep1_va);
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_setsinglestep installed debug bp at 0x%lx, insn 0x%x\n",
			   sstep1_va, sstep1_insn);

	if (sstep2_va == 0)
		goto out;
	
	kdb_getarea(sstep2_insn, sstep2_va);
	kdb_putword(sstep2_va, SPARC64_DB_INSN, 4);
	flushi(sstep2_va);
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_setsinglestep installed debug bp at 0x%lx, insn 0x%x\n",
			   sstep2_va, sstep2_insn);
out:
	sstep_current = current;
	sstep_tstate = ef->tstate;
	ef->tstate &= ~TSTATE_IE; /* disable interrupts */
}

void
kdba_clearsinglestep(kdb_eframe_t ef)
{
	
	if (!sstep1_va)
		return;
	
	if (current != sstep_current) {
		kdb_printf("kdba_clearsinglestep: current (%p) != saved (%p)\n",
			   current, sstep_current);
		return;
	}
	
	ef->tstate |= (sstep_tstate & TSTATE_IE); /* restore IE bit */
	
	kdb_putword(sstep1_va, sstep1_insn, 4);
	flushi(sstep1_va);
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_clearsinglestep clearing at 0x%lx, insn 0x%x\n",
			   sstep1_va, sstep1_insn);
	sstep1_va = 0;

	if (sstep2_va == 0)
		return;

	kdb_putword(sstep2_va, sstep2_insn, 4);
	flushi(sstep2_va);
	if (KDB_DEBUG(BP))
		kdb_printf("kdba_clearsinglestep clearing at 0x%lx, insn 0x%x\n",
			   sstep2_va, sstep2_insn);
	sstep2_va = 0;
}

