#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <time.h>
#include <sys/time.h>

#define TEST_OK		0
#define TEST_FAILED	1
#define TEST_ABORTED	2

/* Wait adjustment in nanoseconds */
#define WAITADJUST	2100000

#define MAXSIGCNT	8

#define BSPFILE		"/nfs/backingstore"

/* Assume that readahead does not happen over 1M */
#define CHUNKSIZE	(1024*1024)
#define BSPSIZE		(MAXSIGCNT + 2)*CHUNKSIZE

int mykill(pid_t pid, int sig, void *bspstore);

static volatile int sigcnt;
static struct timeval sigtimes[MAXSIGCNT];

static void chld_handler(int signo)
{
	gettimeofday(sigtimes + sigcnt, NULL);
	++sigcnt;
}

static int run_debugger(pid_t pid)
{
	struct sigaction sa;
	long sigsum;
	struct timespec tm;
	int stat;
	int i;
	int res;

	printf("Running debugger on pid %ld\n", (long) pid);

	if (waitpid(pid, &stat, 0) == -1) {
		perror("waitpid attach");
		goto error_out;
	}

	sa.sa_handler = chld_handler;
	sa.sa_flags = 0;
	sigemptyset(&sa.sa_mask);
	if (sigaction(SIGCHLD, &sa, NULL)) {
		perror("sigaction");
		goto error_out;
	}

	while (sigcnt < MAXSIGCNT) {
		int oldsigcnt = sigcnt;
		if (ptrace(PTRACE_CONT, pid, 0, 0) == -1) {
			perror("PTRACE_CONT loop");
			goto error_out;
		}
		while (oldsigcnt == sigcnt);
	}

	for (i = 0, sigsum = 0; i < MAXSIGCNT - 1; ++i)
		sigsum += sigtimes[i+1].tv_usec - sigtimes[i].tv_usec
			+ (sigtimes[i+1].tv_sec - sigtimes[i].tv_sec) * 1000000;

	tm.tv_sec = 0;
	tm.tv_nsec = 1000 * sigsum / (MAXSIGCNT - 1) - WAITADJUST;
	printf("Waiting for %f ms\n", tm.tv_nsec / 1000.0);

	if (ptrace(PTRACE_CONT, pid, 0, 0) == -1) {
		perror("PTRACE_CONT");
		goto error_out;
	}
	if (nanosleep(&tm, NULL))
		printf("Notification already delivered\n");

	if (kill(pid, SIGKILL) == -1) {
		perror("kill SIGKILL");
		goto error_out;
	}

	if (waitpid(pid, &stat, 0) == -1) {
		perror("waitpid exit");
		res = TEST_FAILED;
	} else if (WIFSIGNALED(stat)) {
		if (WTERMSIG(stat) != SIGKILL) {
			fprintf(stderr, "Child terminated by signal %d\n",
				WTERMSIG(stat));
			res = TEST_FAILED;
		}
	} else if (WIFSTOPPED(stat)) {
		fprintf(stderr, "Child notified us about signal %d - FAILED\n",
			WSTOPSIG(stat));
		res = TEST_FAILED;
	} else if (!WIFEXITED(stat)) {
		fprintf(stderr, "Child terminated abnormally, stat=0x%x\n",
			stat);
		res = TEST_FAILED;
	} else {
		fprintf(stderr, "Child exited with code %d\n",
			WEXITSTATUS(stat));
		res = TEST_FAILED;
		if (res < WEXITSTATUS(stat))
			res = WEXITSTATUS(stat);
	}

	return res;

error_out:
	ptrace(PTRACE_KILL, pid, 0, 0);

	return TEST_ABORTED;
}

static void *getbspbase(void)
{
	int fd;
	void *ret;

	if ((fd = open(BSPFILE, O_RDWR|O_CREAT|O_TRUNC, 0600)) < 0) {
		perror("create " BSPFILE);
		return NULL;
	}

	if (ftruncate(fd, BSPSIZE)) {
		perror("ftruncate");
		close(fd);
		return NULL;
	}

	ret = mmap(NULL, BSPSIZE, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if (fsync(fd)) {
		perror("fsync");
		return NULL;
	}

	if (close(fd)) {
		perror("close");
		return NULL;
	}

	return ret;
}

static int run_child(void)
{
	void *bsp = NULL;
	pid_t mypid = getpid();

	/* Attach ourselves */
	if (ptrace(PTRACE_TRACEME, 0, 0, 0) == -1) {
		perror("TRACE_ME");
		return TEST_ABORTED;
	}

	if ((bsp = getbspbase()) == NULL)
		return TEST_ABORTED;

	for (;;) {
		struct timespec ts;
		ts.tv_sec = 0;
		ts.tv_nsec = 1000000;
		nanosleep(&ts, NULL);
		mykill(mypid, SIGCHLD, bsp);
		bsp += CHUNKSIZE;
	}

	/* Should not be reached */
	return TEST_FAILED;
}

int main()
{
	pid_t pid;

	if ((pid = fork())) {
		int ret;
		ret = run_debugger(pid);
		unlink(BSPFILE);
		return ret;
	} else
		return run_child();
}
