/*
 * Queue spinlock
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * (C) Copyright 2013-2014 Hewlett-Packard Development Company, L.P.
 *
 * Authors: Waiman Long <waiman.long@hp.com>
 *          Peter Zijlstra <pzijlstr@redhat.com>
 */

/*
 * The basic principle of a queue-based spinlock can best be understood
 * by studying a classic queue-based spinlock implementation called the
 * MCS lock. The paper below provides a good description for this kind
 * of lock.
 *
 * http://www.cise.ufl.edu/tr/DOC/REP-1992-71.pdf
 *
 * This queue spinlock implementation is based on the MCS lock, however to make
 * it fit the 4 bytes we assume spinlock_t to be, and preserve its existing
 * API, we must modify it some.
 *
 * In particular; where the traditional MCS lock consists of a tail pointer
 * (8 bytes) and needs the next pointer (another 8 bytes) of its own node to
 * unlock the next pending (next->locked), we compress both these: {tail,
 * next->locked} into a single u32 value.
 *
 * Since a spinlock disables recursion of its own context and there is a limit
 * to the contexts that can nest; namely: task, softirq, hardirq, nmi, we can
 * encode the tail as and index indicating this context and a cpu number.
 *
 * We can further change the first spinner to spin on a bit in the lock word
 * instead of its node; whereby avoiding the need to carry a node from lock to
 * unlock, and preserving API.
 */

#include "qspinlock.h"
#include "linux.h"

struct mcs_spinlock {
	struct mcs_spinlock *next;
	int locked;
	int count;
};

#ifndef arch_mcs_spin_lock_contended
/*
 * Using smp_load_acquire() provides a memory barrier that ensures
 * subsequent operations happen after the lock is acquired.
 */
#define arch_mcs_spin_lock_contended(l)					\
do {									\
	while (!(smp_load_acquire(l)))					\
		arch_mutex_cpu_relax();					\
} while (0)
#endif

#ifndef arch_mcs_spin_unlock_contended
/*
 * smp_store_release() provides a memory barrier to ensure all
 * operations in the critical section has been completed before
 * unlocking.
 */
#define arch_mcs_spin_unlock_contended(l)				\
	smp_store_release((l), 1)
#endif

/*
 * Per-CPU queue node structures; we can never have more than 4 nested
 * contexts: task, softirq, hardirq, nmi.
 *
 * Exactly fits one cacheline.
 */
static DEFINE_PER_CPU_ALIGNED(struct mcs_spinlock, mcs_nodes[4]);

/*
 * We must be able to distinguish between no-tail and the tail at 0:0,
 * therefore increment the cpu number by one.
 */
// BUILD_BUG_ON(NR_CPUS >= (1U << _Q_TAIL_CPU_BITS));

static inline u32 encode_tail(int cpu, int idx)
{
	u32 tail;

	tail  = (cpu + 1) << _Q_TAIL_CPU_OFFSET;
	tail |= idx << _Q_TAIL_IDX_OFFSET; /* assume < 4 */

	return tail;
}

static inline struct mcs_spinlock *decode_tail(u32 tail)
{
	int cpu = (tail >> _Q_TAIL_CPU_OFFSET) - 1;
	int idx = (tail >> _Q_TAIL_IDX_OFFSET) & _Q_TAIL_IDX_MASK;

	return per_cpu_ptr(&mcs_nodes[idx], cpu);
}

/*
 * 0,0,1 -> 0,1,* ; pending
 *
 * Ignore the locked bit; if we set pending and locked happens to be clear
 * we'll fall through on the subsequent wait.
 */
static int __always_inline
try_set_pending(struct qspinlock *lock, u32 val)
{
	if (val & ~_Q_LOCKED_MASK)
		return 0; /* fail; queue */

	/*
	 * if we find the pending bit was already set; fail and queue.
	 */
	if (atomic_test_and_set_bit(_Q_PENDING_OFFSET, &lock->val))
		return 0;

	/*
	 * if we raced and someone concurrently set a tail; we must release
	 * the pending bit because they might not have observed our pending
	 * bit and have claimed ownership.
	 */
	if (unlikely(atomic_read(&lock->val) & _Q_TAIL_MASK)) {
		atomic_clear_mask(_Q_PENDING_MASK, &lock->val);
		return 0; /* fail; queue */
	}

	return 1;
}

#if _Q_PENDING_BITS == 8

struct __qspinlock {
	union {
		atomic_t val;
		struct {
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
			u8	locked;
			u8	pending;
			u16	tail;
#else
			u16	tail;
			u8	pending;
			u8	locked;
#endif
		};
	};
};

/*
 * take ownership and clear the pending bit.
 *
 * *,1,0 -> *,0,1
 */
static int __always_inline
try_clear_pending_set_locked(struct qspinlock *lock, u32 val)
{
	struct __qspinlock *l = (void *)lock;

	ACCESS_ONCE(l->locked) = 1;
	/*
	 * we must order the stores of locked and pending such that the
	 * (locked,pending) tuple never observably becomes 0.
	 *
	 * 'matched' by the queue wait loop.
	 */
	smp_wmb();
	ACCESS_ONCE(l->pending) = 0;

	return 1;
}

/*
 * xchg(lock, tail)
 *
 * p,*,* -> n,*,* ; prev = xchg(lock, node)
 */
static u32 __always_inline
xchg_tail(struct qspinlock *lock, u32 tail)
{
	struct __qspinlock *l = (void *)lock;

	return (u32)xchg(&l->tail, tail >> _Q_TAIL_OFFSET) << _Q_TAIL_OFFSET;
}

#else

/*
 * take ownership and clear the pending bit.
 *
 * *,1,0 -> *,0,1
 */
static int __always_inline
try_clear_pending_set_locked(struct qspinlock *lock, u32 val)
{
	u32 new, old;

	for (;;) {
		new = (val & ~_Q_PENDING_MASK) | _Q_LOCKED_VAL;

		old = atomic_cmpxchg(&lock->val, val, new);
		if (old == val)
			break;

		if (unlikely(old & _Q_LOCKED_MASK))
			return 0;

		val = old;
	}

	return 1;
}

/*
 * xchg(lock, tail)
 *
 * p,*,* -> n,*,* ; prev = xchg(lock, node)
 */
static u32 __always_inline
xchg_tail(struct qspinlock *lock, u32 tail)
{
	u32 old, new, val = atomic_read(&lock->val);

	/*
	 * guess the pending bit; if we race no harm done because we'll
	 * unconditionally set the locked bit, so we can always fix up.
	 *
	 * we must always assume the lock bit is set; accidentially clearing it
	 * would release pending waiters and screw up our ordering.
	 */
	new = tail | (val & _Q_PENDING_MASK) | _Q_LOCKED_VAL;
	old = atomic_xchg(&lock->val, new);

	/*
	 * fixup the pending bit; since we now have a tail the pending bit is
	 * stable, see try_pending() and since we have the locked bit set,
	 * nothing can claim the lock and make progress.
	 */
	if (unlikely((old & _Q_PENDING_MASK) != (new & _Q_PENDING_MASK))) {
		if (old & _Q_PENDING_MASK)
			atomic_clear_mask(_Q_PENDING_MASK, &lock->val);
		else
			atomic_set_mask(_Q_PENDING_VAL, &lock->val);
	}

	/*
	 * fixup the locked bit; having accidentally set the locked bit
	 * we must make sure our wait-acquire loops are robust and not
	 * set the locked bit when its already set, otherwise we cannot
	 * release here.
	 */
	if (unlikely(!(old & _Q_LOCKED_MASK)))
		queue_spin_unlock(lock);

	return old; /* tail bits are still fine */
}

#endif /* _Q_PENDING_BITS == 8 */

#define _Q_LOCKED_PENDING_MASK	(_Q_LOCKED_MASK | _Q_PENDING_MASK)

/**
 * queue_spin_lock_slowpath - acquire the queue spinlock
 * @lock: Pointer to queue spinlock structure
 * @val: Current value of the queue spinlock 32-bit word
 *
 *
 *              fast      :    slow                                  :    unlock
 *                        :                                          :
 * uncontended  (0,0,0) --:--> (0,0,1) ------------------------------:--> (*,*,0)
 *                        :       | ^--------.------.             /  :
 *                        :       v           \      \            |  :
 * pending                :    (0,1,1) +--> (0,1,0)   \           |  :
 *                        :       | ^--'              |           |  :
 *                        :       v                   |           |  :
 * uncontended            :    (n,x,y) +--> (n,0,0) --'           |  :
 *   queue                :       | ^--'                          |  :
 *                        :       v                               |  :
 * contended              :    (*,x,y) +--> (*,0,0) ---> (*,0,1) -'  :
 *   queue                :         ^--'                             :
 *
 */
void queue_spin_lock_slowpath(struct qspinlock *lock, u32 val)
{
	struct mcs_spinlock *prev, *next, *node;
	u32 new, old, tail;
	int idx;

	/*
	 * 0,0,1 -> 0,1,1 ; pending
	 */
	if (!try_set_pending(lock, val))
		goto queue;

	/*
	 * we're pending, wait for the owner to go away.
	 *
	 * *,1,1 -> *,1,0
	 *
	 * this wait loop must be a load-acquire such that we match the
	 * store-release that clears the locked bit and create lock
	 * sequentiality; this because not all try_clear_pending_set_locked()
	 * implementations imply full barriers.
	 */
retry_pending_wait:
	while ((val = smp_load_acquire(&lock->val.counter)) & _Q_LOCKED_MASK)
		cpu_relax();

	/*
	 * take ownership and clear the pending bit.
	 *
	 * *,1,0 -> *,0,1
	 */
	if (!try_clear_pending_set_locked(lock, val))
		goto retry_pending_wait;

	return;

queue:
	node = this_cpu_ptr(&mcs_nodes[0]);
	idx = node->count++;
	tail = encode_tail(smp_processor_id(), idx);

	node += idx;
	node->locked = 0;
	node->next = NULL;

	/*
	 * we touched a (possibly) cold cacheline; attempt the trylock once
	 * more in the hope someone let go while we weren't watching.
	 */
	if (queue_spin_trylock(lock))
		goto release;

	/*
	 * we already touched the queueing cacheline; don't bother with pending
	 * stuff.
	 *
	 * p,*,* -> n,*,*
	 */
	old = xchg_tail(lock, tail);

	/*
	 * if there was a previous node; link it and wait.
	 */
	if (old & _Q_TAIL_MASK) {
		prev = decode_tail(old);
		ACCESS_ONCE(prev->next) = node;

		arch_mcs_spin_lock_contended(&node->locked);
	}

	/*
	 * we're at the head of the waitqueue, wait for the owner & pending to
	 * go away.
	 *
	 * *,x,y -> *,0,0
	 */
retry_queue_wait:
	while ((val = atomic_read(&lock->val)) & _Q_LOCKED_PENDING_MASK)
		cpu_relax();

	/*
	 * claim the lock:
	 *
	 * n,0,0 -> 0,0,1 : lock, uncontended
	 * *,0,0 -> *,0,1 : lock, contended
	 */
	for (;;) {
		new = _Q_LOCKED_VAL;
		if (val != tail)
			new |= val;

		old = atomic_cmpxchg(&lock->val, val, new);
		if (old == val)
			break;

		/*
		 * ensure to never claim a locked value; see xchg_tail().
		 */
		if (unlikely(old & _Q_LOCKED_PENDING_MASK))
			goto retry_queue_wait;

		val = old;
	}

	/*
	 * contended path; wait for next, release.
	 */
	if (new != _Q_LOCKED_VAL) {
		while (!(next = ACCESS_ONCE(node->next)))
			arch_mutex_cpu_relax();

		arch_mcs_spin_unlock_contended(&next->locked);
	}

release:
	/*
	 * release the node
	 */
	this_cpu_ptr(&mcs_nodes[0])->count--;
//	this_cpu_dec(mcs_nodes[0].count);
}
EXPORT_SYMBOL(queue_spin_lock_slowpath);
