#include "tf.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <ext2fs/ext2fs.h>
#include <ext2fs/ext2_fs.h>

int Report(long int rc, char *node, int fd)
{
	int ret;
	reportrec tmp;

	strcpy(tmp.node, node);
	tmp.flag = rc;

	ret = write(fd, &tmp, sizeof(tmp));
	if (ret != sizeof(tmp))
	{
		printf("Partial/Failed write to report file\n");
		return -2;
	}

	return 0;
}

long int ext2_cmp(struct ext2_inode *org, struct ext2_inode *cur)
{
	int i;
	long int flg = 0;
	
	if (org->i_mode != cur->i_mode)
		flg |= 1;
	if (org->i_uid != cur->i_uid)
		flg |= (1 << 1);
	if (org->i_size != cur->i_size)
		flg |= (1 << 2);
	if (org->i_atime != cur->i_atime)
		flg |= (1 << 3);
	if (org->i_ctime != cur->i_ctime)
		flg |= (1 << 4);
	if (org->i_mtime != cur->i_mtime)
		flg |= (1 << 5);
	if (org->i_dtime != cur->i_dtime)
		flg |= (1 << 6);
	if (org->i_gid != cur->i_gid)
		flg |= (1 << 7);
	if (org->i_links_count != cur->i_links_count)
		flg |= (1 << 8);
	if (org->i_blocks != cur->i_blocks)
		flg |= (1 << 9);
	if (org->i_flags != cur->i_flags)
		flg |= (1 << 10);
	for (i = 0; i < EXT2_N_BLOCKS; i++)
	{
		if (org->i_block[i] != cur->i_block[i])
		{
			flg |= (1 << 11);
			break;
		}
	}
	
	return flg;
}

int ext2_compare(char *dev_name, char *statefile, int n)
{
	errcode_t rc;
	ext2_filsys fs;
	strec tmp;
	struct ext2_inode ind_cur;

	long int flags = EXT2_FLAG_FORCE;
	int sb = 0;
	int blksz = 0;

	int sfd, rfd;
	int ret, i;

	rc = ext2fs_open(dev_name, flags, sb, blksz, unix_io_manager, &fs);
	if (rc)
	{
		printf("Error in opening file system\n");
		return -2;
	}

	sfd = open(statefile, O_RDONLY);
	if (!sfd)
	{
		printf("Error opening state file\n");
		ext2fs_close(fs);
		return -2;
	}

	rfd = open("report", O_WRONLY | O_CREAT | O_SYNC);
	if (!rfd)
	{
		printf("Error opening report file\n");
		ext2fs_close(fs);
		close(sfd);
		return -2;
	}

	for (i = 0; i < n; i++)
	{
		ret = read(sfd, &tmp, sizeof(tmp));
		if (ret != sizeof(tmp))
		{
			printf("Partial/Failed read from state file\n");
			// return -2;	// should we return?
			continue;
		}

		rc = ext2fs_read_inode(fs, tmp.ino, &ind_cur);
		if (rc)
		{
			printf("Node lost = %s\n", tmp.node);
			Report(0, tmp.node, rfd);
		}
		else if ( (flags = ext2_cmp(&tmp.ind, &ind_cur) != 0) )
		{
			printf("Node corrupt = %s, code = %ld\n", tmp.node, flags);
			Report(flags, tmp.node, rfd);
		}
	}

	close(rfd);
	close(sfd);
	ext2fs_close(fs);

	return 0;
}

int main(int argc, char *argv[])
{
	int rc, fd, n;

	if (argc < 4)
	{
		printf("Usage: %s <device name> <fstype> <state file>\n", argv[0]);
		printf("\tdevice name = name of the device/name of file system image whose state is to be recorded\n");
		printf("\tfstype = file system type on the named device\n");
		printf("\tstate file = name of the file where the state information is to be stored\n");

		return -1;
	}

	fd = open("icount", O_RDONLY);
	if (!fd)
	{
		printf("Error opening icount file\n");
		return -2;
	}

	rc = read(fd, &n, sizeof(n));
	if (rc != sizeof(n))
	{
		printf("Partial/Failed read from icount file\n");
		return -2;
	}

	close(fd);

	printf("Inode count = %d\n", n);
	
	if (strcmp(argv[2], "ext2") == 0)
	{
		rc = ext2_compare(argv[1], argv[3], n);
		return rc;
	}
	else
	{
		printf("Unrecognised file system type\n");
		return -1;
	}
}
