#include "tf.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <ext2fs/ext2_fs.h>
#include <ext2fs/ext2fs.h>

char *oclog;
char *iclog;

int ofd = 0;
int ifd = 0;

int getseed(void)
{
	int fd = open("/dev/urandom", O_RDONLY);
	int r;
	
	if (fd < 0)
	{
		perror("***open");
		exit(0);
	}
	
	read(fd, &r, sizeof(r));
	close(fd);
	
	return r;
}

void LogInit()
{
	ofd = open(oclog, O_WRONLY | O_CREAT | O_SYNC);
	if (!ofd)
	{
		printf("***Failed to open output log file\n");
	}

	if (iclog)
	{
		ifd = open(iclog, O_RDONLY);
		if (!ifd)
		{
			printf("***Failed to open input log file\n");
		}
	}
}

int LogWrite(void *f, int n)
{
	int rc;
	
	rc = write(ofd, f, n);
	if (rc == -1)
	{
		printf("***Error in writing to the corruption log file\n");
	}
	else if (rc != n)
	{
		printf("***Incomplete write to corruption log file\n");
	}
	
	return 0;
}

int LogRead(void *f, int n)
{
	int rc;

	rc = read(ifd, f, n);
	if (rc == -1)
	{
		printf("***Error in reading from corruption log file\n");
		return -1;
	}
	else if (rc != n)
	{
		printf("***Incomplete read from corruption log file\n");
		return -1;
	}
	
	return 0;
}

void LogExit()
{
	int rc;
	
	if (ofd)
	{
		rc = close(ofd);
		if (rc)
		{
			printf("***Failed to close output log file\n");
		}
	}

	if (ifd)
	{
		rc = close(ifd);
		if (rc)
		{
			printf("***Failed to close input log file\n");
		}
	}
}

// replayable corruption routine
int ext2_corrupt(char *dev_name, char *state_file, int icount)
{
	errcode_t ret;
	ext2_filsys fs;
	ext2_ino_t ino;
	struct ext2_inode *ind;

	long int flags = EXT2_FLAG_RW | EXT2_FLAG_FORCE;
	int sb = 0;
	int blksz = 0;

	unsigned int i, n, k;
	int rc, fd;
	crec tmp;
	strec t;
	struct stat stbuf;

	fd = open(state_file, O_RDONLY);
	if (!fd)
	{
		printf("Error opening state file\n");
		return -2;
	}

	ret = ext2fs_open(dev_name, flags, sb, blksz, unix_io_manager, &fs);
	if (ret)
	{
		printf("Error in opening file system\n");
		close(fd);
		return ret;
	}

	LogInit();

	// only a small percentage of files to be corrupted
	// TODO - decide on the small percentage
	if (!ifd)
	{
		srand(getseed());
		n = (unsigned) rand() % icount;

		sprintf(tmp.node, "%u", n);
		
		if (ofd)
		{
			rc = LogWrite(&tmp, sizeof(tmp));
		}
	}
	else
	{
		rc = LogRead(&tmp, sizeof(tmp));
		if (rc == -1)
		{
			return -1;
		}
		
		n = atoi(tmp.node);
		
		if (ofd)
		{
			rc = LogWrite(&tmp, sizeof(tmp));
		}
	}

	ind = (struct ext2_inode*) malloc(sizeof(struct ext2_inode));
	if (!ind)
	{
		printf("Out of memory\n");
		return -1;
	}
	
	memset(ind, 0, sizeof(struct ext2_inode));

	for (i = 0; i < n; i++)
	{
		if (!ifd)
		{
			k = (unsigned) rand() % icount;

			rc = lseek(fd, k*sizeof(strec), SEEK_SET);
			if (rc != k*sizeof(strec))
			{
				perror("***Error in seeking");
				return -1;
			}

			rc = read(fd, &t, sizeof(t));
			if (rc != sizeof(t))
			{
				printf("***Error in reading state file");
				return -1;	// should we return?
			}

			strcpy(tmp.node, t.node);
			k = t.ino;

			printf("ino to corrupt = %u\n", k);
		}
		else
		{
			rc = LogRead(&tmp, sizeof(tmp));
			if (rc == -1)
			{
				return -1;
			}
			
			rc = lstat(tmp.node, &stbuf);
			if (rc != 0)
			{
				printf("***Error in stating file  %s\n", tmp.node);
				return -1;
			}
			
			k = stbuf.st_ino;
		}

		if (ofd)
		{
			rc = LogWrite(&tmp, sizeof(tmp));
		}

		ino = (ext2_ino_t)k;
		
		ret = ext2fs_write_inode(fs, ino, ind);
		if (ret)
		{
			printf("Error in writing corrupt inode\n");
			// should we return here
		}
	}
	
	ext2fs_close(fs);
	LogExit();
	close(fd);

	return 0;
}

int main(int argc, char *argv[])
{
	int rc;
	int fd;
	int n;

	if (argc < 4)
	{
		printf("Usage: %s <device|file name> <fstype> <state file> [-i <input log file>] [-o <output log file>]\n", argv[0]);
		printf("\tdevice|file name = file system on which has to be corrupted\n");
		printf("\tfstype = file system type\n");
		printf("\tstate file = state file recorded for the named device\n");
		printf("\tinput log file = file which specifies how corruption is to be done\n");
		printf("\toutput log file = file to record the corruption\n");
		
		return -1;
	}

	iclog = NULL;
	oclog = "Clog";

	if (argc >= 6)
	{
		if (strcmp(argv[4], "-o") == 0)
		{
			oclog = argv[5];
		}
		else if (strcmp(argv[4], "-i") == 0)
		{
			iclog = argv[5];
		}
	}

	if (argc >= 8)
	{
		if (strcmp(argv[6], "-o") == 0)
		{
			oclog = argv[7];
		}
		else if (strcmp(argv[6], "-i") == 0)
		{
			iclog = argv[7];
		}

	}

	fd = open("icount", O_RDONLY);
	if (!fd)
	{
		printf("Error opening icount file\n");
		return -2;
	}

	rc = read(fd, &n, sizeof(n));
	if (rc != sizeof(n))
	{
		printf("Partial/Failed read from icount file\n");
		return -2;
	}

	printf("\nInode count = %d\n", n);

	close(fd);

	if (strcmp(argv[2], "ext2") == 0)
	{
		rc = ext2_corrupt(argv[1], argv[3], n);
		return rc;
	}
	else
	{
		printf("Unrecognised file system\n");
		return -1;
	}
}
