#!/bin/bash
# Test Framework
#----------------------------------------------------------------
# Global variables

dur=0

fstype=$1
imgname=$2
sz=$3
blksize=$4
mountpt=$5
statefile=$6
iter=$7
ilog=$8

#----------------------------------------------------------------
# Usage

Usage()
{
	echo -e "Usage: $0 <fs type> <fs image name> <fs image size> <fs block size> <mount point> <state file> <iterations> [input log]"
	echo -e "\tfs type = file system type (ext2)"
	echo -e "\tfs image name = name of file system image"
	echo -e "\tfs image size = size of the file system image (in KB)"
	echo -e "\tfs block size = block size to be used for file system image (in B)"
	echo -e "\tmount point = mount point for the file system"
	echo -e "\tstate file = file used to store the state of file system"
	echo -e "\titerations = number of iterations to perform"
	echo -e "\tinput log = corruption log file used to perform corruption (optional)"

	exit 1
}

#----------------------------------------------------------------
# Prepare Phase

PreparePhase()
{
	if [ -e $imgname ]
	then
		echo "**Old file system image found....."
	else
		echo "**Creating new file system image....."
		dd if=/dev/zero of=$imgname bs=1k count=$sz
		case $fstype in
			ext2)
				/sbin/mke2fs -b $blksize -F $imgname
				;;
			ext3)
				/sbin/mke2fs -b $blksize -j -F $imgname
				;;
			vfat)
				/sbin/mkfs.vfat -C $imgname 1440
				;;
			msdos)
				/sbin/mkdosfs -C $imgname 1440
				;;
			hfs)
				hformat $imgname
				;;
			swap)
				/sbin/mkswap $imgname
				;;
			# Let xfs extend the file image, it needs about 16m
			xfs)
				/sbin/mkfs.xfs -b size=$blksize -f -d name=$imgname,agsize=16m
				;;
			*)
				/sbin/mkfs -t $fstype -f $imgname
				;;	
		esac
	fi

	mkdir -p $mountpt
	if [ $? -ne 0 ]
	then
		echo "***Failed to create or find mountpoint"
		exit $?
	fi

	mount -t $fstype -o loop $imgname $mountpt
	if [ $? -ne 0 ]
	then
		echo "***Unable to mount file system"
		exit $?
	fi

	echo "***Populating file system with data"
	# logic to check the avail space and then populate it
	cp -r /etc $mountpt
	#cp -r /dev $mountpt
	#cp -r /lib $mountpt
	cp -r /sbin $mountpt
	#cp -r /boot $mountpt
	#cp -r /var $mountpt

	# aging of file system can be done here

	echo "***PP done"

	sync
	return
}

#----------------------------------------------------------------
# State Record

StateRecord()
{
	./StateRecord $imgname $mountpt $fstype $statefile
	if [ $? -le 0 ]
	then
		echo "***Failed to record state of the file system"
		exit $?
	fi

	sync
	return
}

#----------------------------------------------------------------
# Corruption Phase

CorruptionPhase()
{
	if [ -z $ilog ]
	then
		./CorruptionPhase $imgname $fstype $statefile
	else
		./CorruptionPhase $imgname $fstype $statefile -i $ilog
	fi

	if [ $? -ne 0 ]
	then
		echo "***Failed corruption phase"
		exit $?
	fi

	sync
	return
}

#----------------------------------------------------------------
# Repair Phase

RepairPhase()
{
	sync
	umount -dfl $mountpt
	if [ $? -ne 0 ]
	then
		echo "***Error umounting file system before fsck"
		exit $?
	fi

	start_time=`date +%s`
	fsck.$fstype -f -y $imgname
	if [ $? -ge 4 ]
	then
		echo "***fsck error"
		exit $?
	fi
	end_time=`date +%s`

	tmp=`expr $end_time - $start_time`
	dur=`expr $dur + $tmp`

	echo "Duration = "$tmp

	sync
	mount -t $fstype -o loop $imgname $mountpt
	if [ $? -ne 0 ]
	then
		echo "***Recovered file system can't be mounted"
		exit $?
	fi

	echo "**RP done"

	sync
	return
}

#----------------------------------------------------------------
# Compare Phase

ComparePhase()
{
	./ComparePhase $imgname $fstype $statefile
	if [ $? -ne 0 ]
	then
		echo "***Failed Compare Phase"
		exit $?
	fi

	sync	
	return
}

#----------------------------------------------------------------
# Summary Phase

SummaryPhase()
{
	./SummaryPhase $fstype $imgname $sz $blksize $iter $dur
	if [ $? -ne 0 ]
	then
		echo "***Failed Summary Phase"
		exit $?
	fi

	sync
	return
}

#----------------------------------------------------------------
# Main

if [ $# -lt 7 ]
then
	Usage
fi

i=0

PreparePhase
while [ $i -lt $iter ]
do
	StateRecord
	CorruptionPhase
	RepairPhase
	ComparePhase

	mv report report$i
	i=`expr $i + 1`
done
SummaryPhase

exit 0
