#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <time.h>
#include "minilzo.h"
#include "helpers.h"

#define LZO1X_WORKMEM_SIZE LZO1X_1_MEM_COMPRESS
#define IFNAME "/home/madman/tinyLZO/test.c"

char *base, *comp, *decomp, *work;
unsigned int setup() {
  size_t len;
  int err, t, input_file;
  char errbuff[4096];
  struct stat f_info;

  err = 0;
  BUGCHK( lzo_init() != LZO_E_OK, "Unable to initialize miniLZO: %s",
	  "Internal Error" );
  BUGCHK( (input_file = open( IFNAME, O_RDWR )) == -1,
	  "Unable to open %s: %s", IFNAME, strerror(errno) );
  BUGCHK( lstat( IFNAME, &f_info ) == -1,
	  "fstat( [fd = ]%i, &f_info ) failed: %s",
	  input_file, strerror(errno) );
  len = f_info.st_size;
  len += 3;
 
  base = malloc( len );
  BUGCHK( base == NULL, "malloc( [size =]%i )", len, strerror(errno) );
  memset( base, 0, len );
  t = read( input_file, base, len );
  BUGCHK( t == -1, "read() failed: %s", strerror(errno) );
  BUGCHK(t<(len-3),"read() failed - was supposed to read %i bytes, read %i",
	  len, t );

  comp = (char *)malloc( len );
  BUGCHK( comp == NULL, "comp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( comp, 0, len ); 
  decomp = (char *)malloc( len );
  BUGCHK( decomp == NULL, "decomp = (char *)malloc( [len = ]%i ) failed: %s",
	  len, strerror(errno) );
  memset( decomp, 0, len );
  work = (char *)malloc( LZO1X_WORKMEM_SIZE );
  BUGCHK( work == NULL, "work = (char *)malloc( [len =]%i ) failed: %s",
	  LZO1X_WORKMEM_SIZE, strerror(errno) );
  memset( work, 0, LZO1X_WORKMEM_SIZE );

 err_out: if( err ) {
   free( work );
   free( decomp );
   free( comp );
   free( base );
   close( input_file );
   return -1;
 }

  close( input_file );
  return len;
}

int compress( unsigned int len ) {
  unsigned int rlen, olen;
  int err = 0, t, k;
  unsigned long long val;
  char errbuff[4096];
  struct timeval b, a;

  rlen = len - 3;
  olen = rlen;
  
  k = gettimeofday( &b, NULL );
  t = lzo1x_1_compress( base, rlen, comp, &olen, work );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != LZO_E_OK,
	  "%s ( t == %i )", "lzo1x_compress has failed", t);
  if( a.tv_sec > b.tv_sec ) {
    t = a.tv_sec - b.tv_sec;
    val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
    INFO( "run took %i seconds, %llu microseconds", t, val );
  } else {
    val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
    INFO( "run took %llu microseconds", val );
  }

 err_out: if( err ) {
    free( work );
    free( decomp );
    free( comp );
    free( base );
    return -1;
  }

  return olen;
}

int decompress( unsigned int size, unsigned int orig ) {
  unsigned int olen;
  int err = 0, t, k;
  unsigned long long val;
  char errbuff[4096];
  struct timeval a, b;

  olen = orig;
  k = gettimeofday( &b, NULL );
  t = lzo1x_decompress( comp, size, decomp, &olen, NULL );
  k = gettimeofday( &a, NULL );
  BUGCHK( t != 0,
	  "%s ( t == %i )", "lzo1x_decompress has failed", t);

  if( a.tv_sec > b.tv_sec ) {
    t = a.tv_sec - b.tv_sec;
    val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
    INFO( "run took %i seconds, %llu microseconds", t, val );
  } else {
    val = a.tv_usec > b.tv_usec?a.tv_usec - b.tv_usec:b.tv_usec - a.tv_usec;
    INFO( "run took %llu microseconds", val );
  }

 err_out: if( err ) {
    free( work );
    free( decomp );
    free( comp );
    free( base );
    return -1;
  }

  return olen;
}

int main() {
  unsigned int clen, dlen, err, len;
  char errbuff[4096];

  len = setup();
  BUGCHK( len <= 0, "setup() failed - result code: %i", len );

  clen = compress( len );
  BUGCHK( clen == -1, "%s", "Compression routine failure" );
  dlen = decompress( clen, len );
  BUGCHK( dlen == -1, "%s", "Decompression routine failure" );

 err_out:
  if( err ) {
    return 1;
  }
  return 0;
}
