#include <linux/module.h>
#include <linux/device.h>
#include <linux/sysfs.h>
#include <linux/debugfs.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <asm/uaccess.h>

#ifndef CONFIG_DEBUG_FS
#error sysfs-interpreter requires debug FS
#endif

#define BUF_SIZE	PAGE_SIZE
#define MAX_TOKENS	4
#define MAX_PATH_LEN	480

#define SINTP_TYPE_DIR	0
#define SINTP_TYPE_FILE	1
#define SINTP_TYPE_BIN	2
#define SINTP_TYPE_LINK	3

#define SINTP_ROOT_PATH	"test"

struct sintp_cmd_ent {
	const char *cmdstr;
	int min_args;
	int max_args;
	int (*cmdfn)(char **toks, int nr_toks, int *pcmd_result);
};

struct sintp_sd_ent {
	struct list_head list;
	struct sysfs_dirent *sd;
	char *data;
	char path[MAX_PATH_LEN];
};

struct sintp_open_file {
	int cmd_result;
	char *buf;
};

static struct dentry *sintp_dentry;
static struct sysfs_dirent *sintp_root;

static DEFINE_SPINLOCK(sintp_sd_list_lock);
static DEFINE_SPINLOCK(sintp_data_lock);
static LIST_HEAD(sintp_sd_list);

static struct sintp_sd_ent *sintp_find_sd_ent(const char *path)
{
	struct sintp_sd_ent *sd_ent;

	list_for_each_entry(sd_ent, &sintp_sd_list, list)
		if (!strcmp(path, sd_ent->path))
			return sd_ent;
	return NULL;
}

static struct sintp_sd_ent *sintp_get_sd_ent(const char *path)
{
	struct sintp_sd_ent *sd_ent, *new_sd_ent = NULL;

 retry:
	spin_lock(&sintp_sd_list_lock);
	sd_ent = sintp_find_sd_ent(path);
	if (!sd_ent && new_sd_ent) {
		sd_ent = new_sd_ent;
		list_add(&sd_ent->list, &sintp_sd_list);
		new_sd_ent = NULL;
	}
	spin_unlock(&sintp_sd_list_lock);

	if (sd_ent) {
		if (new_sd_ent)
			kfree(new_sd_ent);
		return sd_ent;
	}

	new_sd_ent = kzalloc(sizeof(*new_sd_ent), GFP_KERNEL);
	if (!new_sd_ent)
		return NULL;

	strlcpy(new_sd_ent->path, path, MAX_PATH_LEN);
	goto retry;
}

static struct sysfs_dirent *sintp_find_sd(const char *path)
{
	struct sintp_sd_ent *sd_ent;

	spin_lock(&sintp_sd_list_lock);
	sd_ent = sintp_find_sd_ent(path);
	spin_unlock(&sintp_sd_list_lock);

	if (sd_ent)
		return sd_ent->sd;
	return NULL;
}

static void sintp_modify_sd_list(const char *from, const char *to)
{
	struct sintp_sd_ent *sd_ent;
	size_t from_len = strlen(from);

	spin_lock(&sintp_sd_list_lock);

	list_for_each_entry(sd_ent, &sintp_sd_list, list) {
		if (strncmp(from, sd_ent->path, from_len) ||
		    (sd_ent->path[from_len] != '/' &&
		     sd_ent->path[from_len] != '\0'))
			continue;

		if (to) {
			size_t copy_len;
			size_t to_len = strlen(to);

			copy_len = min_t(size_t,
					 strlen(sd_ent->path) - from_len,
					 MAX_PATH_LEN - to_len - 1);

			memmove(sd_ent->path + to_len, sd_ent->path + from_len,
				copy_len + 1);
			memcpy(sd_ent->path, to, to_len);
		} else
			sd_ent->sd = NULL;
	}

	spin_unlock(&sintp_sd_list_lock);
		
}

/*
 * sysfs file/bin ops
 */
static ssize_t sintp_sysfs_show(char *page, void *data, void *parent_data)
{
	struct sintp_sd_ent *self_sd_ent = data;
	struct sintp_sd_ent *parent_sd_ent = parent_data;

	return snprintf(page, PAGE_SIZE, "parent: %s\nself: %s\ndata: %s\n",
			parent_sd_ent->path, self_sd_ent->path,
			self_sd_ent->data ? self_sd_ent->data : "<none>");
}

static ssize_t sintp_sysfs_store(const char *page, size_t size, void *data,
				 void *parent_data)
{
	struct sintp_sd_ent *sd_ent = data;
	char *new_data;

	new_data = kzalloc(size + 1, GFP_KERNEL);
	if (!new_data)
		return -ENOMEM;

	memcpy(new_data, page, size);

	spin_lock(&sintp_data_lock);
	kfree(sd_ent->data);
	sd_ent->data = new_data;
	spin_unlock(&sintp_data_lock);

	return size;
}

static const struct sysfs_file_ops sintp_sysfs_fops = {
	.show = sintp_sysfs_show,
	.store = sintp_sysfs_store,
};

static ssize_t sintp_sysfs_read(char *buf, loff_t off, size_t size, void *data,
				void *parent_data)
{
	struct sintp_sd_ent *self_sd_ent = data;
	struct sintp_sd_ent *parent_sd_ent = parent_data;
	char *page;
	size_t count;

	page = kzalloc(PAGE_SIZE, GFP_KERNEL);
	if (!page)
		return -ENOMEM;

	count = snprintf(page, PAGE_SIZE, "parent: %s\nself: %s\ndata: %s\n",
			 parent_sd_ent->path, self_sd_ent->path,
			 self_sd_ent->data);

	if (off >= count)
		count = 0;
	count = min(count, size);
	memcpy(buf, page, count);
	kfree(page);

	return count;
}

static ssize_t sintp_sysfs_write(char *buf, loff_t off, size_t size, void *data,
				 void *parent_data)
{
	struct sintp_sd_ent *sd_ent = data;
	char *new_data;

	if (off)
		return -EINVAL;

	new_data = kzalloc(size + 1, GFP_KERNEL);
	if (!new_data)
		return -ENOMEM;

	memcpy(new_data, buf, size);

	spin_lock(&sintp_data_lock);
	kfree(sd_ent->data);
	sd_ent->data = new_data;
	spin_unlock(&sintp_data_lock);

	return size;
}

static const struct sysfs_bin_ops sintp_sysfs_bops = {
	.read	= sintp_sysfs_read,
	.write	= sintp_sysfs_write,
};

/*
 * sintp_cmd_*()
 */
static int sintp_sanitize_path(char *path)
{
	char *s, *d, c;
	int was_slash;

	s = d = path;
	was_slash = 1;

	while ((c = *s++)) {
		if (c == '/') {
			if (!was_slash)
				*d++ = '/';
			was_slash = 1;
			continue;
		}
		*d++ = c;
		was_slash = 0;
	}

	if (d == path)
		return -EINVAL;
	if (was_slash)
		d--;
	*d = '\0';

	if (d - path > MAX_PATH_LEN)
		return -ENAMETOOLONG;

	return 0;
}

static void sintp_split_basename(char *path, const char **pparent_path,
				 const char **pbasename)
{
	char *p;

	p = path + strlen(path) - 1;

	while (*p != '/' && p > path)
		p--;

	if (*p == '/') {
		*p = '\0';
		*pparent_path = path;
		*pbasename = p + 1;
	} else {
		*pparent_path = "";
		*pbasename = p;
	}
}

static int sintp_cmd_add(char **toks, int nr_toks, int type, int *pcmd_result)
{
	char *path = toks[0];
	struct sintp_sd_ent *sd_ent;
	struct sysfs_dirent *parent, *target = NULL;
	const char *parent_path, *name;
	struct sysfs_dirent *sd;
	mode_t mode = SYSFS_COPY_NAME;
	int rc;

	rc = sintp_sanitize_path(path);
	if (rc)
		return rc;

	/* get sd_ent */
	sd_ent = sintp_get_sd_ent(path);
	if (!sd_ent)
		return -ENOMEM;

	/* split basename and grab parent */
	sintp_split_basename(path, &parent_path, &name);

	parent = sintp_find_sd(parent_path);
	if (!parent)
		return -ENOENT;

	if (type == SINTP_TYPE_LINK) {
		char *target_path = toks[1];

		/* grab target */
		rc = sintp_sanitize_path(target_path);
		if (rc)
			return rc;

		target = sintp_find_sd(target_path);
		if (!target)
			return -ENOENT;
	} else {
		char *end;

		/* parse mode */
		mode = simple_strtoul(toks[1], &end, 0);
		if (end == toks[1] || *end != '\0')
			return -EINVAL;
	}

	/* plugged? */
	if (nr_toks == 3) {
		if (tolower(toks[2][0]) == 'p' && toks[2][1] == '\0')
			mode |= SYSFS_PLUGGED;
		else
			return -EINVAL;
	}

	/* copy name */
	mode |= SYSFS_COPY_NAME;

	/* okay, add it */
	switch (type) {
	case SINTP_TYPE_DIR:
		sd = sysfs_add_dir(parent, name, mode, sd_ent);
		break;
	case SINTP_TYPE_FILE:
		sd = sysfs_add_file(parent, name, mode,
				    &sintp_sysfs_fops, sd_ent);
		break;
	case SINTP_TYPE_BIN:
		sd = sysfs_add_bin(parent, name, mode, PAGE_SIZE,
				   &sintp_sysfs_bops, sd_ent);
		break;
	case SINTP_TYPE_LINK:
		sd = sysfs_add_link(parent, name, mode, target);
		break;
	default:
		BUG();
	}

	if (IS_ERR(sd))
		*pcmd_result = PTR_ERR(sd);

	if (!IS_ERR(sd) || (!sd_ent->sd || IS_ERR(sd_ent->sd)))
		sd_ent->sd = sd;

	return 0;
}

static int sintp_cmd_add_dir(char **toks, int nr_toks, int *pcmd_result)
{
	return sintp_cmd_add(toks, nr_toks, SINTP_TYPE_DIR, pcmd_result);
}

static int sintp_cmd_add_file(char **toks, int nr_toks, int *pcmd_result)
{
	return sintp_cmd_add(toks, nr_toks, SINTP_TYPE_FILE, pcmd_result);
}

static int sintp_cmd_add_bin(char **toks, int nr_toks, int *pcmd_result)
{
	return sintp_cmd_add(toks, nr_toks, SINTP_TYPE_BIN, pcmd_result);
}

static int sintp_cmd_add_link(char **toks, int nr_toks, int *pcmd_result)
{
	return sintp_cmd_add(toks, nr_toks, SINTP_TYPE_LINK, pcmd_result);
}

static int sintp_cmd_check_batch_error(char **toks, int nr_toks,
				       int *pcmd_result)
{
	struct sysfs_dirent *sd;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sd = sintp_find_sd(toks[0]);
	if (!sd)
		return -ENOENT;

	*pcmd_result = sysfs_check_batch_error(sd);
	return 0;
}

static int sintp_cmd_unplug(char **toks, int nr_toks, int *pcmd_result)
{
	struct sysfs_dirent *sd;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sd = sintp_find_sd(toks[0]);
	if (!sd)
		return -ENOENT;

	sysfs_unplug(sd);
	return 0;
}

static int sintp_cmd_find_child(char **toks, int nr_toks, int *pcmd_result)
{
	const char *parent_path, *name;
	struct sysfs_dirent *parent, *sd;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sintp_split_basename(toks[0], &parent_path, &name);

	parent = sintp_find_sd(parent_path);
	if (!parent)
		return -ENOENT;

	sd = sysfs_find_child(parent, name);
	if (IS_ERR(sd))
		*pcmd_result = PTR_ERR(sd);
	else if (sd)
		*pcmd_result = 1;

	return 0;
}

static int sintp_cmd_remove(char **toks, int nr_toks, int *pcmd_result)
{
	struct sysfs_dirent *sd;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sd = sintp_find_sd(toks[0]);
	if (!sd)
		return -ENOENT;

	sysfs_remove(sd);

	sintp_modify_sd_list(toks[0], NULL);

	return 0;
}

static int sintp_cmd_rename(char **toks, int nr_toks, int *pcmd_result)
{
	char *target_path;
	const char *target_parent_path, *target_name;
	struct sysfs_dirent *source, *target_parent;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	rc = sintp_sanitize_path(toks[1]);
	if (rc)
		return rc;

	target_path = kstrdup(toks[1], GFP_KERNEL);
	if (!target_path)
		return -ENOMEM;

	sintp_split_basename(target_path, &target_parent_path, &target_name);

	source = sintp_find_sd(toks[0]);
	target_parent = sintp_find_sd(target_parent_path);

	if (!source || !target_parent) {
		kfree(target_path);
		return -ENOENT;
	}

	*pcmd_result = sysfs_rename(source, target_parent, target_name);
	if (*pcmd_result == 0)
		sintp_modify_sd_list(toks[0], toks[1]);

	kfree(target_path);
	return 0;
}

static int sintp_cmd_notify(char **toks, int nr_toks, int *pcmd_result)
{
	struct sysfs_dirent *sd;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sd = sintp_find_sd(toks[0]);
	if (!sd)
		return -ENOENT;

	sysfs_notify_file(sd);

	return 0;
}

static int sintp_cmd_chmod(char **toks, int nr_toks, int *pcmd_result)
{
	struct sysfs_dirent *sd;
	mode_t mode;
	char *end;
	int rc;

	rc = sintp_sanitize_path(toks[0]);
	if (rc)
		return rc;

	sd = sintp_find_sd(toks[0]);
	if (!sd)
		return -ENOENT;

	mode = simple_strtoul(toks[1], &end, 0);
	if (end == toks[1] || *end != '\0')
		return -EINVAL;

	*pcmd_result = sysfs_chmod(sd, mode);

	return 0;	
}

/*
 * sintp_write() and sintp_read()
 */
static const struct sintp_cmd_ent sintp_cmdtbl[] = {
	{ "add-dir",		2, 3, sintp_cmd_add_dir },
	{ "add-file",		2, 3, sintp_cmd_add_file },
	{ "add-bin",		2, 3, sintp_cmd_add_bin },
	{ "add-link",		2, 3, sintp_cmd_add_link },
	{ "check-batch-error",	1, 1, sintp_cmd_check_batch_error },
	{ "unplug",		1, 1, sintp_cmd_unplug },
	{ "find-child",		1, 1, sintp_cmd_find_child },
	{ "remove",		1, 1, sintp_cmd_remove },
	{ "rename",		2, 2, sintp_cmd_rename },
	{ "notify",		1, 1, sintp_cmd_notify },
	{ "chmod",		2, 2, sintp_cmd_chmod },
	{ NULL },
};

static int sintp_tokener(char *str, char **tokens)
{
	char *p = str;
	int idx = 0;

	while (isspace(*p))
		p++;

	while (*p) {
		if (idx >= MAX_TOKENS)
			return -EINVAL;

		tokens[idx++] = p;
		while (*p && !isspace(*p))
			p++;
		if (*p)
			*p++ = '\0';
		while (isspace(*p))
			p++;
	}

	tokens[idx] = NULL;
	return idx;
}

static ssize_t sintp_write(struct file *file, const char __user *ubuf,
			   size_t count, loff_t *ppos)
{
	struct sintp_open_file *openf = file->private_data;
	char *buf = openf->buf;
	char *toks[MAX_TOKENS + 1];
	const struct sintp_cmd_ent *cent;
	int nr_toks, rc;

	if (count >= BUF_SIZE)
		return -ENOSPC;

	if (copy_from_user(buf, ubuf, count))
		return -EFAULT;
	buf[count] = '\0';

	rc = sintp_tokener(buf, toks);
	if (rc < 0)
		return rc;
	if (rc == 0)
		return -EINVAL;
	nr_toks = rc;

	for (cent = sintp_cmdtbl; cent->cmdstr; cent++)
		if (!strcmp(cent->cmdstr, toks[0]))
			break;

	if (!cent->cmdstr)
		return -EINVAL;

	if (nr_toks - 1 < cent->min_args || nr_toks - 1 > cent->max_args)
		return -EINVAL;

	openf->cmd_result = 0;
	rc = cent->cmdfn(toks + 1, nr_toks - 1, &openf->cmd_result);
	if (rc)
		return rc;

	return count;
}

static ssize_t sintp_read(struct file *file, char __user *ubuf,
			  size_t count, loff_t *ppos)
{
	struct sintp_open_file *openf = file->private_data;
	char *buf = openf->buf;
	size_t sz;

	sz = snprintf(buf, BUF_SIZE - 1, "%d\n", openf->cmd_result);

	return simple_read_from_buffer(ubuf, count, ppos, buf, sz);
}

static int sintp_open(struct inode *inode, struct file *file)
{
	struct sintp_open_file *openf;

	openf = kzalloc(sizeof(*openf), GFP_KERNEL);
	if (!openf)
		return -ENOMEM;

	openf->buf = kmalloc(BUF_SIZE, GFP_KERNEL);
	if (!openf->buf) {
		kfree(openf->buf);
		return -ENOMEM;
	}

	file->private_data = openf;
	return 0;
}

static int sintp_close(struct inode *inode, struct file *file)
{
	struct sintp_open_file *openf = file->private_data;

	kfree(openf->buf);
	kfree(openf);

	return 0;
}

static const struct file_operations sintp_fops = {
	.write		= sintp_write,
	.read		= sintp_read,
	.open		= sintp_open,
	.release	= sintp_close,
};

extern int sysfs_debug;

static int __init sintp_init(void)
{
	struct sintp_sd_ent *sd_ent;
	int rc;

	sd_ent = sintp_get_sd_ent("");
	if (!sd_ent) {
		rc = -ENOMEM;
		goto err;
	}

	sintp_root = sysfs_add_dir(sysfs_root, SINTP_ROOT_PATH, 0777, sd_ent);
	if (IS_ERR(sintp_root)) {
		rc = PTR_ERR(sintp_root);
		goto err;
	}

	sd_ent->sd = sintp_root;

	sintp_dentry = debugfs_create_file("sysfs-interpreter", 0200, NULL,
					   NULL, &sintp_fops);
	if (IS_ERR(sintp_dentry)) {
		rc = PTR_ERR(sintp_dentry);
		goto err;
	}
	if (!sintp_dentry) {
		rc = -ENOMEM;
		goto err;
	}

	sysfs_debug = 1;
	return 0;
 err:
	if (sintp_root)
		sysfs_remove(sintp_root);
	kfree(sd_ent);
	return rc;
}

static void __exit sintp_exit(void)
{
	struct sintp_sd_ent *sd_ent, *tsd_ent;

	debugfs_remove(sintp_dentry);
	sysfs_remove(sintp_root);

	list_for_each_entry_safe(sd_ent, tsd_ent, &sintp_sd_list, list) {
		kfree(sd_ent->data);
		kfree(sd_ent);
	}
	sysfs_debug = 0;
}

module_init(sintp_init);
module_exit(sintp_exit);
MODULE_LICENSE("GPL");
