#!/bin/sh
#

# Command line arguments.
SECS=$1
CLIENTS=$2
PORTS=$3
PERIOD=$4
RUNS=$5
PLYBK_PORTS=$6

# Parameter defaults.
[ -z "${SECS}"    ] && SECS=300
[ -z "${CLIENTS}" ] && CLIENTS=14
[ -z "${PORTS}"   ] && PORTS=4
[ -z "${PERIOD}"  ] && PERIOD=64
[ -z "${RUNS}"    ] && RUNS=1

[ -z "${PLYBK_PORTS}"] && PLYBK_PORTS=32

# Local tools must be on same directory...
BASEDIR=`dirname $0`

# Make sure our local tools are in place.
(cd ${BASEDIR}; make all > /dev/null) || exit 1

# nmeter configuration.
NMETER="${BASEDIR}/jack_test4_nmeter"
NMETER_CMDLINE="${NMETER} t c i x b m"

# jackd configuration.
JACKD="jackd"
JACKD_DRIVER="alsa"
JACKD_DEVICE="hw:0"
JACKD_PRIO=60
JACKD_RATE=44100
JACKD_PORTS=$((${CLIENTS} * ${PORTS} * 2 + ${PLYBK_PORTS}))
JACKD_CMDLINE="${JACKD} -Rv -P${JACKD_PRIO} -p${JACKD_PORTS} -d${JACKD_DRIVER} -d${JACKD_DEVICE} -r${JACKD_RATE} -p${PERIOD} -n2 -P"

# client test configuration.
CLIENT="${BASEDIR}/jack_test4_client"

# process/thread list configuration.
PIDOF="/sbin/pidof"
PLIST="ps -o pid,tid,class,rtprio,ni,pri,pcpu,stat,comm"

# Log filename.
LOG="jack_test4-`uname -r`-`date +'%Y%m%d%H%M'`.log"

# Command executive and logger.
exec_log ()
{
	CMD="$*"
	echo "-----------------------"	>> ${LOG} 2>&1
	echo "# ${CMD}"					>> ${LOG} 2>&1
	${CMD}							>> ${LOG} 2>&1
	echo							>> ${LOG} 2>&1
}

# IRQ thread handler status logger.
ilist_log ()
{
	echo "- - - - - - - - - - - -"		>> ${LOG} 2>&1
	${PLIST} --sort -rtprio -e \
	| egrep '(^[ ]+PID|IRQ)'			>> ${LOG} 2>&1
	echo								>> ${LOG} 2>&1
}

# Top CPU% process/thread status logger.
plist_log ()
{
	echo "- - - - - - - - - - - -"		>> ${LOG} 2>&1
	${PLIST} --sort -pcpu -me \
	| head -$((3 * ${CLIENTS} + 20))	>> ${LOG} 2>&1
	echo								>> ${LOG} 2>&1
}


#
# Log headings -- show some relevant info...
#
echo "*** Started `date` ***" >> ${LOG} 2>&1

echo >> ${LOG} 2>&1
echo "Seconds to run        (SECS) = ${SECS}"			>> ${LOG} 2>&1
echo "Number of clients  (CLIENTS) = ${CLIENTS}"		>> ${LOG} 2>&1
echo "Ports per client     (PORTS) = ${PORTS}"			>> ${LOG} 2>&1
echo "Frames per buffer   (PERIOD) = ${PERIOD}"			>> ${LOG} 2>&1
echo "Number of runs        (RUNS) = ${RUNS}"			>> ${LOG} 2>&1
echo "Playback ports (PLYBK_PORTS) = ${PLYBK_PORTS}"	>> ${LOG} 2>&1
echo >> ${LOG} 2>&1

exec_log "uname -a"

exec_log "cat /proc/asound/version"
exec_log "cat /proc/asound/cards"

#exec_log "grep . /proc/sys/kernel/*_preemption"
#exec_log "grep . /proc/irq/*/*/threaded"
#exec_log "grep . /sys/block/hd*/queue/max_sectors_kb"

exec_log "cat /proc/interrupts"

# This is just about to be sure...
ilist_log

#
# Launch nmeter in the background...
#
echo -n "Launching `basename ${NMETER}`..."
(${NMETER_CMDLINE} >> ${LOG} 2>&1) &
sleep 2
echo "done."
sleep 1

#
# Launch the client instance(s) run(s)...
#
SLEEP=0
for RUN in `seq 1 ${RUNS}`; do
	# Launch one test client suite...
	SLEEP=$((${SLEEP} + 6))
	for NUM in `seq 1 ${CLIENTS}`; do
		CLIENT_CMDLINE="${CLIENT} ${SECS} ${PORTS} ${NUM} ${CLIENTS}"
		SLEEP=$((${SLEEP} + 3))
		(
			sleep ${SLEEP}
			printf "[%d/%d] `basename ${CLIENT}` (%2d/%d) started...\n" \
				${RUN} ${RUNS} ${NUM} ${CLIENTS}
			exec_log ${CLIENT_CMDLINE}
			printf "[%d/%d] `basename ${CLIENT}` (%2d/%d) stopped.\n" \
				${RUN} ${RUNS} ${NUM} ${CLIENTS}
		) &
	done
	# Let there be some periodic evidence
	# of current process/thread status...
	SLEEP=$((${SLEEP} + 3))
	(
		sleep ${SLEEP}
		TSTEP=$((3 * ${CLIENTS} + 1))
		TIMER=$((${SECS} - ${TSTEP}))
		echo -n "Running"
		while [ ${TIMER} -gt ${TSTEP} ]; do
			plist_log
			sleep ${TSTEP}
			echo -n "."
			TIMER=$((${TIMER} - ${TSTEP}))
		done
		echo "done."
	) &
	# Let it be extended but of limited time...
	SLEEP=$((${SLEEP} + ${SECS}))
done

# There will be the time to kill jackd...
(
	sleep ${SLEEP}
	echo -n "Stopping `basename ${JACKD}`..."
	killall `basename ${JACKD}` > /dev/null 2>&1 && echo "OK." || echo "FAILED."
) &

#
# Launch jackd and wait for it...
#
echo "Starting `basename ${JACKD}`..."
exec_log ${JACKD_CMDLINE}
sleep 1

echo -n "Killing `basename ${CLIENT}`..."
killall `basename ${CLIENT}` && echo "OK." || echo "FAILED."
sleep 1

echo -n "Stopping `basename ${NMETER}`..."
killall `basename ${NMETER}` > /dev/null 2>&1 && echo "OK." || echo "FAILED."

echo "*** Terminated `date` ***" >> ${LOG} 2>&1

sync
sleep 1
sync

# Summary log analysis...
cat ${LOG} | awk -f ${BASEDIR}/jack_test4_summary.awk | tee -a ${LOG}

# Finally, plot log output...
${BASEDIR}/jack_test4_plot.sh ${LOG}
