
#include <linux/module.h>
#include <linux/hrtimer.h>


/* interval is one millisecond */
#define INTERVAL_NS 1000000


static struct hrtimer hrtimer_demo_timer;
static int hrtimer_demo_counter;
static u64 hrtimer_demo_maxlatency;



/* ---- timer function ---- */

static enum hrtimer_restart hrtimer_demo_function(struct hrtimer *timer) {
  struct timespec now;
  ktime_t latency_kt;
  u64 latency_ns;

  /* check latency */
  ktime_get_ts(&now);
  latency_kt = ktime_sub(timespec_to_ktime(now), timer->expires);
  latency_ns = ktime_to_ns(latency_kt);
  if(latency_ns > hrtimer_demo_maxlatency) hrtimer_demo_maxlatency = latency_ns;

  /* do something useful, here */
  hrtimer_demo_counter++;

  /* setup next expire date */
  timer->expires = ktime_add_ns(timer->expires, INTERVAL_NS);
  return HRTIMER_RESTART;
}



/* ---- start/stop ---- */


static void hrtimer_demo_start(void) {
  struct timespec now;
  ktime_t expires;

  /* setup the demo hrtimer */
  hrtimer_init(&hrtimer_demo_timer, CLOCK_MONOTONIC, HRTIMER_MODE_ABS);
  hrtimer_demo_timer.function = hrtimer_demo_function;
  hrtimer_demo_timer.cb_mode  = HRTIMER_CB_IRQSAFE;

  /* start timer in 100 milliseconds */
  ktime_get_ts(&now);
  expires = ktime_add_ns(timespec_to_ktime(now), 100000000);
  hrtimer_start(&hrtimer_demo_timer, expires, HRTIMER_MODE_ABS);
};


static void hrtimer_demo_stop(void) {
  hrtimer_cancel(&hrtimer_demo_timer);
};



/* ---- init/exit ---- */

static int hrtimer_demo_init(void) {
  printk("\nhrtimer_demo_init\n");
  hrtimer_demo_start();
  return 0;
}


static void hrtimer_demo_exit(void) {
  hrtimer_demo_stop();
  printk("hrtimer_demo_exit\n");
  printk("hrtimer_demo_counter=%i\n", hrtimer_demo_counter);
  printk("hrtimer_demo_maxlatency=%Ld\n", hrtimer_demo_maxlatency);
}


module_init(hrtimer_demo_init);
module_exit(hrtimer_demo_exit);

MODULE_LICENSE("GPL");
