/*
 *  Simple scattered buffer management for Linux SCSI.
 *
 *	The purpose of this simple buffer is to allow large buffering to 
 *	really work under Linux SCSI. Candidates for use of it are:
 *	- sg driver
 *	- mt driver
 *	- SCSI_IOCTL_SEND_COMMAND
 *
 *  Copyright (C) 1999, Gerard Roudier <groudier@club-internet.fr>
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License version 2 as published 
 *  by the Free Software Foundation.
 */

#include <scbuf.h>

#ifdef DEBUG
#define PDEBUG(where, args)					\
do {								\
	printk(KERN_DEBUG "%s/%d: ", where, __LINE__);		\
	printk args;						\
} while (0);
#else
#define PDEBUG(where, args)
#endif

/*
 *	Copy from/to user space functions
 *
 *	Since Linux kernel copy services aren't guaranteed to be real 
 *	function, the below trivial stuff is needed.
 */
int scbuf_copy_in(void *kptr, void *uptr, int len)
{
	return copy_from_user(kptr, uptr, len);
}

int scbuf_copy_out(void *uptr, void *kptr, int len)
{
	return copy_to_user(uptr, kptr, len);
}

/*
 *	Generic copy from/to user space to/from a scattered buffer
 *
 *	copy	: copy function (one of the above ones)
 *	bp	: pointer to the scattered buffer
 *	offs	: offset in the scattered buffer the copy started from
 *	uptr	: pointer to user buffer in user space
 *	usize	: size to copy from/to user buffer
 *
 *	returns the number of bytes not copied (generally zero)
 */
#define _here_ "scbuf_copy"
int scbuf_copy(int copy(void *, void *, int),
		scbuf_t *bp, int offs, char *uptr, int usize)
{
	int csize = PAGE_SIZE<<bp->order; /* Size of each scatter entry */
	int size, rest;
	int i;

	/*
	 *	If offset isn't zero, compute the number of the entry 
	 *	and the offset within that entry we have to start from.
	 */
	i = offs ? offs/csize : 0;
	if (i) {
		if (i >= bp->sgnbr)
			goto out;
		offs = offs % csize;
	}
	PDEBUG(_here_, ("usize=%d i=%d offs=%d\n", usize, i, offs));

	/*
	 *	If offset isn't zero, copy first this partial chunk.
	 */
	if (offs) {
		size = offs + usize > csize ? csize - offs : usize;
		rest = copy(uptr, bp->sglist[i].address, size);
		size -= rest;
		uptr += size; usize -= size;
		if (rest)
			goto out;
		++i;
	}

	/*
	 *	Then copy the remaining data.
	 */
	for (i = 0; usize && i < bp->sgnbr; i++) {
		size = csize < usize ? csize : usize;
		rest = copy(uptr, bp->sglist[i].address, size);
		size -= rest;
		uptr += size; usize -= size;
		if (rest)
			goto out;
	}
out:
	PDEBUG(_here_, ("usize=%d i=%d\n", usize, i));
	return usize;
}
#undef _here_

/*
 *	Resize a scattered buffer
 *
 *	osflags	: flags to be passed to the memory allocator
 *	bp	: pointer to the scattered buffer to resize
 *	size	: new size
 *
 *	Returns the new actual size of the buffer (generally >= size)
 */
#define _here_ "scbuf_resize"
int scbuf_resize(int osflags, scbuf_t *bp, int size)
{
	int csize = PAGE_SIZE<<bp->order;	/* Size of each scatter entry */
	int sgnbr = (size + csize - 1) / csize;	/* Number of entries needed */

	PDEBUG(_here_, ("size=%d sgnbr=%d csize=%d bp->sgnbr=%d bp->sgmax=%d\n",
			size, sgnbr, csize, bp->sgnbr, bp->sgmax));

	/*
	 *	Scale the number of entries to our maximum.
	 */
	if (sgnbr > bp->sgmax)
		sgnbr = bp->sgmax;

	/*
	 *	Free all entries we donnot need anymore if any.
	 */
	while (sgnbr < bp->sgnbr)
		free_pages(bp->sglist[--bp->sgnbr].address, bp->order);

	/*
	 *	Allocates new entries as needed.
	 */
	while (sgnbr > bp->sgnbr) {
		char *p = __get_free_pages(osflags, bp->order);
		if (p) bp->sglist[bp->sgnbr++].address = p;
	}

	/*
	 *	Returns the new actual size of the buffer.
	 */
	PDEBUG(_here_, ("csize=%d bp->sgnbr=%d csize*bp->sgnbr=%d\n",
			csize, bp->sgnbr, csize*bp->sgnbr));
	return csize * bp->sgnbr;
}
#undef _here_

/*
 *	Free a scattered buffer
 *
 *	bp	: pointer to the scattered buffer
 */
#define _here_ "scbuf_free"
void scbuf_free(scbuf_t *bp)
{
	PDEBUG(_here_, ("bp->sgmax=%d bp->sgnbr=%d\n", bp->sgmax, bp->sgnbr));
	/*
	 *	Resize the buffer size to zero bytes.
	 *	This will free all scatter entries.
	 *	Free the scattered buffer structure.
	 */
	scbuf_resize(0, bp, 0);
	kfree(bp);
}
#undef _here_

/*
 *	Allocate a scattered buffer
 *
 *	osflags	: flags to pass to the memory allocator
 *	maxsg	: maximum number of sg entries
 *		  should match the value supported by the low-level driver
 *	maxsize : maximum size in bytes the buffer will ever be sized to
 *	size	: initial size in bytes of the buffer
 */
#define _here_ "scbuf_alloc"
scbuf_t *scbuf_alloc(int osflags, int maxsg, int maxsize, int size)
{
	scbuf_t *bp;
	int order;

	PDEBUG(_here_, ("maxsg=%d maxsize=%d size=%d\n", maxsg, maxsize, size));
	/*
	 *	Compute the minimal order of page allocation that fits 
	 *	the maxsize and maxsg requirements and shorten maxsg 
	 *	if possible.
	 */
	for (order = 0; (PAGE_SIZE<<order)*maxsg < maxsize ; ++order);
	maxsg = (maxsize + (PAGE_SIZE<<order) - 1) / (PAGE_SIZE<<order);

	/*
	 *	Allocate a buffer structure large enough to contains 
	 *	maxsg scatter entries and initialyze it.
	 */
	bp = kmalloc(osflags, offsetof(struct scbuf, sglist[maxsg+1]));
	if (!bp)
		return 0;
	bzero(bp, sizeof(*bp));
	bp->order = order;
	bp->sgmax = maxsg;

	PDEBUG(_here_, ("bp->order=%d bp->sgmax=%d\n", bp->order, bp->sgmax));

	/*
	 *	Size the buffer as asked by user.
	 */
	if (scbuf_resize(osflags, bp, size) < size) {
		scbuf_free(bp);
		bp = 0;
	}

	return bp;
}
#undef _here_

/*
 *	Prepare a scattered buffer for IO
 *
 *	sglist	: pointer to the returned scatter list head
 *	bp	: pointer to the scattered buffer
 *	size	: size in bytes for the IO
 *
 *	Returns the number of scatter entries to IO or a negative 
 *	number if the buffer is too small.
 */
#define _here_ "scbuf_prepare_for_io"
int scbuf_prepare_for_io(scatt_t **sglist, scbuf_t *bp, int size)
{
	int csize = PAGE_SIZE<<bp->order; /* Size of each scatter entry */
	int i;

	PDEBUG(_here_, ("csize=%d size=%d\n", csize, size));

	for (i = 0 ; i < bp->sgnbr && size; i++) {
		bp->sglist[i].length = size < csize ? size : csize;
		size -= bp->sglist[i].length;
	}

	PDEBUG(_here_, ("size=%d i=%d\n", size, i));
	return size ? -1 : i;
}
#undef _here_
