/*
 *	ssfdc_mgr.c
 *
 *	SSFDC Abstract Management.
 *
 *  parts (c) 2001 Beier & Dauskardt IT (www.beier-dauskardt.com)
 *
 *
 *	many thanks to all sddr09.c contributors.
 *
 *
 * The usual legal:
 *
 * This program is distributed in e hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

/*

a few words about SSFDC Card Layout.

A SSFDC Card is organized into Blocks and Pages (with Redundancy):

----------+-----------+----------------+
 Block 0  |  Page 0   |  Redundancy 0  |
          +-----------+----------------+
          |  Page 1   |  Redundancy 1  |
          +-----------+----------------+
          |      ..   |      ..        |
          +-----------+----------------+
          |  Page n   |  Redundancy n  |
----------+-----------+----------------+
 Block 1  |  Page 0   |  Redundancy 0  |
          +-----------+----------------+
          |  Page 1   |  Redundancy 1  |
          +-----------+----------------+
          |      ..   |      ..        |



Known Page/Blocksizes:

 Size   PageSize Pages/Block BlkSize  PhysBlocks Log-Blocks    3,3V ID's  5V       ROM
 -----+---------+-----------+-------+-----------+-----------+-----------+-------+-------
  1MB |  256+8  |     16    |  4096 |    256    |    250    |  E6,E8,EC | 6E    |  
  2MB |  256+8  |     16    |  4096 |    512    |    500    |  EA       | 64    | 5D (512 Bytes/Block !)
     -+---------+-----------+-------+-----------+-----------+-----------+-------+-------
  4MB |  512+8  |     16    |  8192 |    512    |    500    |  E3,E5    | 6B,E5 | D5
  8MB |  512+8  |     16    |  8192 |   1024    |   1000    |  E6       |       | D6
     -+---------+-----------+-------+-----------+-----------+-----------+-------+-------
 16MB |  512+8  |     32    | 16384 |   1024    |   1000    |  73       |       |
 32MB |  512+8  |     32    | 16384 |   2048    | 2*1000 (Z)|  75       
     -+---------+-----------+-------+-----------+-----------+-------
 64MB |  512+8  |     32    | 16384 |   4096    | 4*1000 (Z)|  76   
128MB |  512+8  |     32    | 16384 |   8192    | 8*1000 (Z)|  79   
     -+---------+-----------+-------+-----------+-----------+-------
256MB |    ?    |      ?    |   ?   |     ?     |     ?     |  ?



Block 0 is reserved for so-called CIS/IDI data.
 --> Block 0 can't be a valid LBA->PBA mapping !
Incase block 0 is bad, block 1 is used.


Note (Z):
	The LBA<-->PBA Mapping information in the redundancy Area of each Page/Block
	only has 10 Bits, which gives us a possible rangs on 0..1023. Valid Range actually
	is only 0..999.
	Card with more than 999 Logical block are spereated into Zones of 1000 blocks.
	We have to watch out with the mapping here.


*/
#include "usb.h"
#include "debug.h"

#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/malloc.h>

#include "ssfdc_mgr.h"



// Allocate Card-info 
//
struct ssfdc_card_info *ssfdc_create(void *forDevice)
{
	struct ssfdc_card_info	*ssfdci;

	ssfdci = (struct ssfdc_card_info *)kmalloc(sizeof(struct ssfdc_card_info), GFP_KERNEL);
	if(!ssfdci)
		return NULL;

	memset(ssfdci,0,sizeof(struct ssfdc_card_info));

	ssfdci->access_device = forDevice;
	return ssfdci;
}

void ssfdc_delete(pssfdc_card_info ssfdci)
{
  	if(!ssfdci)
		return;

	ssfdc_delete_mapping( ssfdci );

	kfree( ssfdci );
}

void ssfdc_delete_mapping(pssfdc_card_info ssfdci)
{
  	if(!ssfdci)
		return;

	if( ssfdci->lba_to_pba ){
	    kfree( ssfdci->lba_to_pba );
	    ssfdci->lba_to_pba = 0;
	}
	    
	if( ssfdci->pba_status ){
	    kfree( ssfdci->pba_status );
	    ssfdci->pba_status = 0;
	}
}


// A few known geomertrys..
//
const struct ssfdc_geometry ssfdcgeo_1MB_Flash={
	attributes:	SSFDCA_PAGE256|SSFDCA_FLASH, 	
	capacity:	0x0100000,
	pagesize:	256,	pageshift:	8,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	256,	logblocks:	250,	zones:		1
};

const struct ssfdc_geometry ssfdcgeo_2MB_Flash={
	attributes:	SSFDCA_PAGE256|SSFDCA_FLASH, 	
	capacity:	0x0200000,
	pagesize:	256,	pageshift:	8,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	512,	logblocks:	500,	zones:		1
};
const struct ssfdc_geometry ssfdcgeo_2MB_ROM={
	attributes:	SSFDCA_PAGE512|SSFDCA_ROM, 
	capacity:	0x0200000,
	pagesize:	512,	pageshift:	9,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	256,	logblocks:	250,	zones:		1
};

const struct ssfdc_geometry ssfdcgeo_4MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH, 
	capacity:	0x0400000,
	pagesize:	512,	pageshift:	9,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	512,	logblocks:	500,	zones:		1
};
const struct ssfdc_geometry ssfdcgeo_4MB_ROM={
	attributes:	SSFDCA_PAGE512|SSFDCA_ROM, 
	capacity:	0x0400000,
	pagesize:	512,	pageshift:	9,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	512,	logblocks:	500,	zones:		1
};

const struct ssfdc_geometry ssfdcgeo_8MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH, 
	capacity:	0x0800000,
	pagesize:	512,	pageshift:	9,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	1024,	logblocks:	1000,	zones:		1
};
const struct ssfdc_geometry ssfdcgeo_8MB_ROM={
	attributes:	SSFDCA_PAGE512|SSFDCA_ROM, 
	capacity:	0x0800000,
	pagesize:	512,	pageshift:	9,
	blocksize:	16,	blockshift:	4,	blockmask:	0xF,
	physblocks:	1024,	logblocks:	1000,	zones:		1
};

const struct ssfdc_geometry ssfdcgeo_16MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH, 
	capacity:	0x1000000,
	pagesize:	512,	pageshift:	9,
	blocksize:	32,	blockshift:	5,	blockmask:	0x1F,
	physblocks:	1024,	logblocks:	1000,	zones:		1
};
const struct ssfdc_geometry ssfdcgeo_32MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH|SSFDCA_MULTIZONE, 
	capacity:	0x2000000,
	pagesize:	512,	pageshift:	9,
	blocksize:	32,	blockshift:	5,	blockmask:	0x1F,
	physblocks:	2048,	logblocks:	2000,	zones:		2
};
const struct ssfdc_geometry ssfdcgeo_64MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH|SSFDCA_MULTIZONE, 
	capacity:	0x4000000,
	pagesize:	512,	pageshift:	9,
	blocksize:	32,	blockshift:	5,	blockmask:	0x1F,
	physblocks:	4096,	logblocks:	4000,	zones:		4
};
const struct ssfdc_geometry ssfdcgeo_128MB_Flash={
	attributes:	SSFDCA_PAGE512|SSFDCA_FLASH|SSFDCA_MULTIZONE, 
	capacity:	0x8000000,
	pagesize:	512,	pageshift:	9,
	blocksize:	32,	blockshift:	5,	blockmask:	0x1F,
	physblocks:	8192,	logblocks:	8000,	zones:		8
};


// Set Card geometry from 
//
int ssfdc_determine_card_params(struct ssfdc_card_info *ssfdci,unsigned char VendorID,unsigned char DeviceID)
{
  	if(!ssfdci)
		return SSFDC_NOT_INITIALISED;

	// Force new mapping to be created, incase we missed the change...
	//
	if( ssfdci->DeviceID != DeviceID )
		ssfdc_delete_mapping( ssfdci );

	ssfdci->VendorID	= VendorID;
	ssfdci->DeviceID	= DeviceID;

	switch( DeviceID ){
	case 0x6E:
	case 0xE8:
	case 0xEC:	ssfdci->geo = ssfdcgeo_1MB_Flash;	break;

	case 0xEA:
	case 0x64:	ssfdci->geo = ssfdcgeo_2MB_Flash;	break;
	case 0x5D:	ssfdci->geo = ssfdcgeo_2MB_ROM;	break;

	case 0xE3:
	case 0xE5:
	case 0x6B:	ssfdci->geo = ssfdcgeo_4MB_Flash;	break;
	case 0xD5:	ssfdci->geo = ssfdcgeo_4MB_ROM;	break;
 
	case 0xE6:	ssfdci->geo = ssfdcgeo_8MB_Flash;	break;
	case 0xD6:	ssfdci->geo = ssfdcgeo_8MB_ROM;	break;

	case 0x73:	ssfdci->geo = ssfdcgeo_16MB_Flash;	break;

	case 0x75:	ssfdci->geo = ssfdcgeo_32MB_Flash;	break;

	case 0x76:	ssfdci->geo = ssfdcgeo_64MB_Flash;	break;

	case 0x79:	ssfdci->geo = ssfdcgeo_128MB_Flash;	break;

	default:
		// Undefined state.
		//
		US_DEBUGP("ssfdc_determine_card_params: card %02x/%02x unknown. Loosing geometry.\n",
			VendorID,
			DeviceID
		);

		memset(&ssfdci->geo,0,sizeof(ssfdci->geo));
		return SSFDC_CARD_UNKNOWN;
	}


	US_DEBUGP("ssfdc_determine_card_params: got geometry definition for card %02x/%02x:\n",
		VendorID,
		DeviceID
	);
	US_DEBUGP("attributes: %04x\n",ssfdci->geo.attributes);
	US_DEBUGP("capacity:   %08lx\n",ssfdci->geo.capacity);
	US_DEBUGP("pagesize:   %3d  pageshift:  %d\n",ssfdci->geo.pagesize,ssfdci->geo.pageshift);
	US_DEBUGP("blocksize:  %2d   blockshift:  %d    blockmask: 0x%02x\n",ssfdci->geo.blocksize,ssfdci->geo.blockshift,ssfdci->geo.blockmask);
	US_DEBUGP("physblocks: %4d logblocks:   %4d     zones: %d\n",ssfdci->geo.physblocks,ssfdci->geo.logblocks,ssfdci->geo.zones);

	return SSFDC_OK;
}



static const unsigned char fast_bit_count[16] = {
	0, 1, 1, 2, 
	1, 2, 2, 3,
	1, 2, 2, 3,
	2, 3, 3, 4
};

static const unsigned char fast_parity[16] = {
	0, 1, 1, 0, 
	1, 0, 0, 1,
	1, 0, 0, 1, 
	0, 1, 1, 0
};


// Count how manny bits are set somwhere.
//
int ssfdc_bits_set(int where) 
{
	int	bitcount= 0; 

	for(;where;where >>= 4)
		bitcount += fast_bit_count[where&0x0F];

	return bitcount;
}

// stolen: 
//
unsigned char ssfdc_16bitparity(int what)
{
	return	fast_parity[ (what      ) & 0x0F] ^
			fast_parity[ (what >>  4) & 0x0F] ^
			fast_parity[ (what >>  8) & 0x0F] ^
			fast_parity[ (what >> 12) & 0x0F];
}





int ssfdc_calc_block_status(unsigned char red_buffer[],int *block_lba)
{
	int		lba1,lba2;
	
	// Block valid ??
	//
	// red_buffer[0]..red_buffer[3] are reserved.
	//
	if( ssfdc_bits_set(red_buffer[5]) < 7 )
		return SSFDCPB_DAMAGED;			// this block is really bad.

	if( ssfdc_bits_set(red_buffer[4]) < 5 )
		return SSFDCPB_EMPTY;		// this block is empty.

	// o.k. this block is supposed to be o.k.
	//
	lba1 = ((int)red_buffer[ 6] << 8) | ((int)red_buffer[ 7]);
	lba2 = ((int)red_buffer[11] << 8) | ((int)red_buffer[12]);

	if( lba1 != lba2 ){

		// more than one bit-error ?
		//
		if( ssfdc_bits_set(lba1^lba2) > 1 )
			return SSFDCPB_BAD;			// too bad, can't figgure out which one would be ok.
	}

	if( lba1 == 0xFFFF || lba2 == 0xFFFF )
		return SSFDCPB_EMPTY;		// ??


	if( ((lba1 & 0xF800) == 0x1000) && !ssfdc_16bitparity(lba1) ){
		*block_lba = (lba1 >> 1) & 0x3FF;
		return SSFDCPB_VALID;
	}

	if( ((lba2 & 0xF800) == 0x1000) && !ssfdc_16bitparity(lba2) ){
		*block_lba = (lba2 >> 1) & 0x3FF;
		return SSFDCPB_VALID;
	}

	// Both lba's bad.
	//
	return SSFDCPB_BAD;	
}


// Load all mappings from the device.
//
// Assumes:
//	- Device-ID has been set.
// 
int ssfdc_build_mapping(struct ssfdc_card_info *ssfdci)
{
	unsigned char	red_buffer[16];
	int				red_per_page;
	int				error;
	int				block;
	int				blockstat;
	int				lba;

	int				blk_empty;
	int				blk_bad;
	int				blk_damaged;
	int				blk_ok;
	int				blk_double;


	US_DEBUGP("ssfdc_build_mapping: enter...\n");


	if(!ssfdci || !ssfdci->funcs_valid )
		return SSFDC_NOT_INITIALISED;

	if( !ssfdci->geo.capacity )
		return SSFDC_CARD_UNKNOWN;

	// get rid of old mapping...
	//
	ssfdc_delete_mapping( ssfdci );


	// With 256 byte/page devices, redundancy is spread over two pages.
	//
	if( ssfdci->geo.attributes & SSFDCA_PAGE256 ){
		red_per_page = 8;
	}else{
		red_per_page = 16;
	}


	// Watch out: different sizes !
	//
	ssfdci->lba_to_pba = (int *)kmalloc( ssfdci->geo.logblocks*sizeof(int), GFP_KERNEL );
	ssfdci->pba_status = (char *)kmalloc( ssfdci->geo.physblocks*sizeof(char), GFP_KERNEL );

	if( !ssfdci->lba_to_pba || !ssfdci->pba_status  ){
		ssfdc_delete_mapping( ssfdci );
		return SSFDC_NO_MEMORY;
	}

	memset( ssfdci->lba_to_pba,0,ssfdci->geo.logblocks*sizeof(int) );
	memset( ssfdci->pba_status ,0,ssfdci->geo.physblocks*sizeof(char) );



	blk_empty	= 0;
	blk_bad		= 0;
	blk_damaged	= 0;
	blk_ok		= 0;
	blk_double	= 0;

	US_DEBUGP("ssfdc_build_mapping: entering loop for %d redpage, %d phys-blocks...\n",red_per_page,ssfdci->geo.physblocks);

	error = SSFDC_OK;

	// We know take a look at all physical block (0..1023)
	//
	for( block = 0; block < ssfdci->geo.physblocks ; block++ ){

		if( red_per_page == 8 ){

    		// Read redundancy that's spread over two pages...
    		//
    		error = ssfdci->read_redundancy_func(ssfdci,block,0,&red_buffer[0],red_per_page);
    		if( error )
    			break;

    		error = ssfdci->read_redundancy_func(ssfdci,block,1,&red_buffer[8],red_per_page);
    		if( error )
    			break;

		}else{

    		// Read redundancy...
    		//
    		error = ssfdci->read_redundancy_func(ssfdci,block,0,red_buffer,red_per_page);
    		if( error )
    			break;
   		}

	
		// Block is either valid, empty, bad or damaged.
		// 
		blockstat = ssfdc_calc_block_status(red_buffer,&lba);
		if( blockstat == SSFDCPB_VALID ){

			/* Stolen:
			 *
			 * Every 1024 physical blocks ("zone"), the LBA numbers
             * go back to zero, but are within a higher
             * block of LBA's. Also, there is a maximum of
             * 1000 LBA's per zone. In other words, in PBA
             * 1024-2047 you will find LBA 0-999 which are
             * really LBA 1000-1999. Yes, this wastes 24
             * physical blocks per zone. Go figure.
             */
			if( ssfdci->geo.attributes & SSFDCA_MULTIZONE )
				lba += 1000*(block/0x400);

			if( lba >= ssfdci->geo.logblocks ){

				// Oops, left the logical rang of the card..
				//
				blockstat = SSFDCPB_BAD;

			}else{

				// Block already mapped ?
				//
    			if( ssfdci->lba_to_pba[ lba ] != 0 ){

					// TODO: be intelligent ?
					//
					blk_double++;
				}
    
    			// Also assing a lba mapping...
    			// 
    			ssfdci->lba_to_pba[ lba ] = block;
   			}

		}else{

			if( blockstat == SSFDCPB_BAD ){

				// TODO: go check redundancy info on some other page in this block...

			}
		}

		ssfdci->pba_status[ block ] = blockstat;

		switch( blockstat ){	
		case SSFDCPB_BAD:		
			US_DEBUGP("ssfdc_build_mapping: block %d marked BAD.\n",block);
			blk_bad++;		
			break;
		case SSFDCPB_DAMAGED:	
			US_DEBUGP("ssfdc_build_mapping: block %d marked DAMAGED.\n",block);
			blk_damaged++;	
			break;
		case SSFDCPB_EMPTY:		blk_empty++;	break;
		default:				blk_ok++;		break;	
		}


		if( block % 512 == 511 )
			US_DEBUGP("ssfdc_build_mapping:   ...done block %d\n",block);

	}

	US_DEBUGP("ssfdc_build_mapping: left loop on block %d. status %d.\n",block,error);

	US_DEBUGP("mappings stats: %d empty, %d ok\n",blk_empty,blk_ok);
	US_DEBUGP("mappings stats: %d bad, %d damaged   %d mapped-twice\n",blk_bad,blk_damaged,blk_double);

	// Should do.
	//
	return error;
}



// Call this, if you think there could be a new card in the drive.
//
int ssfdc_detect_new_media(struct ssfdc_card_info *ssfdci)
{
	int				error;

	error = ssfdc_read_id(ssfdci);
	if( error )
		return error;

	return ssfdc_build_mapping(ssfdci);
}


// How many bytes can we actually use ?
//
int ssfdc_get_formated_capacity(struct ssfdc_card_info *ssfdci)
{
    return	ssfdci->geo.pagesize *
			ssfdci->geo.blocksize *
			ssfdci->geo.logblocks;
}

 


//
//
int ssfdc_read_id(struct ssfdc_card_info *ssfdci)
{
	int				error;
	unsigned char	VendorID;
	unsigned char	DeviceID;

	if(!ssfdci || !ssfdci->funcs_valid )
		return SSFDC_NOT_INITIALISED;

	error = ssfdci->get_device_id_func(ssfdci,&VendorID,&DeviceID);
	if( error )
		return error;

	error = ssfdc_determine_card_params(ssfdci,VendorID,DeviceID);
	if( error )
		return error;

	return SSFDC_OK;
}


// Wan't some sectors ?
//
int ssfdc_read_data(struct ssfdc_card_info *ssfdci,unsigned long sector,unsigned short sectors,unsigned char *content,int use_sg)
{
	unsigned int		lba,pba;
	unsigned short		page,pages;
    unsigned char		*buffer = NULL;
    unsigned char		*ptr;
   	struct scatterlist	*sg = NULL;
    int					i;
    int					len;
    int					transferred;
	int					error;
	unsigned long		addr;

	if(!ssfdci || !ssfdci->funcs_valid || !ssfdci->lba_to_pba || !ssfdci->pba_status  )
		return SSFDC_NOT_INITIALISED;

	addr = sector*512;
	US_DEBUGP("ssfdc_read_data: request %d sectors from %ld (%ld)\n",sectors,sector,addr);

	// What do we really have to do ?
	//
	if( ssfdci->geo.attributes & SSFDCA_PAGE256 )
		sectors *= 2;

    len = sectors*512;

    // If we're using scatter-gather, we have to create a new
    // buffer to read all of the data in first, since a
    // scatter-gather buffer could in theory start in the middle
    // of a page, which would be bad. A developer who wants a
    // challenge might want to write a limited-buffer
    // version of this code.
	//
    if( use_sg ){
		sg = (struct scatterlist *)content;
        buffer = kmalloc(len, GFP_KERNEL);
		if( buffer == NULL )
			return SSFDC_NO_MEMORY;

        ptr = buffer;
	} else {
	    ptr = content;
   	}


    // Figure out the initial LBA and page
	//
	lba  = addr >> (ssfdci->geo.pageshift + ssfdci->geo.blockshift);
	page = (addr >> ssfdci->geo.pageshift) & ssfdci->geo.blockmask;

	error = SSFDC_OK;

	for(;sectors > 0;){

		pba = ssfdci->lba_to_pba[lba];
		if( !pba ){
			error = SSFDC_FAILED_NO_MAPPING;
			US_DEBUGP("ssfdc_read_data: leave loop: no mapping for LBA %d.\n",lba);
			break;
		}

		// Read as many sectors as possible in this block
		//
		pages = ssfdci->geo.blocksize - page;
		if( pages > sectors )
			pages = sectors;

		US_DEBUGP("ssfdc_read_data: issuing device-read: LBA %d -> PBA %d Page %d, %d Pages.\n",lba,pba,page,pages);

		error = ssfdci->read_data_func(ssfdci,pba,page,ptr,pages << ssfdci->geo.pageshift );
		if( error )
			break;

		page = 0;
		lba++;
		sectors -= pages;
		ptr += (pages << ssfdci->geo.pageshift);
	}


    if( use_sg ){
		if(!error){
            transferred = 0;
            for (i=0; i<use_sg && transferred<len; i++) {
                memcpy(
					sg[i].address,
					buffer+transferred,
					len-transferred > sg[i].length ? sg[i].length : len-transferred
				);
                transferred += sg[i].length;
	        }
		}

		kfree(buffer);
    }

	US_DEBUGP("ssfdc_read_data: leaving with code %d.\n",error);

    return error;
}
