
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <asm/uaccess.h>
#include <linux/usb.h>


/* Define these values to match your devices */
#define USB_OPENRADIO_VENDOR_ID	0x04b4
#define USB_OPENRADIO_PRODUCT_ID	0x8613
#define USB_OPENRADIO_INTERFACE	0
#define USB_OPENRADIO_ENDPOINT	0x86
#define USB_OPENRADIO_MINOR_BASE	240

#define USB_OPENRADIO_MAXURB	4
#define USB_OPENRADIO_BUFSIZE	512

#define MAX_TRANSFER		( PAGE_SIZE - 512 )
#define WRITES_IN_FLIGHT	8

#define USB_OPENRADIO_MULTI

/* table of devices that work with this driver */
static struct usb_device_id openradio_table [] = {
	{ USB_DEVICE(USB_OPENRADIO_VENDOR_ID, USB_OPENRADIO_PRODUCT_ID) },
	{ }					/* Terminating entry */
};
MODULE_DEVICE_TABLE (usb, openradio_table);

typedef struct
{
	struct urb *		urb;
	char *			buf;
	int			running;
} usb_readtask;

/* Structure to hold all of our device specific stuff */
struct usb_openradio {
	struct usb_device *	udev;			/* the usb device for this device */
	struct usb_interface *	interface;		/* the interface for this device */
	__u8			bulk_in_endpointAddr;	/* the address of the bulk in endpoint */
	struct kref		kref;
	size_t			bulk_in_size;		/* the size of the receive buffer */
	int			running;
	int			callbacks;
#ifdef USB_OPENRADIO_MULTI
	usb_readtask		read[USB_OPENRADIO_MAXURB];
	struct semaphore	sem_wantdata;		
	struct semaphore	sem_hasdata;		
	char			*read_ptr;
	int			read_len;
#else
	unsigned char *		bulk_in_buffer;		/* the buffer to receive data */
#endif
};
#define to_openradio_dev(d) container_of(d, struct usb_openradio, kref)

static struct usb_driver openradio_driver;

#ifdef USB_OPENRADIO_MULTI
static void openradio_read_bulk_callback(struct urb *urb, struct pt_regs *regs);
#endif


static void openradio_delete(struct kref *kref)
{	
	struct usb_openradio *dev = to_openradio_dev(kref);
	printk(KERN_ERR "delete\n");

	usb_put_dev(dev->udev);
#ifndef USB_OPENRADIO_MULTI
	kfree (dev->bulk_in_buffer);
#endif
	kfree (dev);
}

#ifdef USB_OPENRADIO_MULTI
static void usb_openradio_read_start(struct usb_openradio *dev,int *retval)
{
	int i;
	printk(KERN_ERR "read_start\n");
	dev->running=1;
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
	{
		usb_fill_bulk_urb(dev->read[i].urb, dev->udev,
				  usb_rcvbulkpipe(dev->udev, USB_OPENRADIO_ENDPOINT),
				  dev->read[i].buf, USB_OPENRADIO_BUFSIZE, openradio_read_bulk_callback, dev);
		dev->read[i].urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		*retval = usb_submit_urb(dev->read[i].urb, GFP_KERNEL);
		if(!*retval) dev->read[i].running=1;
	}

}

static void usb_openradio_read_stop(struct usb_openradio *dev)
{
	int i;
	printk(KERN_ERR "read_stop\n");
	dev->running=0;
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
	{
		if(dev->read[i].running)
		{
			usb_kill_urb(dev->read[i].urb);
			dev->read[i].running=1;
		}
	}
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
		up(&dev->sem_wantdata);
	while(dev->callbacks > 0) schedule();

}

#endif

static void usb_openradio_deallocbuffers(struct usb_openradio *dev)
{
#ifdef USB_OPENRADIO_MULTI
	int i;
	printk(KERN_ERR "Dealloc buffers");
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
	{
		if(dev->read[i].urb != NULL) { usb_free_urb(dev->read[i].urb); dev->read[i].urb=NULL; }
		if(dev->read[i].buf != NULL) { kfree(dev->read[i].buf); dev->read[i].buf=NULL; }
	}
#endif
}

static int usb_openradio_allocbuffers(struct usb_openradio *dev)
{
#ifdef USB_OPENRADIO_MULTI
	int i;
	printk(KERN_ERR "alloc buffers\n");
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
	{
		if(dev->read[i].urb == NULL)
		{
			dev->read[i].urb = usb_alloc_urb(0, GFP_KERNEL);
		}
		if(dev->read[i].buf == NULL)
		{
			dev->read[i].buf = kmalloc(USB_OPENRADIO_BUFSIZE,GFP_KERNEL);
		}
		if (!dev->read[i].urb  || !dev->read[i].buf)
		{
			printk(KERN_INFO "%d %p %p\n",i,dev->read[i].urb,dev->read[i].buf);
			usb_openradio_deallocbuffers(dev);
			return -ENOMEM;
		}
	}
#endif
	printk(KERN_ERR "alloc success\n");
	return 0;
}

static int openradio_open(struct inode *inode, struct file *file)
{
	struct usb_openradio *dev=NULL;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;
	printk(KERN_ERR "open\n");

	subminor = iminor(inode);
	dev = (struct usb_openradio *)file->private_data;

	interface = usb_find_interface(&openradio_driver, subminor);
	if (!interface) {
		printk ("%s - error, can't find device for minor %d\n",
		     __FUNCTION__, subminor);
		return  -ENODEV;
	}

	dev = usb_get_intfdata(interface);
	if (!dev) {
		return -ENODEV;
	}

	/* increment our usage count for the device */
	kref_get(&dev->kref);

	/* save our object in the file's private structure */
	file->private_data = dev;
	retval |=usb_openradio_allocbuffers(dev);
	printk(KERN_ERR "retval=%d\n",retval);
// todo 	if(!retval) return retval;
	usb_openradio_read_start(dev,&retval);
	return retval;
}

static int openradio_release(struct inode *inode, struct file *file)
{
	struct usb_openradio *dev;
	printk(KERN_ERR "release\n");

	dev = (struct usb_openradio *)file->private_data;
	if (dev == NULL)
		return -ENODEV;

#ifdef USB_OPENRADIO_MULTI
	usb_openradio_read_stop(dev);
	usb_openradio_deallocbuffers(dev);
#endif

	/* decrement the count on our device */
	kref_put(&dev->kref, openradio_delete);
	return 0;
}

#ifdef USB_OPENRADIO_MULTI

static void openradio_read_bulk_callback(struct urb *urb, struct pt_regs *regs)
{
	struct usb_openradio *dev;
	dev = (struct usb_openradio *)urb->context;
	printk(KERN_ERR "callback\n");
	dev->callbacks++;
	if(!dev->running) goto end;
	printk(KERN_ERR "read_bulk_callback\n");

	/* sync/async unlink faults aren't errors */
	if (urb->status && 
	    !(urb->status == -ENOENT || 
	      urb->status == -ECONNRESET ||
	      urb->status == -ESHUTDOWN)) {
		dbg("%s - nonzero write bulk status received: %d",
		    __FUNCTION__, urb->status);
	}

	printk(KERN_ERR "openradio_read_bulk_callback_1\n");
	// nur wenn nicht schon up !

	// as long as data wanted
	down_interruptible(&dev->sem_wantdata);
	if(!dev->running) goto end;
	printk(KERN_ERR "openradio_read_bulk_callback_2\n");
	

	if(dev->read_len > urb->actual_length) dev->read_len=urb->actual_length;
	// villeicht zu viel
	if (copy_to_user(dev->read_ptr, urb->transfer_buffer,dev->read_len))
	{
		dev->read_len=-EFAULT;
	}

	printk(KERN_ERR "openradio_read_bulk_callback_3\n");
	up(&dev->sem_hasdata);
	printk(KERN_ERR "openradio_read_bulk_callback_4\n");

	// start new usb transfer
	if(!dev->running) goto end;
	usb_submit_urb(urb, GFP_KERNEL);
	printk(KERN_ERR "openradio_read_bulk_callback_5\n");
end:
	dev->callbacks--;
	return;

}

static ssize_t openradio_read1(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct usb_openradio *dev;
	int retval = 0;

	printk(KERN_ERR "read_1\n");
	dev = (struct usb_openradio *)file->private_data;
	dev->callbacks++;

	dev->read_ptr=buffer;
	dev->read_len=count;
	up(&dev->sem_wantdata);

	if(!dev->running) goto end;
	down_interruptible(&dev->sem_hasdata);
	if(!dev->running) goto end;
	printk(KERN_ERR "read_1 with data\n");
	retval=dev->read_len;

	dev->read_len=0;
	dev->read_ptr=NULL;

	printk(KERN_ERR "read_1 with data 2\n");

	end:
	dev->callbacks--;
	return retval;
}

#else 

static ssize_t openradio_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct usb_openradio *dev;
	int retval = 0;
	int bytes_read;
	int pipe;

	printk(KERN_ERR "read\n");
	dev = (struct usb_openradio *)file->private_data;
	
	/* do a blocking bulk read to get data from the device */
	pipe=usb_rcvbulkpipe(dev->udev, dev->bulk_in_endpointAddr),
	retval = usb_bulk_msg(dev->udev,
			      pipe,
			      dev->bulk_in_buffer,
			      min(dev->bulk_in_size, count),
			      &bytes_read, 1000);


	/* if the read was successful, copy the data to userspace */
	if (!retval) {
		if (copy_to_user(buffer, dev->bulk_in_buffer, bytes_read))
			retval = -EFAULT;
		else
			retval = bytes_read;
	}

	return retval;
}

#endif


static struct file_operations openradio_fops = {
	.owner =	THIS_MODULE,
#ifdef USB_OPENRADIO_MULTI
	.read =		openradio_read1,
#else
	.read =		openradio_read,
#endif
	.open =		openradio_open,
	.release =	openradio_release,
};

/* 
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with the driver core
 */
static struct usb_class_driver openradio_class = {
	.name =		"openradio%d",
	.fops =		&openradio_fops,
	.minor_base =	USB_OPENRADIO_MINOR_BASE,
};

static int openradio_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	struct usb_openradio *dev = NULL;
	struct usb_host_interface *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	size_t buffer_size;
	int i;
	int retval = -ENOMEM;
	printk(KERN_ERR "probe\n");

	/* allocate memory for our device state and initialize it */
	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL) {
		printk(KERN_ERR "Out of memory\n");
		goto error;
	}
	kref_init(&dev->kref);
#ifdef USB_OPENRADIO_MULTI
	sema_init(&dev->sem_wantdata, 0);
	sema_init(&dev->sem_hasdata, 0);
	for(i=0;i<USB_OPENRADIO_MAXURB;i++)
	{
		dev->read[i].urb=NULL;
		dev->read[i].buf=NULL;
		dev->read[i].running=0;
	}
	dev->running=0;
	dev->callbacks=0;
	dev->read_ptr=NULL;
	dev->read_len=0;
#endif
	dev->udev = usb_get_dev(interface_to_usbdev(interface));
	dev->interface = interface;

        if (usb_set_interface (dev->udev, USB_OPENRADIO_INTERFACE, 1) < 0)
        {
                err("set_interface failed");
                return -EINVAL;
        }


	/* set up the endpoint information */
	/* use only the first bulk-in and bulk-out endpoints */
	iface_desc = interface->cur_altsetting;
	for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {
		endpoint = &iface_desc->endpoint[i].desc;

		if (!dev->bulk_in_endpointAddr &&
		    ((endpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK) == USB_DIR_IN) &&
		    ((endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_BULK) &&
		    endpoint->bEndpointAddress == USB_OPENRADIO_ENDPOINT)  {
			/* we found a bulk in endpoint */
			buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
			dev->bulk_in_size = buffer_size;
			dev->bulk_in_endpointAddr = endpoint->bEndpointAddress;
#ifndef USB_OPENRADIO_MULTI
			dev->bulk_in_buffer = kmalloc(buffer_size, GFP_KERNEL);
			if (!dev->bulk_in_buffer) {
				printk(KERN_ERR "Could not allocate bulk_in_buffer\n");
				goto error;
			}
#endif
		}

	}
	if (!(dev->bulk_in_endpointAddr )) {
		printk(KERN_ERR "Could not find both bulk-in and bulk-out endpoints\n");
		goto error;
	}

	/* save our data pointer in this interface device */
	usb_set_intfdata(interface, dev);


	/* we can register the device now, as it is ready */
	retval = usb_register_dev(interface, &openradio_class);
	if (retval) {
		/* something prevented us from registering this driver */
		printk(KERN_ERR "Not able to get a minor for this device.\n");
		usb_set_intfdata(interface, NULL);
		goto error;
	}

	/* let the user know what node this device is now attached to */
	info("USB Openradio device now attached to USBOpenradio-%d", interface->minor);

	return 0;

error:
	if (dev)
		kref_put(&dev->kref, openradio_delete);
	return retval;
}

static void openradio_disconnect(struct usb_interface *interface)
{
	struct usb_openradio *dev;
	int minor = interface->minor;
	printk(KERN_ERR "disconnect\n");

	dev = usb_get_intfdata(interface);
	usb_openradio_read_stop(dev);
	usb_openradio_deallocbuffers(dev);

	/* prevent openradio_open() from racing openradio_disconnect() */
	lock_kernel();

	dev = usb_get_intfdata(interface);
	usb_set_intfdata(interface, NULL);

	/* give back our minor */
	usb_deregister_dev(interface, &openradio_class);

	unlock_kernel();


	/* decrement our usage count */
	kref_put(&dev->kref, openradio_delete);

	info("USB Openradio #%d now disconnected", minor);
}

static struct usb_driver openradio_driver = {
	.name =		"openradio",
	.probe =	openradio_probe,
	.disconnect =	openradio_disconnect,
	.id_table =	openradio_table,
};

static int __init usb_openradio_init(void)
{
	int result;

	/* register this driver with the USB subsystem */
	result = usb_register(&openradio_driver);
	if (result)
		printk(KERN_ERR "usb_register failed. Error number %d\n", result);

	return result;
}

static void __exit usb_openradio_exit(void)
{
	/* deregister this driver with the USB subsystem */
	usb_deregister(&openradio_driver);
}

module_init (usb_openradio_init);
module_exit (usb_openradio_exit);

MODULE_LICENSE("GPL");
