/*
 * HAUPPAUGE WINTV USB Driver
 *
 * Based on the Linux OV511 driver. 
 *
 * OmniVision OV511 Camera-to-USB Bridge Driver
 * Copyright (c) 1999/2000 Mark W. McClelland
 * Many improvements by Bret Wallach
 *
 * Based on the Linux CPiA driver.
 * 
 * Released under GPL v.2 license.
 *
 * Important keywords in comments:
 *    CAMERA SPECIFIC - Camera specific code; may not work with other cameras.
 *    DEBUG - Debugging code.
 *    FIXME - Something that is broken or needs improvement.
 *
 * Version: 1.07
 *
 * Please see the file: linux/Documentation/usb/wintv.txt 
 * and the website at:  http://people.delphi.com/mmcclelland/linux/ 
 * for more info.
 */

/*
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __NO_VERSION__

/* Handle mangled (versioned) external symbols */

#include <linux/config.h>   /* retrieve the CONFIG_* macros */
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define MODVERSIONS  /* force it on */
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/smp_lock.h>
#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <linux/wrapper.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <asm/io.h>

#include <linux/usb.h>
#include "wintv.h"

#define WINTV_I2C_RETRIES 3

#define SAA7111A_AUTO_ADJUST 1

/* Video Size 640 x 480 x 3 bytes for RGB */
#define MAX_FRAME_SIZE (352 * 288 * 3)
#define MAX_DATA_SIZE (MAX_FRAME_SIZE + sizeof(struct timeval))

// FIXME - Should find a better way to do this.
#define DEFAULT_WIDTH 352
#define DEFAULT_HEIGHT 288

char kernel_version[] = UTS_RELEASE;

/*******************************/
/* Memory management functions */
/*******************************/

#define MDEBUG(x)	do { } while(0)		/* Debug memory management */

static struct usb_driver wintv_driver;

/* Given PGD from the address space's page table, return the kernel
 * virtual mapping of the physical memory mapped at ADR.
 */
static inline unsigned long uvirt_to_kva(pgd_t *pgd, unsigned long adr)
{
	unsigned long ret = 0UL;
	pmd_t *pmd;
	pte_t *ptep, pte;

	if (!pgd_none(*pgd)) {
		pmd = pmd_offset(pgd, adr);
		if (!pmd_none(*pmd)) {
			ptep = pte_offset(pmd, adr);
			pte = *ptep;
			if (pte_present(pte))
				ret = page_address(pte_page(pte)) | (adr & (PAGE_SIZE-1));
		}
	}
	MDEBUG(printk("uv2kva(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long uvirt_to_bus(unsigned long adr)
{
	unsigned long kva, ret;

	kva = uvirt_to_kva(pgd_offset(current->mm, adr), adr);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("uv2b(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long kvirt_to_bus(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("kv2b(%lx-->%lx)", adr, ret));
	return ret;
}

/* Here we want the physical address of the memory.
 * This is used when initializing the contents of the
 * area and marking the pages as reserved.
 */
static inline unsigned long kvirt_to_pa(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = __pa(kva);
	MDEBUG(printk("kv2pa(%lx-->%lx)", adr, ret));
	return ret;
}

static void *rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr, page;

	/* Round it off to PAGE_SIZE */
	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	mem = vmalloc(size);
	if (!mem)
		return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_reserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return mem;
}

static void rvfree(void *mem, unsigned long size)
{
	unsigned long adr, page;

	if (!mem)
		return;

	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	adr=(unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_unreserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}
	vfree(mem);
}

int wintv_reg_write(struct usb_device *dev, unsigned char reg, unsigned char value)
{
	int rc;

	rc = usb_control_msg(dev,
		usb_sndctrlpipe(dev, 1),
		0x33 /* REG_IO */,
		USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
		0, (__u16)reg, &value, 1, HZ);	
			
#if 1
	PDEBUG("reg:write, rc: 0x%02X:0x%02X, 0x%x", reg, value, rc);
#endif
			
	return rc;
}

/* returns: negative is error, pos or zero is data */
int wintv_reg_read(struct usb_device *dev, unsigned char reg)
{
	int rc;
	unsigned char buffer[1];

	rc = usb_control_msg(dev,
		usb_rcvctrlpipe(dev,1),
		0x33 /* REG_IO */,
		USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
		0, (__u16)reg, buffer, 1, HZ);
                               
#if 1
	PDEBUG("reg:read, rc: 0x%02X:0x%02X, 0x%X", reg, buffer[0], rc);
#endif
	
	if(rc < 0)
		return rc;
	else
		return buffer[0];	
}

int wintv_i2c_write(struct usb_device *dev, unsigned char reg, unsigned char value)
{
	int rc, retries;

#if 1
	PDEBUG("i2c write: 0x%02X:0x%02X", reg, value);
#endif
	/* Byte write cycle */
	for(retries = WINTV_I2C_RETRIES;;) {
	        /* Select SAA7111A Address for writing */
		rc = wintv_reg_write(dev, WINTV_REG_SER_ADRS, SAA7111A_I2C_WRITE_ID);
		if (rc < 0) return rc;
	  
		/* Select SAA7111A Sub Address (register) */
		rc = wintv_reg_write(dev, WINTV_REG_SER_DAT1, reg);
		if (rc < 0) return rc;

		/* Write "value" to I2C data port */
		rc = wintv_reg_write(dev, WINTV_REG_SER_DAT2, value);
		if (rc < 0) return rc;

		/* Initiate byte write cycle */
		rc = wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x12);
		if (rc < 0) return rc;
#if 1
		/* Test for Busy and ACK */
		do rc = wintv_reg_read(dev, WINTV_REG_SER_CONT);
		while(rc > 0 && ((rc&0x10) != 0)); /* Retry while busy */
		if (rc < 0) return rc;

		if((rc&0x20) == 0) /* Ack? */
#endif
			break;

		/* I2C abort */	
		wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x00);

		if (--retries < 0) return -1;
	}

	return 0;
}

/* returns: negative is error, pos or zero is data */
int wintv_i2c_read(struct usb_device *dev, unsigned char reg)
{
	int rc, value, retries;

	/* Byte write cycle */
	for(retries = WINTV_I2C_RETRIES;;) {
	  
	        /* Select SAA7111A Address for writing */
		rc = wintv_reg_write(dev, WINTV_REG_SER_ADRS, SAA7111A_I2C_WRITE_ID);
		if (rc < 0) return rc;
	  
		/* Select SAA7111A Sub Address (register) */
		rc = wintv_reg_write(dev, WINTV_REG_SER_DAT1, reg);
		if (rc < 0) return rc;

		/* Initiate byte write cycle */
		rc = wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x11);
		if (rc < 0) return rc;

		/* Test for Busy and ACK */
		do rc = wintv_reg_read(dev, WINTV_REG_SER_CONT);
		while(rc > 0 && ((rc&0x10) != 0)); /* Retry while busy */
		if (rc < 0) return rc;

		if((rc&0x20) == 0) /* Ack? */
			break;


		/* I2C abort */	
		wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x00);

		if (--retries < 0) return -1;
	}

	/* Byte read cycle */
	for(retries = WINTV_I2C_RETRIES;;) {
		/* Select SAA7111A Address for reading */
		rc = wintv_reg_write(dev, WINTV_REG_SER_ADRS, SAA7111A_I2C_READ_ID);
		if (rc < 0) return rc;

		/* Initiate byte read cycle */
		rc = wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x19);
		if (rc < 0) return rc;

		/* Test for Busy and ACK */
		do rc = wintv_reg_read(dev, WINTV_REG_SER_CONT);
		while(rc > 0 && ((rc&0x10) != 0)); /* Retry while busy */
		if (rc < 0) return rc;

		if((rc&0x20) == 0) /* Ack? */
			break;

		/* I2C abort */	
		rc = wintv_reg_write(dev, WINTV_REG_SER_CONT, 0x00);
		if (rc < 0) return rc;

		if (--retries < 0) return -1;
	}

	value = wintv_reg_read(dev, WINTV_REG_SER_DAT1);
#if 1
	PDEBUG("i2c read: 0x%02X:0x%02X", reg, value);
#endif
		
#if 0
	/* This is needed to make wintv_i2c_write() work */
	rc = wintv_reg_write(dev, WINTV_REG_I2C_CONTROL, 0x05);
	if (rc < 0) return rc;
#endif	
	return (value);
}


// This version doesn't always work
#if 0
 /* returns: negative is error, pos or zero is data */
 int wintv_i2c_read(struct usb_device *dev, unsigned char reg)
 {
	int rc, value;

	/* Select camera register */
	rc = wintv_reg_write(dev, WINTV_REG_I2C_SUB_ADDRESS_2_BYTE, reg);
	if (rc < 0) return rc;
 

	/* Initiate 2-byte write cycle */
	rc = wintv_reg_write(dev, WINTV_REG_I2C_CONTROL, 0x03);
	if (rc < 0) return rc;
 

	/* Initiate 2-byte read cycle */
	rc = wintv_reg_write(dev, WINTV_REG_I2C_CONTROL, 0x05);
	if (rc < 0) return rc;
 
 	value = wintv_reg_read(dev, WINTV_REG_I2C_DATA_PORT);
 #if 0
 	PDEBUG("i2c read: 0x%02X:0x%02X", reg, value);
 #endif
 		
 	return (value);
 }
#endif

static int wintv_write_regvals(struct usb_device *dev,
			       struct wintv_regvals * pRegvals)
{
	int ret;
	while(pRegvals->bus != WINTV_DONE_BUS) {
		if (pRegvals->bus == WINTV_REG_BUS) {
			if ((ret = wintv_reg_write(dev, pRegvals->reg,
			                           pRegvals->val)) < 0)
				return ret;
		} else if (pRegvals->bus == WINTV_I2C_BUS) {
			if ((ret = wintv_i2c_write(dev, pRegvals->reg, 
			                           pRegvals->val)) < 0)
				return ret;
		} else {
		  err("Bad regval array");
		}
		pRegvals++;
	}
	return 0;
}

#if 0
static void wintv_dump_i2c_range( struct usb_device *dev, int reg1, int regn)
{
	int i;
	int rc;
	for(i=reg1; i<=regn; i++) {
	  rc = wintv_i2c_read(dev, i);
#if 0
	  PDEBUG("OV7610[0x%X] = 0x%X", i, rc);
#endif
	}
}

static void wintv_dump_i2c_regs( struct usb_device *dev)
{
	PDEBUG("I2C REGS");
	wintv_dump_i2c_range(dev, 0x00, 0x38);
}

static void wintv_dump_reg_range( struct usb_device *dev, int reg1, int regn)
{
	int i;
	int rc;
	for(i=reg1; i<=regn; i++) {
	  rc = wintv_reg_read(dev, i);
	  PDEBUG("WINTV[0x%X] = 0x%X", i, rc);
	}
}

static void wintv_dump_regs( struct usb_device *dev)
{
	PDEBUG("CAMERA INTERFACE REGS");
	wintv_dump_reg_range(dev, 0x10, 0x1f);
	PDEBUG("DRAM INTERFACE REGS");
	wintv_dump_reg_range(dev, 0x20, 0x23);
	PDEBUG("ISO FIFO REGS");
	wintv_dump_reg_range(dev, 0x30, 0x31);
	PDEBUG("PIO REGS");
	wintv_dump_reg_range(dev, 0x38, 0x39);
	wintv_dump_reg_range(dev, 0x3e, 0x3e);
	PDEBUG("I2C REGS");
	wintv_dump_reg_range(dev, 0x40, 0x49);
	PDEBUG("SYSTEM CONTROL REGS");
	wintv_dump_reg_range(dev, 0x50, 0x53);
	wintv_dump_reg_range(dev, 0x5e, 0x5f);
	PDEBUG("OmniCE REGS");
	wintv_dump_reg_range(dev, 0x70, 0x79);
	wintv_dump_reg_range(dev, 0x80, 0x9f);
	wintv_dump_reg_range(dev, 0xa0, 0xbf);

}
#endif

int wintv_reset(struct usb_device *dev, unsigned char reset_type)
{
	int rc;
	
	PDEBUG("Reset: type=0x%X", reset_type);
	rc = wintv_reg_write(dev, WINTV_REG_SYSTEM_RESET, reset_type);
	if (rc < 0)
		err("reset: command failed");

	rc = wintv_reg_write(dev, WINTV_REG_SYSTEM_RESET, 0);
	if (rc < 0)
		err("reset: command failed");

	return rc;
}

int wintv_set_packet_size(struct usb_wintv *wintv, int size)
{
	int alt, multiplier, rc;
		
#if 0
	PDEBUG("set packet size: %d", size);
#endif
	
	switch (size) {
		case 992:
			alt = 0;
			multiplier = 31;
			break;
		case 993:
			alt = 1;
			multiplier = 31;
			break;
		case 768:
			alt = 2;
			multiplier = 24;
			break;
		case 769:
			alt = 3;
			multiplier = 24;
			break;
		case 512:
			alt = 4;
			multiplier = 16;
			break;
		case 513:
			alt = 5;
			multiplier = 16;
			break;
		case 257:
			alt = 6;
			multiplier = 8;
			break;
		case 0:
			alt = 7;
			multiplier = 1; // FIXME - is this correct?
			break;
		default:
			err("Set packet size: invalid size (%d)", size);
			return -EINVAL;
	}

	rc = wintv_reg_write(wintv->dev, WINTV_REG_FIFO_PACKET_SIZE,
	                     multiplier);
	if (rc < 0) {
		err("Set packet size: Set FIFO size ret %d", rc);
		return -ENOMEM;
	}
	
	if (usb_set_interface(wintv->dev, wintv->iface, alt) < 0) {
		err("Set packet size: set interface error");
		return -EBUSY;
	}

	// FIXME - Should we only reset the FIFO?
	if (wintv_reset(wintv->dev, WINTV_RESET_NOREGS) < 0)
		return -ENOMEM;

	return 0;
}

static inline int ov7610_set_picture(struct usb_wintv *wintv,
                                     struct video_picture *p)
{
	int ret;

	/* Stop the camera */
	if (wintv_reg_write(wintv->dev, WINTV_REG_SYSTEM_RESET, 0x3d) < 0) {
		err("reset: command failed");
		return -EIO;
	}

	if((ret = wintv_i2c_read(wintv->dev, OV7610_REG_COM_B)) < 0)
		return -EIO;
#if 0
	if(wintv_i2c_write(wintv->dev, OV7610_REG_COM_B, ret & 0xfe) < 0)
		return -EIO;
#endif

	if(wintv_i2c_write(wintv->dev, OV7610_REG_SAT, p->colour >> 8) < 0)
		return -EIO;

	if(wintv_i2c_write(wintv->dev, OV7610_REG_CNT, p->contrast >> 8) < 0)
		return -EIO;

	if(wintv_i2c_write(wintv->dev, OV7610_REG_BRT, p->brightness >> 8) < 0)
		return -EIO;

	/* Restart the camera */
	if (wintv_reg_write(wintv->dev, WINTV_REG_SYSTEM_RESET, 0x0) < 0) {
		err("reset: command failed");
		return -EIO;
	}

	return 0;
}

static inline int ov7610_get_picture(struct usb_wintv *wintv,
                                     struct video_picture *p)
{
	int ret;

	/* Stop the camera */
	if (wintv_reg_write(wintv->dev, WINTV_REG_SYSTEM_RESET, 0x3d) < 0) {
		err("reset: command failed");
		return -EIO;
	}

	if((ret = wintv_i2c_read(wintv->dev, OV7610_REG_SAT)) < 0) return -EIO;
	p->colour = ret << 8;

	if((ret = wintv_i2c_read(wintv->dev, OV7610_REG_CNT)) < 0) return -EIO;
	p->contrast = ret << 8;

	if((ret = wintv_i2c_read(wintv->dev, OV7610_REG_BRT)) < 0) return -EIO;
	p->brightness = ret << 8;

	p->hue = 0x8000;
	p->whiteness = 105 << 8;
	p->depth = 24;
	p->palette = VIDEO_PALETTE_RGB24;

	/* Restart the camera */
	if (wintv_reg_write(wintv->dev, WINTV_REG_SYSTEM_RESET, 0x0) < 0) {
		err("reset: command failed");
		return -EIO;
	}

	return 0;
}

static int wintv_mode_init_regs(struct usb_wintv *wintv,
				int width, int height, int mode, int sub_flag)
{
	int rc = 0;
	struct usb_device *dev = wintv->dev;

#if 0
	PDEBUG("wintv_mode_init_regs(wintv, %d, %d, %d, %d)",
	       width, height, mode, sub_flag);
#endif

//	wintv_set_packet_size(wintv, 0);
	if (wintv_reg_write(dev, WINTV_REG_SYSTEM_RESET, 0x3d) < 0) {
		err("reset: command failed");
		return -EIO;
	}

	if (mode == VIDEO_PALETTE_GREY) {
		wintv_reg_write(dev, 0x16, 0);
		wintv_i2c_write(dev, 0xe, 0x44);
		wintv_i2c_write(dev, 0x13, 0x21);
	} else {
		wintv_reg_write(dev, 0x16, 1);
		wintv_i2c_write(dev, 0xe, 0x4);
		wintv_i2c_write(dev, 0x13, 0x1);
	}

	if (width == 640 && height == 480) {
		if (sub_flag) {
			wintv_i2c_write(wintv->dev, 0x17, 0x38+(wintv->subx>>2));
			wintv_i2c_write(wintv->dev, 0x18,
					0x3a+((wintv->subx+wintv->subw)>>2));
			wintv_i2c_write(wintv->dev, 0x19, 0x5+(wintv->suby>>1));
			wintv_i2c_write(wintv->dev, 0x1a,
					0x5+((wintv->suby+wintv->subh)>>1));
			wintv_reg_write(wintv->dev, 0x12, (wintv->subw>>3)-1);
			wintv_reg_write(wintv->dev, 0x13, (wintv->subh>>3)-1);
			wintv_i2c_write(dev, 0x11, 0x01);
		} else {
			wintv_i2c_write(wintv->dev, 0x17, 0x38);
			wintv_i2c_write(wintv->dev, 0x18, 0x3a + (640>>2));
			wintv_i2c_write(wintv->dev, 0x19, 0x5);
			wintv_i2c_write(wintv->dev, 0x1c, + (480>>1));
			wintv_reg_write(dev, 0x12, 0x4f);
			wintv_reg_write(dev, 0x13, 0x3d);
			if (mode == VIDEO_PALETTE_GREY) {
			  wintv_i2c_write(dev, 0x11, 4); /* check */
			} else {
			  wintv_i2c_write(dev, 0x11, 6); /* check */
			}
		}

		wintv_reg_write(dev, 0x14, 0x00);
		wintv_reg_write(dev, 0x15, 0x00);
		wintv_reg_write(dev, 0x18, 0x03);

		wintv_i2c_write(dev, 0x12, 0x24);
		wintv_i2c_write(dev, 0x14, 0x04);
		wintv_i2c_write(dev, 0x35, 0x9e);
	} else if (width == 320 && height == 240) {
		wintv_reg_write(dev, 0x12, 0x27);
		wintv_reg_write(dev, 0x13, 0x1f);
		wintv_reg_write(dev, 0x14, 0x00);
		wintv_reg_write(dev, 0x15, 0x00);
		wintv_reg_write(dev, 0x18, 0x03);

		if (mode == VIDEO_PALETTE_GREY) {
		  wintv_i2c_write(dev, 0x11, 1); /* check */
		} else {
		  wintv_i2c_write(dev, 0x11, 1); /* check */
		}

		wintv_i2c_write(dev, 0x12, 0x04);
		wintv_i2c_write(dev, 0x14, 0x24);
		wintv_i2c_write(dev, 0x35, 0x1e);
	} else {
		err("Unknown mode (%d, %d): %d", width, height, mode);
		rc = -EINVAL;
	}

//	wintv_set_packet_size(wintv, 993);

	if (wintv_reg_write(dev, WINTV_REG_SYSTEM_RESET, 0x00) < 0) {
		PDEBUG("reset: command failed");
		return -EIO;
	}

	return rc;
}

	
/*************************************************************

Turn a YUV4:2:0 block into an RGB block

*************************************************************/
#define LIMIT(x) ((((x)>0xffffff)?0xff0000:(((x)<=0xffff)?0:(x)&0xff0000))>>16)
static inline void wintv_move_420_block(int y00, int y01, int y10, int y11,
					int u, int v, int w,
					unsigned char * pOut)
{
	int r    = 68911 * v;
	int g    = -16915 * u + -35101 * v;
	int b    = 87097 * u;
	y00 *= 49152;
	y01 *= 49152;
	y10 *= 49152;
	y11 *= 49152;
	*(pOut+w*3) = LIMIT(r + y10);
	*pOut++     = LIMIT(r + y00);
	*(pOut+w*3) = LIMIT(g + y10);
	*pOut++     = LIMIT(g + y00);
	*(pOut+w*3) = LIMIT(b + y10);
	*pOut++     = LIMIT(b + y00);
	*(pOut+w*3) = LIMIT(r + y11);
	*pOut++     = LIMIT(r + y01);
	*(pOut+w*3) = LIMIT(g + y11);
	*pOut++     = LIMIT(g + y01);
	*(pOut+w*3) = LIMIT(b + y11);
	*pOut++     = LIMIT(b + y01);
}

/***************************************************************

For a 640x480 YUV4:2:0 images, data shows up in 1200 384 byte segments.  The
first 64 bytes of each segment are V, the next 64 are U.  The V and
U are arranged as follows:

  0  1 ...  7
  8  9 ... 15
       ...   
 56 57 ... 63

The next 256 bytes are Y data and represent 4 squares of 8x8 pixels as
follows:

  0  1 ...  7    64  65 ...  71   ...  192 193 ... 199
  8  9 ... 15    72  73 ...  79        200 201 ... 207
       ...              ...                    ...
 56 57 ... 63   120 121     127        248 249 ... 255

If WINTV_DUMPPIX is defined, _parse_data just dumps the
incoming segments, verbatim, in order, into the frame.
When used with vidcat -f ppm -s 640x480 this puts the data
on the standard output and can be analyzed with the parseppm.c
utility I wrote.  That's a much faster way for figuring out how
this data is scrambled.

****************************************************************/ 
#define HDIV 8
#define WDIV (256/HDIV)


static void wintv_parse_data_rgb24(unsigned char * pIn0,
				   unsigned char * pOut0,
				   int iOutY,
				   int iOutUV,
				   int iHalf,
				   int iWidth)
 			    			    
{
#ifndef WINTV_DUMPPIX
    int k, l, m;
    unsigned char * pIn;
    unsigned char * pOut, * pOut1;

    /* Just copy the Y's if in the first stripe */
    if (!iHalf) {
	pIn = pIn0 + 128;
	pOut = pOut0 + iOutY;
	for(k=0; k<4; k++) {
	    pOut1 = pOut;
	    for(l=0; l<8; l++) {
	      for(m=0; m<8; m++) {
		*pOut1 = *pIn++;
		pOut1 += 3;
	      }
	      pOut1 += (iWidth - 8) * 3;
	    }
	    pOut += 8 * 3;
	}
    }

    /* Use the first half of VUs to calculate value */
    pIn = pIn0;
    pOut = pOut0 + iOutUV;
    for(l=0; l<4; l++) {
	for(m=0; m<8; m++) {
	    int y00 = *(pOut);
	    int y01 = *(pOut+3);
	    int y10 = *(pOut+iWidth*3);
	    int y11 = *(pOut+iWidth*3+3);
	    int u   = *(pIn+64) - 128;
	    int v   = *pIn++ - 128;
	    wintv_move_420_block(y00, y01, y10, y11, u, v, iWidth, pOut);
	    pOut += 6;
	}
	pOut += (iWidth*2 - 16) * 3;
    }

    /* Just copy the other UV rows */
    for(l=0; l<4; l++) {
	for(m=0; m<8; m++) {
	  *pOut++ = *(pIn + 64);
	  *pOut = *pIn++;
	  pOut += 5;
	}
	pOut += (iWidth*2 - 16) * 3;
    }

    /* Calculate values if it's the second half */
    if (iHalf) {
	pIn = pIn0 + 128;
	pOut = pOut0 + iOutY;
	for(k=0; k<4; k++) {
	    pOut1 = pOut;
	    for(l=0; l<4; l++) {
	      for(m=0; m<4; m++) {
		int y10 = *(pIn+8);
		int y00 = *pIn++;
		int y11 = *(pIn+8);
		int y01 = *pIn++;
		int u   = *pOut1 - 128;
		int v   = *(pOut1+1) - 128;
		wintv_move_420_block(y00, y01, y10, y11, u, v, iWidth, pOut1);
		pOut1 += 6;
	      }
	      pOut1 += (iWidth*2 - 8) * 3;
	      pIn += 8;
	    }
	    pOut += 8 * 3;
	}
    }

#else
	/* Just dump pix data straight out for debug */
	int i;
	pOut0 += iSegmentY * 384;
	for(i=0; i<384; i++) {
	  *pOut0++ = *pIn0++;
	}
#endif
}

/***************************************************************

For 640x480 RAW BW images, data shows up in 1200 256 byte segments.
The segments represent 4 squares of 8x8 pixels as
follows:

  0  1 ...  7    64  65 ...  71   ...  192 193 ... 199
  8  9 ... 15    72  73 ...  79        200 201 ... 207
       ...              ...                    ...
 56 57 ... 63   120 121     127        248 249 ... 255

****************************************************************/ 
static void wintv_parse_data_grey(unsigned char * pIn0,
				  unsigned char * pOut0,
				  int iOutY,
				  int iWidth)
			    
{
    int k, l, m;
    unsigned char * pIn;
    unsigned char * pOut, * pOut1;

    pIn = pIn0;
    pOut = pOut0 + iOutY;
    for(k=0; k<4; k++) {
      pOut1 = pOut;
      for(l=0; l<8; l++) {
	for(m=0; m<8; m++) {
	  *pOut1++ = *pIn++;
	}
	pOut1 += iWidth - 8;
      }
      pOut += 8;
    }
}

static int wintv_move_data(struct usb_wintv *wintv, urb_t *urb)
{
	unsigned char *cdata;
	int i, totlen = 0;
	int aPackNum[10];
	struct wintv_frame *frame;

	for (i = 0; i < urb->number_of_packets; i++) {
		int n = urb->iso_frame_desc[i].actual_length;
		int st = urb->iso_frame_desc[i].status;
		urb->iso_frame_desc[i].actual_length = 0;
		urb->iso_frame_desc[i].status = 0;
		cdata = urb->transfer_buffer + urb->iso_frame_desc[i].offset;

		aPackNum[i] = n ? cdata[992] : -1;

		if (!n || wintv->curframe == -1) continue;

		if (st)
			PDEBUG("data error: [%d] len=%d, status=%d", i, n, st);

		frame = &wintv->frame[wintv->curframe];
		
		/* Can we find a frame end */
		if ((cdata[0] | cdata[1] | cdata[2] | cdata[3] | 
		     cdata[4] | cdata[5] | cdata[6] | cdata[7]) == 0 &&
		    (cdata[8] & 8) && (cdata[8] & 0x80)) {

		    struct timeval *ts;
		    ts = (struct timeval *)(frame->data + MAX_FRAME_SIZE);
		    do_gettimeofday(ts);
#if 0
		    PDEBUG("Frame End, curframe = %d, packnum=%d, hw=%d, vw=%d",
			   wintv->curframe, (int)(cdata[992]),
			   (int)(cdata[9]), (int)(cdata[10]));
#endif

		    if (frame->scanstate == STATE_LINES) {
		        int iFrameNext;
				frame->grabstate = FRAME_DONE;
		        if (waitqueue_active(&frame->wq)) {
			  		frame->grabstate = FRAME_DONE;
			 	 	wake_up_interruptible(&frame->wq);
				}
				/* If next frame is ready or grabbing, point to it */
				iFrameNext = (wintv->curframe + 1) % WINTV_NUMFRAMES;
				if (wintv->frame[iFrameNext].grabstate== FRAME_READY ||
				    wintv->frame[iFrameNext].grabstate== FRAME_GRABBING) {
				  wintv->curframe = iFrameNext;
				  wintv->frame[iFrameNext].scanstate = STATE_SCANNING;
				} else {
#if 0
				  PDEBUG("Frame not ready? state = %d",
					 wintv->frame[iFrameNext].grabstate);
#endif
				  wintv->curframe = -1;
				}
		    }
		}

		/* Can we find a frame start */
		else if ((cdata[0] | cdata[1] | cdata[2] | cdata[3] | 
			  cdata[4] | cdata[5] | cdata[6] | cdata[7]) == 0 &&
			 (cdata[8] & 8)) {
#if 0
			PDEBUG("wintv: Found Frame Start!, framenum = %d",
			       wintv->curframe);
#endif
		    frame->scanstate = STATE_LINES;
			frame->segment = 0;
		}

		/* Are we in a frame? */
		if (frame->scanstate == STATE_LINES) {
			unsigned char * pData;
			int iPix;

			/* Deal with leftover from last segment, if any */
			if (frame->segment) {
			  pData = wintv->scratch;
			  iPix = - wintv->scratchlen;
			  memmove(pData + wintv->scratchlen, cdata,
				  iPix+frame->segsize);
			} else {
			  pData = &cdata[iPix = 9];
		 	}

			/* Parse the segments */
			while(iPix <= 992 - frame->segsize &&
			      frame->segment < frame->width * frame->height / 256) {
			  int iSegY;
			  int iSegUV;
			  int iY, jY, iUV, jUV;
			  int iOutY, iOutUV;
			  unsigned char * pOut;

			  iSegY = iSegUV = frame->segment;
			  pOut = frame->data;
			  
			  frame->segment++;
			  iPix += frame->segsize;

			  if (frame->sub_flag) {
			    int iSeg1;
			    iSeg1 = iSegY / (wintv->subw / 32);
			    iSeg1 *= frame->width / 32;
			    iSegY = iSeg1 + (iSegY % (wintv->subw / 32));
			    if (iSegY >= frame->width * wintv->subh / 256)
			      break;

			    iSeg1 = iSegUV / (wintv->subw / 16);
			    iSeg1 *= frame->width / 16;
			    iSegUV = iSeg1 + (iSegUV % (wintv->subw / 16));

			    pOut += (wintv->subx +
				     wintv->suby * frame->width) * frame->depth;
			  }

			  iY     = iSegY / (frame->width / WDIV);
			  jY     = iSegY - iY * (frame->width / WDIV);
			  iOutY  = (iY*HDIV*frame->width + jY*WDIV) * frame->depth;
			  iUV    = iSegUV / (frame->width / WDIV * 2);
			  jUV    = iSegUV - iUV * (frame->width / WDIV * 2);
			  iOutUV = (iUV*HDIV*2*frame->width + jUV*WDIV/2) * frame->depth;

			  if (frame->format == VIDEO_PALETTE_GREY) {
			    wintv_parse_data_grey(pData, pOut, iOutY, frame->width);
			  } else if (frame->format == VIDEO_PALETTE_RGB24) {
			    wintv_parse_data_rgb24(pData, pOut, iOutY, iOutUV, iY & 1,
						   frame->width);
			  }
			  pData = &cdata[iPix];
			}

			/* Save extra data for next time */
			if (frame->segment < frame->width * frame->height / 256) {
			  wintv->scratchlen = 992 - iPix;
			  if (wintv->scratchlen < frame->segsize) {
			    memmove(wintv->scratch, pData, wintv->scratchlen);
			  } else {
			    wintv->scratchlen = 0;
			  }
			}
		}
	}

#if 0
	PDEBUG("pn: %d %d %d %d %d %d %d %d %d %d\n",
	       aPackNum[0], aPackNum[1], aPackNum[2], aPackNum[3], aPackNum[4],
	       aPackNum[5],aPackNum[6], aPackNum[7], aPackNum[8], aPackNum[9]);
#endif
	return totlen;
}

static void wintv_isoc_irq(struct urb *urb)
{
	int len;
	struct usb_wintv *wintv = urb->context;
	struct wintv_sbuf *sbuf;

	if (!wintv->dev)
		return;

	if (!wintv->streaming) {
		PDEBUG("hmmm... not streaming, but got interrupt\n");
		return;
	}
	
	sbuf = &wintv->sbuf[wintv->cursbuf];

	/* Copy the data received into our scratch buffer */
	if (wintv->curframe >= 0)
	  len = wintv_move_data(wintv, urb);
	else if (waitqueue_active(&wintv->wq))
	  wake_up_interruptible(&wintv->wq);
	
	/* Move to the next sbuf */
	wintv->cursbuf = (wintv->cursbuf + 1) % WINTV_NUMSBUF;

	return;
}

static int wintv_init_isoc(struct usb_wintv *wintv)
{
	urb_t *urb;
	int fx, err;
	
	wintv->compress = 0;
	wintv->curframe = -1;
	wintv->cursbuf = 0;
	wintv->scratchlen = 0;

	wintv_set_packet_size(wintv, 993);

	/* We double buffer the Iso lists */
	urb = usb_alloc_urb(FRAMES_PER_DESC);
	
	if (!urb) {
		err("wintv_init_isoc: usb_alloc_urb ret. NULL");
		return -ENOMEM;
	}
	wintv->sbuf[0].urb = urb;
	urb->dev = wintv->dev;
	urb->context = wintv;
	urb->pipe = usb_rcvisocpipe(wintv->dev, WINTV_ENDPOINT_ADDRESS);
	urb->transfer_flags = USB_ISO_ASAP;
	urb->transfer_buffer = wintv->sbuf[0].data;
 	urb->complete = wintv_isoc_irq;
 	urb->number_of_packets = FRAMES_PER_DESC;
 	urb->transfer_buffer_length = FRAME_SIZE_PER_DESC * FRAMES_PER_DESC;
 	for (fx = 0; fx < FRAMES_PER_DESC; fx++) {
 		urb->iso_frame_desc[fx].offset = FRAME_SIZE_PER_DESC * fx;
		urb->iso_frame_desc[fx].length = FRAME_SIZE_PER_DESC;
	}

	urb = usb_alloc_urb(FRAMES_PER_DESC);
	if (!urb) {
		err("wintv_init_isoc: usb_alloc_urb ret. NULL");
		return -ENOMEM;
	}
	wintv->sbuf[1].urb = urb;
	urb->dev = wintv->dev;
	urb->context = wintv;
	urb->pipe = usb_rcvisocpipe(wintv->dev, WINTV_ENDPOINT_ADDRESS);
	urb->transfer_flags = USB_ISO_ASAP;
	urb->transfer_buffer = wintv->sbuf[1].data;
 	urb->complete = wintv_isoc_irq;
 	urb->number_of_packets = FRAMES_PER_DESC;
 	urb->transfer_buffer_length = FRAME_SIZE_PER_DESC * FRAMES_PER_DESC;
 	for (fx = 0; fx < FRAMES_PER_DESC; fx++) {
 		urb->iso_frame_desc[fx].offset = FRAME_SIZE_PER_DESC * fx;
		urb->iso_frame_desc[fx].length = FRAME_SIZE_PER_DESC;
	}

	wintv->sbuf[1].urb->next = wintv->sbuf[0].urb;
	wintv->sbuf[0].urb->next = wintv->sbuf[1].urb;

	err = usb_submit_urb(wintv->sbuf[0].urb);
	if (err)
		err("wintv_init_isoc: usb_submit_urb(0) ret %d", err);
	err = usb_submit_urb(wintv->sbuf[1].urb);
	if (err)
		err("wintv_init_isoc: usb_submit_urb(1) ret %d", err);

	wintv->streaming = 1;

	return 0;
}


static void wintv_stop_isoc(struct usb_wintv *wintv)
{
	if (!wintv->streaming || !wintv->dev)
		return;

	wintv_set_packet_size(wintv, 0);

	wintv->streaming = 0;

	/* Unschedule all of the iso td's */
	if (wintv->sbuf[1].urb) {
		wintv->sbuf[1].urb->next = NULL;
		usb_unlink_urb(wintv->sbuf[1].urb);
		usb_free_urb(wintv->sbuf[1].urb);
		wintv->sbuf[1].urb = NULL;
	}
	if (wintv->sbuf[0].urb) {
		wintv->sbuf[0].urb->next = NULL;
		usb_unlink_urb(wintv->sbuf[0].urb);
		usb_free_urb(wintv->sbuf[0].urb);
		wintv->sbuf[0].urb = NULL;
	}
}

static int wintv_new_frame(struct usb_wintv *wintv, int framenum)
{
#if 1
	struct wintv_frame *frame;
	int width, height;

	if (!wintv->dev)
		return -1;

	/* If we're not grabbing a frame right now and the other frame is */
	/*  ready to be grabbed into, then use it instead */
	if (wintv->curframe == -1) {
		if (wintv->frame[(framenum - 1 + WINTV_NUMFRAMES) % WINTV_NUMFRAMES].grabstate == FRAME_READY)
			framenum = (framenum - 1 + WINTV_NUMFRAMES) % WINTV_NUMFRAMES;
	} else
		return 0;

	frame = &wintv->frame[framenum];
	width = frame->width;
	height = frame->height;

	frame->grabstate = FRAME_GRABBING;
	frame->scanstate = STATE_SCANNING;
	frame->scanlength = 0;		/* accumulated in wintv_parse_data() */

	wintv->curframe = framenum;

	/* Make sure it's not too big */
	if (width > DEFAULT_WIDTH)
		width = DEFAULT_WIDTH;
	width = (width / 8) * 8;	/* Multiple of 8 */

	if (height > DEFAULT_HEIGHT)
		height = DEFAULT_HEIGHT;
	height = (height / 4) * 4;	/* Multiple of 4 */

//	/* We want a fresh frame every 30 we get */
//	wintv->compress = (wintv->compress + 1) % 30;

#endif
	return 0;
}


/* Video 4 Linux API */
static int wintv_open(struct video_device *dev, int flags)
{
	int err = -EBUSY;
	struct usb_wintv *wintv = (struct usb_wintv *)dev;

	PDEBUG("wintv_open");

	down(&wintv->lock);
	if (wintv->user)
		goto out_unlock;

	wintv->frame[0].grabstate = FRAME_UNUSED;
	wintv->frame[1].grabstate = FRAME_UNUSED;

	err = -ENOMEM;

	/* Allocate memory for the frame buffers */
	wintv->fbuf = rvmalloc(2 * MAX_DATA_SIZE);
	if (!wintv->fbuf)
		goto open_err_ret;

	wintv->frame[0].data = wintv->fbuf;
	wintv->frame[1].data = wintv->fbuf + MAX_DATA_SIZE;
	wintv->sub_flag = 0;

	PDEBUG("frame [0] @ %p", wintv->frame[0].data);
	PDEBUG("frame [1] @ %p", wintv->frame[1].data);

	wintv->sbuf[0].data = kmalloc(FRAMES_PER_DESC * FRAME_SIZE_PER_DESC, GFP_KERNEL);
	if (!wintv->sbuf[0].data)
		goto open_err_on0;
	wintv->sbuf[1].data = kmalloc(FRAMES_PER_DESC * FRAME_SIZE_PER_DESC, GFP_KERNEL);
	if (!wintv->sbuf[1].data)
		goto open_err_on1;
		
	PDEBUG("sbuf[0] @ %p", wintv->sbuf[0].data);
	PDEBUG("sbuf[1] @ %p", wintv->sbuf[1].data);

	err = wintv_init_isoc(wintv);
	if (err)
		goto open_err_on2;

	wintv->user++;
	up(&wintv->lock);

	MOD_INC_USE_COUNT;

	return 0;

open_err_on2:
	kfree (wintv->sbuf[1].data);
open_err_on1:
	kfree (wintv->sbuf[0].data);
open_err_on0:
	rvfree(wintv->fbuf, 2 * MAX_DATA_SIZE);
open_err_ret:
	return err;
out_unlock:
	up(&wintv->lock);
	return err;

}

static void wintv_close(struct video_device *dev)
{
	struct usb_wintv *wintv = (struct usb_wintv *)dev;

	PDEBUG("wintv_close");
	
	down(&wintv->lock);	
	wintv->user--;

	MOD_DEC_USE_COUNT;

	wintv_stop_isoc(wintv);

	rvfree(wintv->fbuf, 2 * MAX_DATA_SIZE);

	kfree(wintv->sbuf[1].data);
	kfree(wintv->sbuf[0].data);

	up(&wintv->lock);

	if (!wintv->dev) {
		video_unregister_device(&wintv->vdev);
		kfree(wintv);
	}
}

static int wintv_init_done(struct video_device *dev)
{
	return 0;
}

static long wintv_write(struct video_device *dev, const char *buf, unsigned long count, int noblock)
{
	return -EINVAL;
}

static int wintv_ioctl(struct video_device *vdev, unsigned int cmd, void *arg)
{
	struct usb_wintv *wintv = (struct usb_wintv *)vdev;
#if 0	
	PDEBUG("IOCtl: 0x%X", cmd);
#endif	

	if (!wintv->dev)
		return -EIO;	

	switch (cmd) {
		case VIDIOCGCAP:
		{
			struct video_capability b;

			strcpy(b.name, "WINTV USB Camera");
			b.type = VID_TYPE_CAPTURE | VID_TYPE_SUBCAPTURE;
			b.channels = 1;
			b.audios = 0;
			b.maxwidth = DEFAULT_WIDTH;
			b.maxheight = DEFAULT_HEIGHT;
			b.minwidth = 32;
			b.minheight = 16;

			if (copy_to_user(arg, &b, sizeof(b)))
				return -EFAULT;
				
			return 0;
		}
		case VIDIOCGCHAN:
		{
			struct video_channel v;

			if (copy_from_user(&v, arg, sizeof(v)))
				return -EFAULT;
			if (v.channel != 0)
				return -EINVAL;

			v.flags = 0;
			v.tuners = 0;
			v.type = VIDEO_TYPE_CAMERA;
			strcpy(v.name, "Camera");

			if (copy_to_user(arg, &v, sizeof(v)))
				return -EFAULT;
				
			return 0;
		}
		case VIDIOCSCHAN:
		{
			int v;

			if (copy_from_user(&v, arg, sizeof(v)))
				return -EFAULT;

			if (v != 0)
				return -EINVAL;

			return 0;
		}

		case VIDIOCGPICT:
		{
			struct video_picture p;

			if (ov7610_get_picture(wintv, &p))
				return -EIO;
							
			if (copy_to_user(arg, &p, sizeof(p)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCSPICT:
		{
			struct video_picture p;

			if (copy_from_user(&p, arg, sizeof(p)))
				return -EFAULT;
			
			if (ov7610_set_picture(wintv, &p))
				return -EIO;

			return 0;
		}
		case VIDIOCGCAPTURE:
		{
			int vf;
			if (copy_from_user(&vf, arg, sizeof(vf)))
				return -EFAULT;
			wintv->sub_flag = vf;
			return 0;
		}
		case VIDIOCSCAPTURE:
		{
			struct video_capture vc;

			if (copy_from_user(&vc, arg, sizeof(vc)))
				return -EFAULT;
			if (vc.flags)
				return -EINVAL;
			if (vc.decimation)
				return -EINVAL;
			vc.x /= 4;
			vc.x *= 4;
			vc.y /= 2;
			vc.y *= 2;
			vc.width /= 32;
			vc.width *= 32;
			if (vc.width == 0) vc.width = 32;
			vc.height /= 16;
			vc.height *= 16;
			if (vc.height == 0) vc.height = 16;

			wintv->subx = vc.x;
			wintv->suby = vc.y;
			wintv->subw = vc.width;
			wintv->subh = vc.height;

			return 0;
		}
		case VIDIOCSWIN:
		{
			struct video_window vw;

			if (copy_from_user(&vw, arg, sizeof(vw)))
				return -EFAULT;
			if (vw.flags)
				return -EINVAL;
			if (vw.clipcount)
				return -EINVAL;
			if (vw.height != DEFAULT_HEIGHT)
				return -EINVAL;
			if (vw.width != DEFAULT_WIDTH)
				return -EINVAL;

			wintv->compress = 0;

			return 0;
		}
		case VIDIOCGWIN:
		{
			struct video_window vw;

			vw.x = 0;
			vw.y = 0;
			vw.width = DEFAULT_WIDTH;
			vw.height = DEFAULT_HEIGHT;
			vw.chromakey = 0;
			vw.flags = 30;

			if (copy_to_user(arg, &vw, sizeof(vw)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCGMBUF:
		{
			struct video_mbuf vm;

			memset(&vm, 0, sizeof(vm));
			vm.size = 2 * MAX_DATA_SIZE;
			vm.frames = 2;
			vm.offsets[0] = 0;
			vm.offsets[1] = MAX_FRAME_SIZE + sizeof (struct timeval);

			if (copy_to_user((void *)arg, (void *)&vm, sizeof(vm)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCMCAPTURE:
		{
			struct video_mmap vm;

			if (copy_from_user((void *)&vm, (void *)arg, sizeof(vm)))
				return -EFAULT;

#if 0
			PDEBUG("MCAPTURE");
			PDEBUG("frame: %d, size: %dx%d, format: %d",
				vm.frame, vm.width, vm.height, vm.format);
#endif

			if (vm.format != VIDEO_PALETTE_RGB24 &&
			    vm.format != VIDEO_PALETTE_GREY)
				return -EINVAL;

			if ((vm.frame != 0) && (vm.frame != 1))
				return -EINVAL;
				
			if (wintv->frame[vm.frame].grabstate == FRAME_GRABBING)
				return -EBUSY;

			/* Don't compress if the size changed */
			if ((wintv->frame[vm.frame].width != vm.width) ||
			    (wintv->frame[vm.frame].height != vm.height) ||
			    (wintv->frame[vm.frame].format != vm.format) ||
			    (wintv->frame[vm.frame].sub_flag !=
			     wintv->sub_flag)) {
				/* If we're collecting previous frame wait
				   before changing modes */
				interruptible_sleep_on(&wintv->wq);
				if (signal_pending(current)) return -EINTR;
				wintv_mode_init_regs(wintv,
						     vm.width, vm.height,
						     vm.format, wintv->sub_flag);
			}

			wintv->frame[vm.frame].width = vm.width;
			wintv->frame[vm.frame].height = vm.height;
			wintv->frame[vm.frame].format = vm.format;
			wintv->frame[vm.frame].sub_flag = wintv->sub_flag;
			wintv->frame[vm.frame].segsize =
			  vm.format == VIDEO_PALETTE_RGB24 ? 384 : 256;
			wintv->frame[vm.frame].depth =
			  vm.format == VIDEO_PALETTE_RGB24 ? 3 : 1;

			/* Mark it as ready */
			wintv->frame[vm.frame].grabstate = FRAME_READY;

			return wintv_new_frame(wintv, vm.frame);
		}
		case VIDIOCSYNC:
		{
			int frame;

			if (copy_from_user((void *)&frame, arg, sizeof(int)))
				return -EFAULT;

#if 0
			PDEBUG("syncing to frame %d, grabstate = %d", frame,
			       wintv->frame[frame].grabstate);
#endif
			switch (wintv->frame[frame].grabstate) {
				case FRAME_UNUSED:
					return -EINVAL;
				case FRAME_READY:
				case FRAME_GRABBING:
				case FRAME_ERROR:
redo:
				if (!wintv->dev)
					return -EIO;

				do {
#if 0
					init_waitqueue_head(&wintv->frame[frame].wq);
#endif
					interruptible_sleep_on(&wintv->frame[frame].wq);
					if (signal_pending(current))
						return -EINTR;
				} while (wintv->frame[frame].grabstate == FRAME_GRABBING);

				if (wintv->frame[frame].grabstate == FRAME_ERROR) {
					int ret;

					if ((ret = wintv_new_frame(wintv, frame)) < 0)
						return ret;
					goto redo;
				}				
				case FRAME_DONE:
					wintv->frame[frame].grabstate = FRAME_UNUSED;
					break;
			}

			wintv->frame[frame].grabstate = FRAME_UNUSED;
			
			return 0;
		}
		case VIDIOCGFBUF:
		{
			struct video_buffer vb;

			memset(&vb, 0, sizeof(vb));
			vb.base = NULL;	/* frame buffer not supported, not used */

			if (copy_to_user((void *)arg, (void *)&vb, sizeof(vb)))
				return -EFAULT;

 			return 0;
 		}
		case VIDIOCKEY:
			return 0; 		
		case VIDIOCCAPTURE:
			return -EINVAL;
		case VIDIOCSFBUF:
			return -EINVAL;
		case VIDIOCGTUNER:
		case VIDIOCSTUNER:
			return -EINVAL;			
		case VIDIOCGFREQ:
		case VIDIOCSFREQ:
			return -EINVAL;
		case VIDIOCGAUDIO:
		case VIDIOCSAUDIO:
			return -EINVAL;
		default:
			return -ENOIOCTLCMD;
	}
	return 0;
}

static long wintv_read(struct video_device *dev, char *buf, unsigned long count, int noblock)
{
	struct usb_wintv *wintv = (struct usb_wintv *)dev;
	int frmx = -1;
	volatile struct wintv_frame *frame;

	PDEBUG("wintv_read: %ld bytes, noblock=%d", count, noblock);

	if (!dev || !buf)
		return -EFAULT;

	if (!wintv->dev)
		return -EIO;

	/* See if a frame is completed, then use it. */
	if (wintv->frame[0].grabstate >= FRAME_DONE)	/* _DONE or _ERROR */
		frmx = 0;
	else if (wintv->frame[1].grabstate >= FRAME_DONE)/* _DONE or _ERROR */
		frmx = 1;

	if (noblock && (frmx == -1))
		return -EAGAIN;

	/* If no FRAME_DONE, look for a FRAME_GRABBING state. */
	/* See if a frame is in process (grabbing), then use it. */
	if (frmx == -1) {
		if (wintv->frame[0].grabstate == FRAME_GRABBING)
			frmx = 0;
		else if (wintv->frame[1].grabstate == FRAME_GRABBING)
			frmx = 1;
	}

	/* If no frame is active, start one. */
	if (frmx == -1)
		wintv_new_frame(wintv, frmx = 0);

	frame = &wintv->frame[frmx];

restart:
	if (!wintv->dev)
		return -EIO;

	while (frame->grabstate == FRAME_GRABBING) {
		interruptible_sleep_on(&wintv->frame[frmx].wq);
		if (signal_pending(current))
			return -EINTR;
	}

	if (frame->grabstate == FRAME_ERROR) {
		frame->bytes_read = 0;
		err("wintv_read: errored frame %d", wintv->curframe);
		if (wintv_new_frame(wintv, frmx))
			err("wintv_read: wintv_new_frame error");
		goto restart;
	}

	PDEBUG("wintv_read: frmx=%d, bytes_read=%ld, scanlength=%ld", frmx,
		frame->bytes_read, frame->scanlength);

	/* copy bytes to user space; we allow for partials reads */
	if ((count + frame->bytes_read) > frame->scanlength)
		count = frame->scanlength - frame->bytes_read;

	if (copy_to_user(buf, frame->data + frame->bytes_read, count))
		return -EFAULT;

	frame->bytes_read += count;
	PDEBUG("wintv_read: {copy} count used=%ld, new bytes_read=%ld",
		count, frame->bytes_read);

	if (frame->bytes_read >= frame->scanlength) { /* All data has been read */
		frame->bytes_read = 0;

		/* Mark it as available to be used again. */
		wintv->frame[frmx].grabstate = FRAME_UNUSED;
		if (wintv_new_frame(wintv, frmx ? 0 : 1))
			err("wintv_read: wintv_new_frame returned error");
	}

	return count;
}

static int wintv_mmap(struct video_device *dev, const char *adr, unsigned long size)
{
	struct usb_wintv *wintv = (struct usb_wintv *)dev;
	unsigned long start = (unsigned long)adr;
	unsigned long page, pos;

	if (!wintv->dev)
		return -EIO;

	PDEBUG("mmap: %ld (%lX) bytes", size, size);

	if (size > (((2 * MAX_DATA_SIZE) + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1)))
		return -EINVAL;

	pos = (unsigned long)wintv->fbuf;
	while (size > 0)
	{
		page = kvirt_to_pa(pos);
		if (remap_page_range(start, page, PAGE_SIZE, PAGE_SHARED))
			return -EAGAIN;
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return 0;
}

static struct video_device wintv_template = {
	"WINTV USB",
	VID_TYPE_CAPTURE,
	VID_HARDWARE_WINTV,
	wintv_open,
	wintv_close,
	wintv_read,
	wintv_write,
	NULL,
	wintv_ioctl,
	wintv_mmap,
	wintv_init_done,
	NULL,
	0,
	0
};

static int saa7111a_configure(struct usb_device *dev)
{
	int rc;
	
	if(wintv_reg_write(dev, WINTV_REG_SER_MODE, WINTV_BIT_SER_MODE) < 0)
		return -1;

	if (wintv_i2c_write(dev, WINTV_I2C_CHIP_VERS, 0x00) < 0) return -1;
	if ((rc=wintv_i2c_read(dev, WINTV_I2C_CHIP_VERS)) < 0) return -1;
	if (wintv_i2c_write(dev, WINTV_I2C_CHIP_VERS, 0x00) < 0) return -1;
	if ((rc=wintv_i2c_read(dev, WINTV_I2C_CHIP_VERS)) < 0) return -1;
	else if ((rc & 0x20) || (rc & 0x10)) 
		printk(KERN_INFO "wintv: SAA7111A Version %d\n", (rc & 0x30)>>4);
	else {
		err("SAA7111A wrong Version"); 
		return -1;
	}
	

	
	schedule_timeout (1 + 150 * HZ / 1000);

	return 0;
}

static int wintv_configure(struct usb_wintv *wintv)
{
	struct usb_device *dev = wintv->dev;
	int rc;

	static struct wintv_regvals aRegvalsInit[] =
	{{WINTV_REG_BUS,  WINTV_REG_SYSTEM_RESET, 0x7f},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_INIT, 0x01},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_RESET, 0x7f},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_INIT, 0x01},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_RESET, 0x3f},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_INIT, 0x01},
	 {WINTV_REG_BUS,  WINTV_REG_SYSTEM_RESET, 0x3d},
	 {WINTV_DONE_BUS, 0x0, 0x00},
	};
	static struct wintv_regvals aRegvalsNorm[] =
	{{WINTV_REG_BUS, WINTV_REG_LXSIZE_O,  0x60},/* 352 x 288 */
	 {WINTV_REG_BUS, WINTV_REG_MXSIZE_O,  0x01},
	 {WINTV_REG_BUS, WINTV_REG_LYSIZE_O,  0x20},
	 {WINTV_REG_BUS, WINTV_REG_MYSIZE_O,  0x01},
	 {WINTV_REG_BUS, WINTV_REG_VO_MODE,   0x14},/* YUV420, raw */
	 {WINTV_REG_BUS, WINTV_REG_VIN_REG1,  0x03},/* YUV422, 16-bit, CCIR656 sync. */
	 /* PAL */
	 {WINTV_REG_BUS, WINTV_REG_LXSIZE_IN, 0xD0},
	 {WINTV_REG_BUS, WINTV_REG_MXSIZE_IN, 0x02},/* 720 x 625 */
	 {WINTV_REG_BUS, WINTV_REG_LYSIZE_IN, 0x71},
	 {WINTV_REG_BUS, WINTV_REG_MYSIZE_IN, 0x02},
	 {WINTV_REG_BUS, WINTV_REG_FRM_RATE,  0x04},/* Frame rate: 25*(n+1)/32 = 3.90625 */ 
#if 0
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
	 {WINTV_I2C_BUS, 0, 0},
#endif
	 {WINTV_DONE_BUS, 0x0, 0x00},
	};

#if 1 
	/* Set altsetting 0 */
	if (usb_set_interface(dev, wintv->iface, 0) < 0) {
		err("usb_set_interface error");
		return -EBUSY;
	}
#endif
#if 1	
	memcpy(&wintv->vdev, &wintv_template, sizeof(wintv_template));

	init_waitqueue_head(&wintv->frame[0].wq);
	init_waitqueue_head(&wintv->frame[1].wq);
	init_waitqueue_head(&wintv->wq);

	if (video_register_device(&wintv->vdev, VFL_TYPE_GRABBER) == -1) {
		err("video_register_device failed");
		return -EBUSY;
	}
#endif
	/* Power up Video-Hardware */
	if ((rc = wintv_reg_write(dev, WINTV_REG_PWR, 0x00 )) < 0)
		return rc;

	if ((rc = wintv_reg_write(dev, WINTV_REG_PWR, 
		WINTV_BIT_PWR_VID | WINTV_BIT_PWR_RES2 )) < 0)
		return rc;

	/* Configure SAA7111A */
	if(saa7111a_configure(dev) < 0) {
	  err("failed to configure SAA7111A");
	  goto error;	
	}

	wintv->compress = 0;
	
	/* Set default sizes in case IOCTL (VIDIOCMCAPTURE) is not used
	 * (using read() instead). */
	wintv->frame[0].width = DEFAULT_WIDTH;
	wintv->frame[0].height = DEFAULT_HEIGHT;
	wintv->frame[0].bytes_read = 0;
	wintv->frame[1].width = DEFAULT_WIDTH;
	wintv->frame[1].height = DEFAULT_HEIGHT;
	wintv->frame[1].bytes_read = 0;

	/* Initialize to DEFAULT_WIDTH, DEFAULT_HEIGHT, YUV4:2:0, raw */
	if ((rc = wintv_write_regvals(dev, aRegvalsNorm))) return rc;
	/*	if ((rc = wintv_mode_init_regs(wintv, DEFAULT_WIDTH, DEFAULT_HEIGHT,
		VIDEO_PALETTE_RGB24, 0)) < 0) return rc;*/


	return 0;
	
error:
	video_unregister_device(&wintv->vdev);
	usb_driver_release_interface(&wintv_driver,
		&dev->actconfig->interface[wintv->iface]);

	kfree(wintv);

	return -EBUSY;	
}

static void* wintv_probe(struct usb_device *dev, unsigned int ifnum)
{
	struct usb_interface_descriptor *interface;
	struct usb_wintv *wintv;
	int rc;

	PDEBUG("probing for device...");

	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1)
		return NULL;

	interface = &dev->actconfig->interface[ifnum].altsetting[0];

	/* Is it an WINTV? */
	if (dev->descriptor.idVendor != 0x0573)
		return NULL;
	if (dev->descriptor.idProduct != 0x4d11)
		return NULL;

	/* Checking vendor/product should be enough, but what the hell */
	if (interface->bInterfaceClass != 0x00) 
		return NULL;
	if (interface->bInterfaceSubClass != 0x00)
		return NULL;

	/* We found one */
	printk(KERN_INFO "wintv: HAUPPAUGE WINTV-USB box found\n");

	if ((wintv = kmalloc(sizeof(*wintv), GFP_KERNEL)) == NULL) {
		err("couldn't kmalloc wintv struct");
		return NULL;
	}

	memset(wintv, 0, sizeof(*wintv));
	
	wintv->dev = dev;
	wintv->iface = interface->bInterfaceNumber;

/*	rc = wintv_reg_read(dev, WINTV_REG_SYSTEM_CUSTOM_ID);
	if (rc < 0) {
		err("Unable to read camera bridge registers");
		return NULL;
	}
	
	switch(wintv->customid = rc) {
	case 0: * This also means that no custom ID was set *
		printk("wintv: Camera is probably a MediaForte MV300\n");
		break;
	case 3:
		printk("wintv: Camera is a D-Link DSB-C300\n");
		break;
	case 5:
		printk("wintv: Camera is a Puretek PT-6007\n");
		break;
	case 21:
		printk("wintv: Camera is a Creative Labs WebCam 3\n");
		break;
	case 100:
		printk("wintv: Camera is a Lifeview RoboCam\n");
		break;
	case 102:
		printk("wintv: Camera is a AverMedia InterCam Elite\n");
		break;
	case 112:
		printk("wintv: Camera is a MediaForte MV300\n");
		break;
	default:
		err("Specific camera type (%d) not recognized", rc);
		err("Please contact mmcclelland@delphi.com to request");
		err("support for your camera.");
		return NULL;
	}
*/
	if (!wintv_configure(wintv)) {
		wintv->user=0;
		init_MUTEX(&wintv->lock);	/* to 1 == available */
		return wintv;
	}
	else {
		err("Failed to configure wintv box");
		return NULL;
	}
    	
     	return wintv;
}

static void wintv_disconnect(struct usb_device *dev, void *ptr)
{

	struct usb_wintv *wintv = (struct usb_wintv *) ptr;

//	video_unregister_device(&wintv->vdev);

	/* We don't want people trying to open up the device */
	if (!wintv->user)
		video_unregister_device(&wintv->vdev);

	usb_driver_release_interface(&wintv_driver,
		&wintv->dev->actconfig->interface[wintv->iface]);

	wintv->dev = NULL;
	wintv->frame[0].grabstate = FRAME_ERROR;
	wintv->frame[1].grabstate = FRAME_ERROR;
	wintv->curframe = -1;

	/* This will cause the process to request another frame */
	if (waitqueue_active(&wintv->frame[0].wq))
		wake_up_interruptible(&wintv->frame[0].wq);
	if (waitqueue_active(&wintv->frame[1].wq))
		wake_up_interruptible(&wintv->frame[1].wq);
	if (waitqueue_active(&wintv->wq))
		wake_up_interruptible(&wintv->wq);

	wintv->streaming = 0;

	/* Unschedule all of the iso td's */
	if (wintv->sbuf[1].urb) {
		wintv->sbuf[1].urb->next = NULL;
		usb_unlink_urb(wintv->sbuf[1].urb);
		usb_free_urb(wintv->sbuf[1].urb);
		wintv->sbuf[1].urb = NULL;
	}
	if (wintv->sbuf[0].urb) {
		wintv->sbuf[0].urb->next = NULL;
		usb_unlink_urb(wintv->sbuf[0].urb);
		usb_free_urb(wintv->sbuf[0].urb);
		wintv->sbuf[0].urb = NULL;
	}	

	/* Free the memory */
	if (!wintv->user) {
		kfree(wintv);
		wintv = NULL;
	}
}

static struct usb_driver wintv_driver = {
	"wintv",
	wintv_probe,
	wintv_disconnect,
	{ NULL, NULL }
};

int usb_wintv_init(void)
{
	PDEBUG("usb_wintv_init()");
	
	EXPORT_NO_SYMBOLS;
	
	return usb_register(&wintv_driver);
}

void usb_wintv_cleanup(void)
{
	usb_deregister(&wintv_driver);
}

#ifdef MODULE
int init_module(void)
{
	return usb_wintv_init();
}

void cleanup_module(void)
{
	usb_wintv_cleanup();
	
	PDEBUG("Module unloaded");
}
#endif

