/*
 * USB NT1003 Video device driver
 *
 *
 * This driver is based on USB IBM C-It Video Camera driver by
 * Johannes Erdfelt & Randy Dunlap
 *
 * Ideas taken from bttv driver by Ralph Metzler, Marcus Metzler &
 * Gerd Knorr and zoran 36120/36125 driver by Pauline Middelink
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/smp_lock.h>
#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <linux/wrapper.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/usb.h>

#include <asm/io.h>

#include <linux/video_decoder.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-usb.h>
#include "../char/tuner.h"	// Arhggggggggggggg !!!!!!!!!!!

#include "nt1003.h"

/*
 * NT1003_LOCKS_DRIVER_WHILE_DEVICE_IS_PLUGGED: This symbol controls
 * the locking of the driver. If non-zero, the driver counts the
 * probe() call as usage and increments module usage counter; this
 * effectively prevents removal of the module (with rmmod) until the
 * device is unplugged (then disconnect() callback reduces the module
 * usage counter back, and module can be removed).
 *
 * This behavior may be useful if you prefer to lock the driver in
 * memory until device is unplugged. However you can't reload the
 * driver if you want to alter some parameters - you'd need to unplug
 * the camera first. Therefore, I recommend setting 0.
 */
#define NT1003_LOCKS_DRIVER_WHILE_DEVICE_IS_PLUGGED	0

#define	ENABLE_HEXDUMP	1	/* Enable if you need it */
static int debug = 0;
static int verbose = 1;

/* Completion states of the data parser */
typedef enum {
	scan_Continue,		/* Just parse next item */
	scan_NextFrame,		/* Frame done, send it to V4L */
	scan_Out,		/* Not enough data for frame */
	scan_EndParse		/* End parsing */
} scan_state_t;

/* Bit flags (options) */
#define FLAGS_RETRY_VIDIOCSYNC		(1 << 0)
#define	FLAGS_MONOCHROME		(1 << 1)
#define FLAGS_DISPLAY_HINTS		(1 << 2)
#define FLAGS_OVERLAY_STATS		(1 << 3)
#define FLAGS_FORCE_TESTPATTERN		(1 << 4)
#define FLAGS_SEPARATE_FRAMES		(1 << 5)
#define FLAGS_CLEAN_FRAMES		(1 << 6)

static int flags = 0; /* FLAGS_DISPLAY_HINTS | FLAGS_OVERLAY_STATS; */

static int imgwidth = 0;
static int imgheight = 0;
static const int max_imgwidth  = MAX_FRAME_WIDTH;
static const int max_imgheight = MAX_FRAME_HEIGHT;
static const int min_imgwidth  = 8;
static const int min_imgheight = 4;

#define FRAMERATE_MIN	0
#define FRAMERATE_MAX	31

enum {
	VIDEOFORMAT_YUV422 = 0x03,
	VIDEOFORMAT_YUV420 = 0x14,
	VIDEOFORMAT_COMPRESS = 0x60,
};


/*
 * The value of 'scratchbufsize' affects quality of the picture
 * in many ways. Shorter buffers may cause loss of data when client
 * is too slow. Larger buffers are memory-consuming and take longer
 * to work with. This setting can be adjusted, but the default value
 * should be OK for most desktop users.
 */
#define DEFAULT_SCRATCH_BUF_SIZE	(0x20000) /* 128 k */
static const int scratchbufsize = DEFAULT_SCRATCH_BUF_SIZE;

static int init_brightness = 128;
static int init_contrast = 192;
static int init_color = 128;
static int init_hue = 128;
static int hue_correction = 128;

static int next_uv = 0;

static int nt1003_restart_isoc(struct usb_nt1003 *nt1003);
static int nt1003_begin_streaming(struct usb_nt1003 *nt1003);
static int nt1003_muxsel(struct usb_nt1003 *nt1003, int channel, int norm);
static int nt1003_i2c_write(void *data, unsigned char addr, char *buf, short len);
static int nt1003_i2c_read(void *data, unsigned char addr, char *buf, short len);

static int tuner = 5;
static int norm = 0;
static int videoformat = VIDEOFORMAT_YUV422;

MODULE_PARM(debug, "i");
MODULE_PARM(flags, "i");
MODULE_PARM(tuner, "i");
MODULE_PARM(norm, "i");
MODULE_PARM(videoformat, "i");

MODULE_AUTHOR ("module author");
MODULE_DESCRIPTION ("NT1003 USB Video device Driver for Linux");

#define MAX_NT1003	4

struct usb_nt1003 vdevs[MAX_NT1003];

/*******************************/
/* Memory management functions */
/*******************************/

#define MDEBUG(x)	do { } while(0)		/* Debug memory management */

static struct usb_driver nt1003_driver;
static void nt1003_release(struct usb_nt1003 *nt1003);

/* Given PGD from the address space's page table, return the kernel
 * virtual mapping of the physical memory mapped at ADR.
 */
static inline unsigned long uvirt_to_kva(pgd_t *pgd, unsigned long adr)
{
	unsigned long ret = 0UL;
	pmd_t *pmd;
	pte_t *ptep, pte;

	if (!pgd_none(*pgd)) {
		pmd = pmd_offset(pgd, adr);
		if (!pmd_none(*pmd)) {
			ptep = pte_offset(pmd, adr);
			pte = *ptep;
			if (pte_present(pte))
				ret = page_address(pte_page(pte)) | (adr & (PAGE_SIZE-1));
		}
	}
	MDEBUG(printk("uv2kva(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long uvirt_to_bus(unsigned long adr)
{
	unsigned long kva, ret;

	kva = uvirt_to_kva(pgd_offset(current->mm, adr), adr);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("uv2b(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long kvirt_to_bus(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("kv2b(%lx-->%lx)", adr, ret));
	return ret;
}

/* Here we want the physical address of the memory.
 * This is used when initializing the contents of the
 * area and marking the pages as reserved.
 */
static inline unsigned long kvirt_to_pa(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = __pa(kva);
	MDEBUG(printk("kv2pa(%lx-->%lx)", adr, ret));
	return ret;
}

static void *rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr, page;

	/* Round it off to PAGE_SIZE */
	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	mem = vmalloc(size);
	if (!mem)
		return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_reserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return mem;
}

static void rvfree(void *mem, unsigned long size)
{
	unsigned long adr, page;

	if (!mem)
		return;

	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	adr=(unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_unreserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}
	vfree(mem);
}

#if ENABLE_HEXDUMP
static void nt1003_hexdump(const unsigned char *data, int len)
{
	char tmp[80];
	int i, k;

	for (i=k=0; len > 0; i++, len--) {
		if (i > 0 && (i%16 == 0)) {
			printk("%s\n", tmp);
			k=0;
		}
		k += sprintf(&tmp[k], "%02x ", data[i]);
	}
	if (k > 0)
		printk("%s\n", tmp);
}
#endif

/*
 * usb_nt1003_overlaychar()
 *
 */
void
nt1003_overlaychar(
	struct usb_nt1003 *nt1003,
	struct nt1003_frame *frame,
	int x, int y, int ch)
{
	static const unsigned short digits[16] = {
		0xF6DE, /* 0 */
		0x2492, /* 1 */
		0xE7CE, /* 2 */
		0xE79E, /* 3 */
		0xB792, /* 4 */
		0xF39E, /* 5 */
		0xF3DE, /* 6 */
		0xF492, /* 7 */
		0xF7DE, /* 8 */
		0xF79E, /* 9 */
		0x77DA, /* a */
		0xD75C, /* b */
		0xF24E, /* c */
		0xD6DC, /* d */
		0xF34E, /* e */
		0xF348  /* f */
	};
	unsigned short digit;
	int ix, iy;

	if ((nt1003 == NULL) || (frame == NULL))
		return;

	if (ch >= '0' && ch <= '9')
		ch -= '0';
	else if (ch >= 'A' && ch <= 'F')
		ch = 10 + (ch - 'A');
	else if (ch >= 'a' && ch <= 'f')
		ch = 10 + (ch - 'a');
	else
		return;
	digit = digits[ch];

	for (iy=0; iy < 5; iy++) {
		for (ix=0; ix < 3; ix++) {
			if (digit & 0x8000) {
				NT1003_PUTPIXEL(frame, x+ix, y+iy, 0xFF, 0xFF,
						0xFF);
			}
			digit = digit << 1;
		}
	}
}

/*
 * nt1003_overlaystring()
 *
 */
void
nt1003_overlaystring(
	struct usb_nt1003 *nt1003,
	struct nt1003_frame *frame,
	int x, int y, const char *str)
{
	while (*str) {
		nt1003_overlaychar(nt1003, frame, x, y, *str);
		str++;
		x += 4; /* 3 pixels character + 1 space */
	}
}

/*
 * usb_nt1003_overlaystats()
 *
 * Overlays important debugging information.
 *
 */
void 
nt1003_overlaystats(struct usb_nt1003 *nt1003, struct nt1003_frame *frame)
{
	const int y_diff = 8;
	char tmp[16];
	int x = 10;
	int y = 10;

	sprintf(tmp, "%8x", nt1003->frame_num);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->urb_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->urb_length);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->data_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->header_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->scratch_ovf_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->iso_skip_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8lx", nt1003->iso_err_count);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8x", nt1003->vpic.colour);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8x", nt1003->vpic.hue);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8x", nt1003->vpic.brightness >> 8);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8x", nt1003->vpic.contrast >> 12);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;

	sprintf(tmp, "%8d", nt1003->vpic.whiteness >> 8);
	nt1003_overlaystring(nt1003, frame, x, y, tmp);
	y += y_diff;
}

/*
 * nt1003_testpattern()
 *
 * Procedure forms a test pattern (yellow grid on blue background).
 *
 * Parameters:
 * fullframe:   if TRUE then entire frame is filled, otherwise the procedure
 *	        continues from the current scanline.
 * pmode	0: fill the frame with solid blue color (like on VCR or TV)
 *	        1: Draw a colored grid
 *
 */
void 
nt1003_testpattern(struct usb_nt1003 *nt1003, int fullframe, int pmode)
{
	static const char proc[] = "nt1003_testpattern";
	struct nt1003_frame *frame;
	unsigned char *f;
	int num_cell = 0;
	int scan_length = 0;
	static int num_pass = 0;

	if (nt1003 == NULL) {
		printk(KERN_ERR "%s: nt1003 == NULL\n", proc);
		return;
	}
	if ((nt1003->curframe < 0) || (nt1003->curframe >= NT1003_NUMFRAMES)) {
		printk(KERN_ERR "%s: nt1003->curframe=%d.\n", 
				proc, nt1003->curframe);
		return;
	}

	/* Grab the current frame */
	frame = &nt1003->frame[nt1003->curframe];

	/* Optionally start at the beginning */
	if (fullframe) {
		frame->curline = 0;
		frame->scanlength = 0;
	}

	/* Form every scan line */
	for (; frame->curline < imgheight; frame->curline++) {
		int i;

		f = frame->data + (imgwidth * 3 * frame->curline);
		for (i=0; i < imgwidth; i++) {
			unsigned char cb=0x80;
			unsigned char cg = 0;
			unsigned char cr = 0;

			if (pmode == 1) {
				if (frame->curline % 32 == 0)
					cb = 0, cg = cr = 0xFF;
				else if (i % 32 == 0) {
					if (frame->curline % 32 == 1)
						num_cell++;
					cb = 0, cg = cr = 0xFF;
				} else {
					cb = ((num_cell*7) + num_pass) & 0xFF;
					cg = ((num_cell*5) + num_pass*2) & 0xFF;
					cr = ((num_cell*3) + num_pass*3) & 0xFF;
				}
			} else {
				/* Just the blue screen */
			}
				
			*f++ = cb;
			*f++ = cg;
			*f++ = cr;
			scan_length += 3;
		}
	}

	frame->grabstate = FRAME_DONE;
	frame->scanlength += scan_length;
	++num_pass;

	/* We do this unconditionally, regardless of FLAGS_OVERLAY_STATS */
	nt1003_overlaystats(nt1003, frame);
}

/* How much data is left in the scratch buf? */
#define scratch_left(x)	(nt1003->scratchlen - (int)((char *)x - (char *)nt1003->scratch))

/* Grab the remaining */
static void 
nt1003_align_scratch(struct usb_nt1003 *nt1003, unsigned char *data)
{
	unsigned long left;

	left = scratch_left(data);
	memmove(nt1003->scratch, data, left);
	nt1003->scratchlen = left;
}

/*
 * nt1003_find_header()
 *
 * Locate one of supported header markers in the scratch buffer.
 */
static scan_state_t
nt1003_find_header(struct usb_nt1003 *nt1003)
{
	struct nt1003_frame *frame;
	unsigned char *data;
	int len;

	data = nt1003->scratch;
	frame = &nt1003->frame[nt1003->curframe];

	len = scratch_left(data);

	while (len >= 4){
		if ((data[0] == 0x55)&&(data[1] == 0xAA)&&(data[2] == 0x0c)){
			memcpy(&frame->header, data,
				sizeof(struct nt1003_frame_header));
#undef DUMP_HEADER
#ifdef DUMP_HEADER 
			printk("Header: %02x%02x %d %d %d %d %d %02x %u %u\n",
					frame->header.vid_frm_patt[0],
					frame->header.vid_frm_patt[1],
					frame->header.header_length,
					frame->header.frame_numb,
					frame->header.frame_phase,
					frame->header.frame_latency,
					frame->header.data_format,
					frame->header.format_param,
					frame->header.frame_width,
					frame->header.frame_height);
#endif


			/* Architecture dependant */
			frame->frmwidth = frame->header.frame_width;
			frame->frmheight = frame->header.frame_height;
			if (debug > 2)
				printk(KERN_DEBUG "Header found! w:%d h:%d\n",
						frame->frmwidth, 
						frame->frmheight);

			/* Too late to change if imgwidth != frame_width or
			 * imgheight != frame_height
			 */

			/* FIXME: This should not be hardcoded */
			frame->order_uv = 1;
			frame->order_yc = 1;

			data+=12;
			break;
				
		}
		++data;
		--len;
	}

	if (len < 4) {
		/* No header - entire scratch buffer is useless! */
		if (debug > 2)
			printk(KERN_DEBUG "Skipping frame, no header\n");
		nt1003->scratchlen = 0;
		return scan_EndParse;
	}
	/* Header found */

	nt1003->has_hdr = 1;
	nt1003->header_count++;
	frame->scanstate = STATE_LINES;
	frame->curline = 0;

	if (flags & FLAGS_FORCE_TESTPATTERN) {
		nt1003_testpattern(nt1003, 1, 1);
		return scan_NextFrame;
	}
	nt1003_align_scratch(nt1003, data);
	return scan_Continue;
}

static scan_state_t
nt1003_parse_lines_422(struct usb_nt1003 *nt1003, long *pcopylen)
{
	struct nt1003_frame *frame;
	unsigned char *data, *f;
	unsigned int len;
	const int v4l_linesize = imgwidth * V4L_BYTES_PER_PIXEL;
	int i, y, u, v;
	int frame_done=0;

	data = nt1003->scratch;
	frame = &nt1003->frame[nt1003->curframe];

	/* Make sure there's enough data for the entire line */
	len = (frame->frmwidth * 2)+5;
	if ( scratch_left(data) < len ) {
		if (debug >=2)
			printk(KERN_DEBUG "out of data in line %d, need %u.\n",
				frame->curline, len);
		return scan_Out;
	}

	if ((frame->curline + 1) >= imgheight)
		return scan_NextFrame;

	f = frame->data + (v4l_linesize * frame->curline);

#if 1
	for(i=0; i < frame->frmwidth; i++, data+=2){
		unsigned char rv, gv, bv;       /* RGB components */

		y= data[0];
		if( (next_uv & 1) == 0 ){
			u = data[1];
			v = data[3];
		}else{
			v = data[1];
			u = data[3];
		}
		next_uv++;
		YUV_TO_RGB_BY_THE_BOOK(y, u, v, rv, gv, bv);

		*f++ = bv;
		*f++ = gv;
		*f++ = rv;
	}
#else
	for(i=0; i < frame->frmwidth; i+=1, data+=2){
		/*
		 * Search for potential Start-Of-Frame marker. It should
		 * not be here, of course, but if your formats don't match
		 * you might exceed the frame. We must match the marker to
		 * each byte of multi-byte data element if it is multi-byte.
		 */
		if (scratch_left(data) >= (4)) {
			unsigned char *dp;
			int j;

			for (j=0, dp=data; j < 2; j++, dp++) {
				if ((dp[0] == 0x55) && (dp[1] == 0xaa) &&
				(dp[2] == 0x0c)) {
					nt1003->has_hdr = 2;
					frame_done++;
					printk(KERN_DEBUG
						"Found header inside frame\n");
					break;
				}
			}
		}
		*f++ = data[0];
		*f++ = data[0];
		*f++ = data[0];

		if (frame_done)
			break;
	}



#endif

	frame->curline++;
	*pcopylen += v4l_linesize;
	nt1003_align_scratch(nt1003, data);

	if (frame_done || (frame->curline >= frame->frmheight))
		return scan_NextFrame;
	else
		return scan_Continue;
}

/*
 * nt1003_parse_lines_420()
 *
 * Parse two lines from the scratch buffer, put 
 * decoded RGB value into the current frame buffer and add the written
 * number of bytes (RGB) to the *pcopylen.
 * 
 * FIXME: frame width must be a multiple of 64; frame height must be even
 *        the maximum width is 6*64 (Why?)
 */
static scan_state_t
nt1003_parse_lines_420(struct usb_nt1003 *nt1003, long *pcopylen)
{
	struct nt1003_frame *frame;
	unsigned char *data, *f, yuv420_vu[MAX_FRAME_WIDTH];
	unsigned int len, yuv420_vu_size, yuv420_y_size;
	const int v4l_linesize = imgwidth * V4L_BYTES_PER_PIXEL;
	int i, block, pixel_out;
	unsigned char rv, gv, bv;       /* RGB components */ 
	
	
	data = nt1003->scratch;
	frame = &nt1003->frame[nt1003->curframe];
	/* Make sure there's enough data for the entire line */
	/* In this mode nt1003 transfer 3 bytes for every 2 pixels */
	/* I need two lines to decode the color */
        yuv420_y_size = frame->frmwidth;
	len = yuv420_y_size * 3 / 2 * 2;
	if ( scratch_left(data) < len ) {
		//printk(KERN_DEBUG "out of data, need %d\n", len);
		return scan_Out;
	}

	if ((frame->curline + 1) >= imgheight) {
		return scan_NextFrame;
	}

	// get color information
	// start of color blocks: 128, 320, 512, 704, 896, 1088 ...
	// <block> is the number of the color block
	for (block = 0; block < yuv420_y_size / 64; block++) {
	    memcpy(&yuv420_vu[block*64], data+(block*3+2)*64, 64);
	}
	yuv420_vu_size=yuv420_y_size/2; // size of u or v part

	f = frame->data + (v4l_linesize * frame->curline);
	pixel_out=0;
	// get even line (here block is the start offset of the y blocks)
	for (block=0 ; block < yuv420_y_size*2; block += 128, data += 192) {
	    for (i = 0; i < 128; i++, pixel_out++) {
        	YUV_TO_RGB_BY_THE_BOOK(data[i], 
			yuv420_vu[(pixel_out%yuv420_y_size)/2], 
	    		yuv420_vu[yuv420_vu_size+(pixel_out%yuv420_y_size)/2], 
			rv, gv, bv);
                *f++ = bv;
                *f++ = gv;
                *f++ = rv;
	    }
	}
	
	frame->curline +=2;
	*pcopylen += v4l_linesize*2;
	nt1003_align_scratch(nt1003, data);

	if (frame->curline >= frame->frmheight)
		return scan_NextFrame;
	else
		return scan_Continue;
}

/*
 * nt1003_parse_data()
 *
 * Generic routine to parse the scratch buffer. It employs either
 * nt1003_find_header() or nt1003_parse_lines() to do most
 * of work.
 *
 */
static void
nt1003_parse_data(struct usb_nt1003 *nt1003)
{
	struct nt1003_frame *frame;
	unsigned char *data = nt1003->scratch;
	scan_state_t newstate;
	long copylen = 0;

	/* Grab the current frame and the previous frame */
	frame = &nt1003->frame[nt1003->curframe];

/*	printk(KERN_DEBUG "parsing %u.\n", nt1003->scratchlen);*/

	while (1) {

		newstate = scan_Out;
		if (scratch_left(data)) {
			if (frame->scanstate == STATE_SCANNING)
				newstate = nt1003_find_header(nt1003);
			else if (frame->scanstate == STATE_LINES) {
				if (videoformat == VIDEOFORMAT_YUV420)
					newstate = nt1003_parse_lines_420(
							nt1003, &copylen);
				else if (videoformat == VIDEOFORMAT_YUV422)
					newstate = nt1003_parse_lines_422(
							nt1003, &copylen);
			}
		}
		if (newstate == scan_Continue)
			continue;
		else if ((newstate == scan_NextFrame) || (newstate == scan_Out))
			break;
		else
			return; /* scan_EndParse */
	}

	if (newstate == scan_NextFrame) {
		frame->grabstate = FRAME_DONE;
		nt1003->curframe = -1;
		nt1003->frame_num++;

		/* Optionally display statistics on the screen */
		if (flags & FLAGS_OVERLAY_STATS)
			nt1003_overlaystats(nt1003, frame);

		/* This will cause the process to request another frame. */
		if (waitqueue_active(&frame->wq))
			wake_up_interruptible(&frame->wq);
	}

	/* Update the frame's uncompressed length. */
	frame->scanlength += copylen;
}

/*
 * Make all of the blocks of data contiguous
 */
static int
nt1003_compress_isochronous(struct usb_nt1003 *nt1003, urb_t *urb)
{
	unsigned char *cdata, *data, *data0;
	int i, totlen = 0;

	data = data0 = nt1003->scratch + nt1003->scratchlen;
	for (i = 0; i < urb->number_of_packets; i++) {
		int n = urb->iso_frame_desc[i].actual_length;
		int st = urb->iso_frame_desc[i].status;
		
		cdata = urb->transfer_buffer + urb->iso_frame_desc[i].offset;

		/* Detect and ignore errored packets */
		if (st) { // st != 0 ?????????????
			if (debug >= 1) {
				printk(KERN_ERR
				  "nt1003 data error: [%d] len=%d, status=%X\n",
				   i, n, st);
			}
			nt1003->iso_err_count++;
			continue;
		}

		/* Detect and ignore empty packets */
		if (n < 0) {
			if (debug >= 2) {
				printk(KERN_ERR
				  "nt1003 error packet [%d]\n", i);
			}
			nt1003->iso_skip_count++;
			continue;
		}
		else if (n == 0){	/* Frame end ????? */ 
			if (debug >= 2) {
				printk(KERN_DEBUG
				  "nt1003 null packet [%d]\n", i);
			}
			nt1003->iso_skip_count++;
			continue;
		}

		/*
		 * If camera continues to feed us with data but there is no
		 * consumption (if, for example, V4L client fell asleep) we
		 * may overflow the buffer. We have to move old data over to
		 * free room for new data. This is bad for old data. If we
		 * just drop new data then it's bad for new data... choose
		 * your favorite evil here.
		 */
		if ((nt1003->scratchlen + n) > scratchbufsize) {
#if 0
			nt1003->scratch_ovf_count++;
			if (debug >= 3)
				printk(KERN_ERR "nt1003: scratch buf overflow! "
				       "scr_len: %d, n: %d\n", nt1003->scratchlen, n );
			return totlen;
#else
			int mv;

			nt1003->scratch_ovf_count++;
			if (debug >= 3) {
				printk(KERN_ERR "nt1003: scratch buf overflow! "
			        "scr_len: %d, n: %d\n", nt1003->scratchlen, n );
			}
			mv  = (nt1003->scratchlen + n) - scratchbufsize;
			if (nt1003->scratchlen >= mv) {
				int newslen = nt1003->scratchlen - mv;
				memmove(nt1003->scratch, nt1003->scratch + mv,
					newslen);
				nt1003->scratchlen = newslen;
				data = data0 = nt1003->scratch +
					       nt1003->scratchlen;
			} else {
				printk(KERN_ERR
				       "nt1003: scratch buf too small\n");
				return totlen;
			}
#endif
		}

		/* Now we know that there is enough room in scratch buffer */
		memmove(data, cdata, n);
		data += n;
		totlen += n;
		nt1003->scratchlen += n;

/*
printk(KERN_DEBUG "totlen %d scratchlen %d scractchleft %d\n", totlen, 
		  nt1003->scratchlen, scratch_left(data0) );
*/

	}
#if ENABLE_HEXDUMP
	if (totlen > 0) {
		static int foo=0;
		if (foo < 1) {
			printk(KERN_DEBUG "+%d.\n", nt1003->scratchlen);
			nt1003_hexdump(data0, (totlen > 64) ? 64:totlen);
			++foo;
		}
	}
#endif
	return totlen;
}

static void 
nt1003_isoc_irq(struct urb *urb)
{
	int len;
	struct usb_nt1003 *nt1003 = urb->context;
	struct nt1003_sbuf *sbuf;
	int i;

	/* We don't want to do anything if we are about to be removed! */
	if (!NT1003_IS_OPERATIONAL(nt1003))
		return;

#if 0
	if (urb->actual_length > 0) {
		printk(KERN_DEBUG "nt1003_isoc_irq: %p status %d, "
		       " errcount = %d, length = %d\n", urb, urb->status,
		       urb->error_count, urb->actual_length);
	} else {
		static int c = 0;
		if ((c++ % 20) == 0)
			printk(KERN_DEBUG "nt1003_isoc_irq: no data\n");
		// This causes a panic Schedule in interrupt
		// nt1003_begin_streaming(nt1003);
	}
	return;
#endif

	if (!nt1003->streaming) {
		if (debug >= 1)
			printk(KERN_DEBUG 
			       "nt1003: oops, not streaming, but interrupt\n");
		return;
	}
	
	sbuf = &nt1003->sbuf[nt1003->cursbuf];

	/* Copy the data received into our scratch buffer */
	len = nt1003_compress_isochronous(nt1003, urb);

	nt1003->urb_count++;
	nt1003->urb_length = len;
	nt1003->data_count += len;

#if 0
        /* This code prints few initial bytes of ISO data: used to
	 * decode markers */
	if (nt1003->urb_count % 20 == 1) {
		if (nt1003->urb_count == 1) {
			nt1003_hexdump(nt1003->scratch,
				       (nt1003->scratchlen > 32) ? 32 : nt1003->scratchlen);
		}
	}
#endif

	/* If we collected enough data let's parse! */
	if (nt1003->scratchlen > 12) { /* 12 == header_length */
		/*If we don't have a frame we're current working on, complain*/
		if (nt1003->curframe >= 0)
			nt1003_parse_data(nt1003);
		else {
			if (debug >= 1)
				printk(KERN_DEBUG
				       "nt1003: received data, but no frame\n");
		}
	}

	for (i = 0; i < FRAMES_PER_DESC; i++) {
		sbuf->urb->iso_frame_desc[i].status = 0;
		sbuf->urb->iso_frame_desc[i].actual_length = 0;
	}

	/* Move to the next sbuf */
	nt1003->cursbuf = (nt1003->cursbuf + 1) % NT1003_NUMSBUF;

	return;
}

/*************************************/
/* Low level nt1003 access functions */
/*************************************/

/*
 * nt1003_read_reg()
 *
 * return  < 0 -> Error
 *        >= 0 -> Data
 */

static int
nt1003_read_reg(struct usb_nt1003 *nt1003, unsigned char reg)
{
	static const char proc[] = "nt1003_read_reg";
	int rc;
	unsigned char buffer[1];

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return -1;

	rc = usb_control_msg(
			nt1003->dev,
			usb_rcvctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)reg,
			buffer,
			1,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
		return rc;
	}
	return buffer[0];
}

/*
 * nt1003_write_reg()
 *
 * return 1 -> Reg written
 *        0 -> nt1003 is not yet ready
 *       -1 -> Something went wrong
 */

static int
nt1003_write_reg(struct usb_nt1003 *nt1003, unsigned char reg,
		 unsigned char value)
{
	static const char proc[] = "nt1003_write_reg";
	int rc;

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)reg,
			&value,
			1,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
	}
	return rc;
}

/* ----------------------------------------------------------------------- */
/* I2C functions                                                           */
/* ----------------------------------------------------------------------- */

static void 
nt1003_inc_use(struct i2c_adapter *adap)
{
        MOD_INC_USE_COUNT;
}

static void
nt1003_dec_use(struct i2c_adapter *adap)
{
        MOD_DEC_USE_COUNT;
}

static void 
call_i2c_clients(struct usb_nt1003 *nt1003, unsigned int cmd, void *arg)
{
        int i;

        for (i = 0; i < I2C_CLIENTS_MAX; i++) {
                if (NULL == nt1003->i2c_clients[i])
                        continue;
                if (NULL == nt1003->i2c_clients[i]->driver->command)
                        continue;
                nt1003->i2c_clients[i]->driver->command(
                        nt1003->i2c_clients[i],cmd,arg);
        }
}

static int 
attach_inform(struct i2c_client *client)
{
        struct usb_nt1003 *nt1003 = (struct usb_nt1003*)client->adapter->data;
        int i;

        for (i = 0; i < I2C_CLIENTS_MAX; i++) {
                if (nt1003->i2c_clients[i] == NULL ||
                    nt1003->i2c_clients[i]->driver->id == client->driver->id) {
                        nt1003->i2c_clients[i] = client;
                        break;
                }
        }

	/* Look for a better place to put this */
        if ( (nt1003->have_tuner) && (nt1003->tuner_type != -1) )
		call_i2c_clients(nt1003, TUNER_SET_TYPE, &nt1003->tuner_type);
        if ( nt1003->have_videodecoder )
	  {
             call_i2c_clients(nt1003, DECODER_SET_PICTURE, &nt1003->vpic);
	     nt1003_muxsel(nt1003, nt1003->channel, nt1003->vchan.norm);
	  }
        if (verbose)
           printk("usb_nt1003-%d: i2c attach [%s]\n", nt1003->nr, client->name);

        return 0;
}

static int
detach_inform(struct i2c_client *client)
{
        struct usb_nt1003 *nt1003 = (struct usb_nt1003*)client->adapter->data;
        int i;

        if (verbose)
                printk("usb_nt1003-%d: i2c detach [%s]\n", nt1003->nr, 
				client->name);
        for (i = 0; i < I2C_CLIENTS_MAX; i++) {
                if (NULL != nt1003->i2c_clients[i] &&
                    nt1003->i2c_clients[i]->driver->id == client->driver->id) {
                        nt1003->i2c_clients[i] = NULL;
                        break;
                }
        }
        return 0;
}

static int
nt1003_i2c_read_max4(struct usb_nt1003 *nt1003, unsigned char addr,
		     char *buf, short len)
{
	int rc, retries;

	for(retries = 5;;) {
                rc = nt1003_write_reg(nt1003, NT1003_SER_ADRS, addr);
	        if (rc < 0) return rc;

	        /* Initiate byte read cycle */
	        rc = nt1003_write_reg(nt1003, NT1003_SER_CONT,
				(len&0x07) | 0x18);
	        if (rc < 0) return rc;

                /* Test for Busy and ACK */
                do rc = nt1003_read_reg(nt1003, NT1003_SER_CONT);
                while(rc > 0 && ((rc&0x10) != 0)); /* Retry while busy */
                if (rc < 0) return rc;

                if((rc&0x20) == 0) /* Ack? */
                        break;

                /* I2C abort */
                rc = nt1003_write_reg(nt1003, NT1003_SER_CONT, 0x00);
                if (rc < 0) return rc;

                if (--retries < 0) return -1;
        }
	
	switch (len) {
	   case 4:
              buf[3] = nt1003_read_reg(nt1003, NT1003_SER_DAT4);
	   case 3:
              buf[2] = nt1003_read_reg(nt1003, NT1003_SER_DAT3);
	   case 2:
              buf[1] = nt1003_read_reg(nt1003, NT1003_SER_DAT2);
	   case 1:
              buf[0] = nt1003_read_reg(nt1003, NT1003_SER_DAT1);
	      break;
	   default :
	      printk(KERN_ERR "nt1003_i2c_read_max4: buffer length > 4\n");
	}

	return len;
}


static int
nt1003_i2c_write_max4(struct usb_nt1003 *nt1003, unsigned char addr,
		      const char *buf, short len)
{
	int rc, retries;
	int i;
	unsigned char value[6];
	unsigned char ser_cont;

	ser_cont = (len&0x07) | 0x10;

	value[0] = addr;
	value[1] = ser_cont;
	for (i=0; i < len; i++) value[i+2] = buf[i];

        for (retries = 5;;)
          {
               rc = usb_control_msg(nt1003->dev,
                  	usb_sndctrlpipe(nt1003->dev, 1),
			0x33,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0, (__u16)NT1003_SER_ADRS, value, len+2, HZ);

               if (rc < 0) return rc;

               rc = nt1003_write_reg(nt1003, NT1003_SER_CONT, 
			       (len&0x07) | 0x10);
	       if (rc < 0) return rc;

	       /* Test for Busy and ACK */
	       do rc = nt1003_read_reg(nt1003, NT1003_SER_CONT);
	       while(rc > 0 && ((rc&0x10) != 0)); /* Retry while busy */
	       if (rc < 0) return rc;

	       if((rc&0x20) == 0) /* Ack? */
	             break;

	       /* I2C abort */
	       nt1003_write_reg(nt1003, NT1003_SER_CONT, 0x00);

	       if (--retries < 0) return -1;

	  }

	return len;
}

static int
nt1003_i2c_write(void *data, unsigned char addr, char *buf, short len)
{
	char *temp = buf;
	int retval;
	int wrcount=0;
	int count;
	int tmp;
	struct usb_nt1003 *nt1003 = (struct usb_nt1003*)data;

	if (addr == 0x48)
		tmp = 2;
	else
		tmp=4;

	while (len > 0) {
		count = (len > tmp)? tmp : len;
		retval = nt1003_i2c_write_max4(nt1003, addr, temp, count);
		if (retval>0) {
			len-= count;
			temp+= count;
			wrcount+= count;
		}
		else
			return (retval<0)? retval : -EFAULT;
	}
	return wrcount;
}

static int
nt1003_i2c_read(void *data, unsigned char addr, char *buf, short len)
{
	char temp[4];
	int retval,i;
	int rdcount=0;
	int count;
	struct usb_nt1003 *nt1003 = (struct usb_nt1003*)data;

	while (len > 0) {
		count = (len > 3)? 4 : len;
		retval = nt1003_i2c_read_max4(nt1003, addr, temp, count);
		if (retval>0) {
			for (i=0; i < len; i++)
				buf[rdcount+i]= temp[i];
			len-= count;
			rdcount+= count;
		}
		else 
			return (retval<0)? retval : -EFAULT;
	}
	return rdcount;
}

static struct i2c_algo_usb_data i2c_algo_template = {
        NULL,
        nt1003_i2c_read,
        nt1003_i2c_write,
        10, 10, 100,
};

static struct i2c_adapter i2c_adap_template = {
        "nt1003",
        I2C_HW_B_BT848,  /* FIXME */
        NULL,
        NULL,
        nt1003_inc_use,
        nt1003_dec_use,
        attach_inform,
        detach_inform,
        NULL,
};

static struct i2c_client i2c_client_template = {
        "nt1003 internal",
        -1,
        0,
        0,
        NULL,
        NULL
};

static int
nt1003_init_i2c(struct usb_nt1003 *nt1003)
{
	memcpy(&nt1003->i2c_adap, &i2c_adap_template,
		sizeof(struct i2c_adapter));
	memcpy(&nt1003->i2c_algo, &i2c_algo_template,
		sizeof(struct i2c_algo_usb_data));
	memcpy(&nt1003->i2c_client, &i2c_client_template,
	sizeof(struct i2c_client));

	sprintf(nt1003->i2c_adap.name+strlen(nt1003->i2c_adap.name),
		" #%d", nt1003->nr);
	nt1003->i2c_algo.data = nt1003;
	nt1003->i2c_adap.data = nt1003;
	nt1003->i2c_adap.algo_data = &nt1003->i2c_algo;
	nt1003->i2c_client.adapter = &nt1003->i2c_adap;

	/* Joerg: change SER_MODE to NACK */
	if (nt1003_write_reg(nt1003, NT1003_SER_MODE, 0x30) < 0) {
		printk(KERN_ERR "nt1003_init_i2c: can't wirte reg\n");
		return -EBUSY;
	}

	nt1003->i2c_ok = i2c_usb_add_bus(&nt1003->i2c_adap);

	return nt1003->i2c_ok;

}

/****************************/
/* nt1003 utility functions */
/****************************/

/*
 * nt1003_calculate_fps()
 *
 * This procedure roughly calculates the real frame rate based
 * on FPS code (framerate=NNN option). The actual frame rate is 
 * determined by the nt1003. Since I don't know how to ask
 * the chip what FPS is now I have to use the FPS code instead.
 *
 * The FPS code is in range [0..31], 0 is slowest, 6 is fastest.
 * Corresponding real FPS should be in range [0??..30] frames per second
 * in NTSC and [0..25] in PAL
 *
 * real_fps NTSC = 30 * (fps_code + 1) / 32
 * real_fps PAL  = 25 * (fps_code + 1) / 32
 *
 */
static int
nt1003_calculate_fps(void)
{
//	if pal
//		return 25 * (framerate + 1) / 32;
        return 0;
}


static int
nt1003_power_off(struct usb_nt1003 *nt1003)
{

	return (nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x02));
}

/*
 * nt1003_set_video_format()
 *
 */

static int
nt1003_set_video_format(struct usb_nt1003 *nt1003, int format)
{
	static const char proc[] = "nt1003_set_video_format";
	int rc;
	unsigned char value[2];

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;
//	if(format == videoformat)
//		return 1;

	switch (format){
		case VIDEOFORMAT_YUV422:
			value[1] = 0x03;
			break;
		case VIDEOFORMAT_YUV420:
			value[1] = 0x14;
			break;
		case VIDEOFORMAT_COMPRESS:
			value[1] = 0x60;
			break;
		default:
			printk(KERN_ERR "nt1003: unknown video format %02x",
					format);
			return -1;
	}
	value[0] = 0x00;  /*TODO: See the effect of the filter */ 

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)NT1003_FILT_CONT,
			value,
			2,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
	}
	videoformat = format;
	return rc;
}

/*
 * nt1003_set_output_size()
 *
 */

static int
nt1003_set_output_size(struct usb_nt1003 *nt1003, int width, int height)
{
	static const char proc[] = "nt1003_set_output_size";
	int rc;
	unsigned int framerate;
	unsigned char value[4];
//	unsigned char value[] = {0x40, 0x01, 0xf0, 0x00}; // 320x240
//	unsigned char value[] = {0x80, 0x02, 0xe0, 0x01}; // 640x480
//	unsigned char value[] = {0xff, 0xff, 0xff, 0xff}; // ????

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;

	/* Take care with this, it depends of the architecture */
	value[0] = width & 0x00ff;
	value[1] = (width >> 8) ;
	value[2] = height & 0x00ff;
	value[3] = (height >> 8);

	if ( (width == imgwidth) && (height == imgheight) )
		return 1;

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)NT1003_LXSIZE_O,
			value,
			4,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
		return rc;
	}
	imgwidth = width;
	imgheight = height;

	if (videoformat == VIDEOFORMAT_YUV422)
		framerate = (nt1003->iso_packet_len * 1000)
			    / (imgwidth * imgheight * 2);
	else
		framerate = (nt1003->iso_packet_len * 1000)
			    / ((imgwidth * imgheight * 12)/8);

printk("framerate %d\n", framerate);

        RESTRICT_TO_RANGE(framerate, FRAMERATE_MIN, FRAMERATE_MAX);
        return nt1003_write_reg(nt1003, NT1003_FRM_RATE, framerate);

}

/*
 * nt1003_set_compress_params()
 *
 */

static int
nt1003_set_compress_params(struct usb_nt1003 *nt1003)
{
	static const char proc[] = "nt1003_set_compresion_params";
	int rc;
	unsigned char value[] = {0x01, 0x00, 0x00};

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)NT1003_FORCE_INTRA,
			value,
			3,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
		return rc;
	}

	return rc;

}

/*
 * nt1003_set_input()
 *
 * Set the input (saa7111) size x y and other misc input params
 * I've no idea if this parameters are right
 *
 */

static int
nt1003_set_input(struct usb_nt1003 *nt1003)
{
	static const char proc[] = "nt1003_set_input";
	int rc;
	unsigned char value[8];

	if (nt1003->vchan.norm == VIDEO_MODE_PAL){
		value[0] = 0xBF; value[1] = 0x02;
		value[2] = 0x20; value[3] = 0x01;
		value[4] = 0x05; value[5] = 0x00;
		value[6] = 0x10; value[7] = 0x00;
	}
	else{
		value[0] = 0x00; value[1] = 0x03;
		value[2] = 0x20; value[3] = 0x01;
		value[4] = 0x51; value[5] = 0x00;
		value[6] = 0x17; value[7] = 0x00;
	}

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)NT1003_LXSIZE_I,
			value,
			8,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
		return rc;
	}

	/* U comes after V, Ya comes after U/V, Yb comes after Yb */
	return (nt1003_write_reg(nt1003, NT1003_DVI_YUV, 0x00));
}

/*
 * nt1003_set_dram_settings()
 *
 * Set the buffer address needed by the nt1003 dram to operate
 * This values has been taken with usbsnoop.
 *
 */

static int
nt1003_set_dram_settings(struct usb_nt1003 *nt1003)
{
	static const char proc[] = "nt1003_set_dram_settings";
	int rc;
	unsigned char value[] = {0x42, 0x00, 0xff, 0x00, 
		                 0x00, 0x00, 0x00, 0xff};

        if (!NT1003_IS_OPERATIONAL(nt1003))
                return 0;

	rc = usb_control_msg(
			nt1003->dev,
			usb_sndctrlpipe(nt1003->dev,1),
			0x33, /* NT1003 specific code */
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT,
			0,
			(__u16)NT1003_DRM_PRM1,
			value,
			8,
			HZ);

	if (rc < 0) {
		printk(KERN_ERR "%s: ERROR=%d. NT1003 stopped - "
		       "reconnect or reload driver.\n", proc, rc);
		return rc;
	}

	/* Restart the video buffer logic */
//	if ( (rc = nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x22) ) < 0)
//		return rc;
	if ( (rc = nt1003_write_reg(nt1003, NT1003_DRM_CONT, 0x1c) ) < 0)
		return rc;
//	if ( (rc = nt1003_write_reg(nt1003, NT1003_DRM_CONT, 0x26) ) < 0)
//		return rc;
	rc = nt1003_write_reg(nt1003, NT1003_DRM_CONT, 0x00);

	return rc;
}

/*
 * nt1003_power_on()
 *
 * Power on the device, enables suspend-resume logic 
 * &  reset the isoc End-Point
 *
 */

static int
nt1003_power_on(struct usb_nt1003 *nt1003)
{
	nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x02);
	nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x06);
	/* I'm sure only about d2-d0 [010] 16 bit 4:2:2 usin sync pulses 
	 * as that is how saa7111 is configured */
	nt1003_write_reg(nt1003, NT1003_VIN_REG1, 0x8a);
	nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x22);
	return (nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x26));
}

/*
 * nt1003_begin_streaming()
 * Sure you have to put bit 7 to 0, if not incoming frames are droped, but no
 * idea about the rest
 */

static int 
nt1003_begin_streaming(struct usb_nt1003 *nt1003)
{
	return (nt1003_write_reg(nt1003, NT1003_VIN_REG2, 0x00));

	return 0;
}

/*
 * nt1003_restart_isoc()
 * Not sure yet if touching here PWR_REG make loose the config
 */

static int 
nt1003_restart_isoc(struct usb_nt1003 *nt1003)
{
	int ret;

	if ( (ret = nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x22)) < 0 )
		return ret;
	if ( (ret = nt1003_write_reg(nt1003, NT1003_PWR_REG, 0x26)) < 0 )
		return ret;
	if ( (ret = nt1003_write_reg(nt1003, NT1003_VIN_REG2, 0x80)) < 0 )
		return ret;

	/* TODO: schedule timeout */
	while ( ( nt1003_read_reg(nt1003, NT1003_STATUS_REG) && 0x01 ) != 1);

	return 0;
}

static int 
nt1003_setup(struct usb_nt1003 *nt1003)
{

//	ret = nt1003_power_on(nt1003);
//	ret = nt1003_init_i2c(nt1003);

	if (nt1003_write_reg(nt1003, NT1003_IOPIN_REG, 0x01) < 0)
          {
             printk(KERN_ERR "can't wirte reg\n");
             return -1;
          }

	nt1003_set_dram_settings(nt1003);
	nt1003_set_compress_params(nt1003);
	nt1003_set_input(nt1003);
	nt1003_set_video_format(nt1003, videoformat);
	nt1003_set_output_size(nt1003, 320, 240);
	nt1003_restart_isoc(nt1003);
	nt1003_begin_streaming(nt1003);

	/* cosas del PCM */

	return NT1003_IS_OPERATIONAL(nt1003);
}


/*
 * nt1003_init_isoc()
 *
 */
static int
nt1003_init_isoc(struct usb_nt1003 *nt1003)
{
	struct usb_device *dev = nt1003->dev;
	int i, err;

	if (!NT1003_IS_OPERATIONAL(nt1003))
		return -EFAULT;

	nt1003->compress = 0;
	nt1003->curframe = -1;
	nt1003->cursbuf = 0;
	nt1003->scratchlen = 0;

	/* Alternate interface 1 is is the biggest frame size */
	i = usb_set_interface(dev, nt1003->iface, nt1003->ifaceAltActive);
	if (i < 0) {
		printk(KERN_ERR "usb_set_interface error\n");
		nt1003->last_error = i;
		return -EBUSY;
	}

	/* We double buffer the Iso lists */

	for (i=0; i < NT1003_NUMSBUF; i++) {
		int j, k;
		urb_t *urb;

		urb = usb_alloc_urb(FRAMES_PER_DESC);
		if (urb == NULL) {
			printk(KERN_ERR
			       "nt1003_init_isoc: usb_init_isoc() failed.\n");
			return -ENOMEM;
		}
		nt1003->sbuf[i].urb = urb;
		urb->dev = dev;
		urb->context = nt1003;
		urb->pipe = usb_rcvisocpipe(dev, nt1003->video_endp);
		urb->transfer_flags = USB_ISO_ASAP;
		urb->transfer_buffer = nt1003->sbuf[i].data;
		urb->complete = nt1003_isoc_irq;
		urb->number_of_packets = FRAMES_PER_DESC;
		urb->transfer_buffer_length = 
			nt1003->iso_packet_len * FRAMES_PER_DESC;
		for (j=k=0; j < FRAMES_PER_DESC; j++, 
		     k += nt1003->iso_packet_len) {
			urb->iso_frame_desc[j].offset = k;
			urb->iso_frame_desc[j].length = nt1003->iso_packet_len;
		}
	}

	/* Link URBs into a ring so that they invoke each other infinitely */
	for (i=0; i < NT1003_NUMSBUF; i++) {
		if ((i+1) < NT1003_NUMSBUF)
			nt1003->sbuf[i].urb->next = nt1003->sbuf[i+1].urb;
		else
			nt1003->sbuf[i].urb->next = nt1003->sbuf[0].urb;
	}

	/* Submit all URBs */
	for (i=0; i < NT1003_NUMSBUF; i++) {
		err = usb_submit_urb(nt1003->sbuf[i].urb);
		if (err)
			printk(KERN_ERR
			       "nt1003_init_isoc: usb_run_isoc(%d) ret %d\n",
			       i, err);
	}


	nt1003->streaming = 1;
	printk(KERN_DEBUG
	       "streaming=1 nt1003->video_endp=$%02x\n", nt1003->video_endp);
	return 0;
}

/*
 * nt1003_stop_isoc()
 *
 * This procedure stops streaming and deallocates URBs. Then it
 * activates zero-bandwidth alt. setting of the video interface.
 *
 */
static void 
nt1003_stop_isoc(struct usb_nt1003 *nt1003)
{
	static const char proc[] = "nt1003_stop_isoc";
	int i, j;

	if (!nt1003->streaming || (nt1003->dev == NULL))
		return;

	/* Unschedule all of the iso td's */
	for (i=0; i < NT1003_NUMSBUF; i++) {
		j = usb_unlink_urb(nt1003->sbuf[i].urb);
		if (j < 0)
			printk(KERN_ERR
			       "%s: usb_unlink_urb() error %d.\n", proc, j);
	}
	/* printk(KERN_DEBUG "streaming=0\n"); */
	nt1003->streaming = 0;

	/* Delete them all */
	for (i=0; i < NT1003_NUMSBUF; i++)
		usb_free_urb(nt1003->sbuf[i].urb);

	if (!nt1003->remove_pending) {

		/* Set packet size to 0 */
		j = usb_set_interface(nt1003->dev, nt1003->iface,
				      nt1003->ifaceAltInactive);
		if (j < 0) {
			printk(KERN_ERR
			       "%s: usb_set_interface() error %d.\n", proc, j);
			nt1003->last_error = j;
		}
	}
}

/*
 * nt1003_new_frame()
 *
 */
static int
nt1003_new_frame(struct usb_nt1003 *nt1003, int framenum)
{
	struct nt1003_frame *frame;
	int n, width, height;

	/* If we're not grabbing a frame right now and the other frame is */
	/*  ready to be grabbed into, then use it instead */
	if (nt1003->curframe != -1)
		return 0;

	n = (framenum - 1 + NT1003_NUMFRAMES) % NT1003_NUMFRAMES;
	if (nt1003->frame[n].grabstate == FRAME_READY)
		framenum = n;

	frame = &nt1003->frame[framenum];

	frame->grabstate = FRAME_GRABBING;
	frame->scanstate = STATE_SCANNING;
	frame->scanlength = 0;		/* Accumulated in nt1003_parse_data() */
	nt1003->curframe = framenum;

	/*
	 * Normally we would want to copy previous frame into the current one
	 * before we even start filling it with data; this allows us to stop
	 * filling at any moment; top portion of the frame will be new and
	 * bottom portion will stay as it was in previous frame. If we don't
	 * do that then missing chunks of video stream will result in flickering
	 * portions of old data whatever it was before.
	 *
	 * If we choose not to copy previous frame (to, for example, save few
	 * bus cycles - the frame can be pretty large!) then we have an option
	 * to clear the frame before using. If we experience losses in this
	 * mode then missing picture will be black (no flickering).
	 *
	 * Finally, if user chooses not to clean the current frame before
	 * filling it with data then the old data will be visible if we fail
	 * to refill entire frame with new data.
	 */
	if (!(flags & FLAGS_SEPARATE_FRAMES)) {
		/* This copies previous frame into this one to mask losses */
		memmove(frame->data, nt1003->frame[1-framenum].data, 
			MAX_FRAME_SIZE);
	} else {
		if (flags & FLAGS_CLEAN_FRAMES) {
			/*This provides a "clean" frame but slows things down */
			memset(frame->data, 0, MAX_FRAME_SIZE);
		}
	}

	width = frame->width;
	RESTRICT_TO_RANGE(width, min_imgwidth, max_imgwidth);
//	width &= ~7;		/* Multiple of 8 */

	height = frame->height;
	RESTRICT_TO_RANGE(height, min_imgheight, max_imgheight);
//	height &= ~3;		/* Multiple of 4 */

	return 0;
}

static int
nt1003_muxsel(struct usb_nt1003 *nt1003, int channel, int norm)
{

	/* set the new video norm */
	if (nt1003->vchan.norm != norm) {
		int video_command=VIDEO_MODE_AUTO;
//		call_i2c_clients(nt1003, DECODER_SET_NORM, &norm);
		call_i2c_clients(nt1003, DECODER_SET_NORM, &video_command);
		video_command=1;
		call_i2c_clients(nt1003, DECODER_ENABLE_OUTPUT, &video_command);
		nt1003->vchan.norm = norm;
	}

	/* set the new channel */
	call_i2c_clients(nt1003, DECODER_SET_INPUT, &channel);
	nt1003->channel = channel;
	return 0;
}


/*
 * nt1003_open()
 *
 * This is part of Video 4 Linux API. The driver can be opened by one
 * client only (checks internal counter 'nt1003->user'). The procedure
 * then allocates buffers needed for video processing.
 *
 */
static int
nt1003_open(struct video_device *dev, int flags)
{
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *)dev;
	const int sb_size = FRAMES_PER_DESC * nt1003->iso_packet_len;
	/* sb_size should be FR_PER_DESC * max_packet_len if wanna change 
	 * packet len dinamically for bw management
	 */
	int i, err = 0;

	down(&nt1003->lock);

	if (nt1003->user)
		err = -EBUSY;
	else {
		/* Clean pointers so we know if we allocated something */
		for (i=0; i < NT1003_NUMSBUF; i++)
			nt1003->sbuf[i].data = NULL;

		/* Allocate memory for the frame buffers */
		nt1003->fbuf_size = NT1003_NUMFRAMES * MAX_FRAME_SIZE;
		nt1003->fbuf = rvmalloc(nt1003->fbuf_size);
		nt1003->scratch = kmalloc(scratchbufsize, GFP_KERNEL);
		nt1003->scratchlen = 0;
		if ((nt1003->fbuf == NULL) || (nt1003->scratch == NULL))
			err = -ENOMEM;
		else {
			/* Allocate all buffers */
			for (i=0; i < NT1003_NUMFRAMES; i++) {
				nt1003->frame[i].grabstate = FRAME_UNUSED;
				nt1003->frame[i].data = nt1003->fbuf + 
					i*MAX_FRAME_SIZE;
				/*
				 * Set default sizes in case IOCTL
				 * (VIDIOCMCAPTURE)
				 * is not used (using read() instead).
				 */
				nt1003->frame[i].width = imgwidth;
				nt1003->frame[i].height = imgheight;
				nt1003->frame[i].bytes_read = 0;
			}
			for (i=0; i < NT1003_NUMSBUF; i++) {
				nt1003->sbuf[i].data = 
					kmalloc(sb_size, GFP_KERNEL);
				if (nt1003->sbuf[i].data == NULL) {
					err = -ENOMEM;
					break;
				}
			}
		}
		if (err) {
			/* Have to free all that memory */
			if (nt1003->fbuf != NULL) {
				rvfree(nt1003->fbuf, nt1003->fbuf_size);
				nt1003->fbuf = NULL;
			}
			if (nt1003->scratch != NULL) {
				kfree(nt1003->scratch);
				nt1003->scratch = NULL;
			}
			for (i=0; i < NT1003_NUMSBUF; i++) {
				if (nt1003->sbuf[i].data != NULL) {
					kfree (nt1003->sbuf[i].data);
					nt1003->sbuf[i].data = NULL;
				}
			}
		}
	}

	/* If so far no errors then we shall start the camera */
	if (!err) {
		/* Send init sequence only once, it's large! */
		if (!nt1003->initialized) {
			int setup_ok = 0;
			setup_ok = nt1003_setup(nt1003);
			if (setup_ok)
				nt1003->initialized = 1;
			else
				err = -EBUSY;
		}
		nt1003_begin_streaming(nt1003);
		if (!err) 
			err = nt1003_init_isoc(nt1003);
		if (!err) {
			nt1003->user++;
			MOD_INC_USE_COUNT;
		}
	}

	up(&nt1003->lock);
	return err;
}

/*
 * nt1003_close()
 *
 * This is part of Video 4 Linux API. The procedure
 * stops streaming and deallocates all buffers that were earlier
 * allocated in nt1003_open().
 *
 */
static void 
nt1003_close(struct video_device *dev)
{
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *)dev;
	int i;

	down(&nt1003->lock);	

	nt1003_restart_isoc(nt1003);
	nt1003_stop_isoc(nt1003);
//	nt1003->initialized=0;

	rvfree(nt1003->fbuf, nt1003->fbuf_size);
	kfree(nt1003->scratch);
	for (i=0; i < NT1003_NUMSBUF; i++)
		kfree(nt1003->sbuf[i].data);

	nt1003->user--;
	MOD_DEC_USE_COUNT;

	if (nt1003->remove_pending) {
		printk(KERN_INFO "nt1003_close: Final disconnect.\n");
		nt1003_release(nt1003);
	}
	up(&nt1003->lock);
}

static int nt1003_init_done(struct video_device *dev)
{
	return 0;
}

static long 
nt1003_write(struct video_device *dev, const char *buf, unsigned long count,
		int noblock)
{
	return -EINVAL;
}

/*
 * nt1003_ioctl()
 *
 * This is part of Video 4 Linux API. The procedure handles ioctl() calls.
 *
 */
static int
nt1003_ioctl(struct video_device *dev, unsigned int cmd, void *arg)
{
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *)dev;

	if (!NT1003_IS_OPERATIONAL(nt1003))
		return -EFAULT;

	switch (cmd) {
		case VIDIOCGCAP:
		{
			if (copy_to_user(arg, &nt1003->vcap,
					sizeof(nt1003->vcap)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCGCHAN:
		{
                        struct video_channel v;
                        if(copy_from_user(&v, arg,sizeof(v)))
                                return -EFAULT;
                        v.flags= nt1003->vchan.flags;
                        v.tuners= nt1003->vchan.tuners;
                        v.type= nt1003->vchan.type;
                        v.norm = nt1003->vchan.norm;
                        if (v.channel>=nt1003->video_inputs)
                                return -EINVAL;
                        if(v.channel==0){
				strcpy(v.name,"Television");
			        v.flags|=VIDEO_VC_TUNER;
			        v.type=VIDEO_TYPE_TV;
			        v.tuners=1;
			}
			else if(v.channel==1)
			        strcpy(v.name,"S-Video");
			else
				sprintf(v.name,"Composite%d",v.channel);

			if(copy_to_user(arg,&v,sizeof(v)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCSCHAN:
		{
                        struct video_channel v;
                        if(copy_from_user(&v, arg,sizeof(v)))
                                return -EFAULT;

                        if (v.channel>nt1003->video_inputs)
                                return -EINVAL;

                        nt1003->vchan.channel=v.channel;
                        nt1003->vchan.norm = v.norm;
                        down(&nt1003->lock);
//                        nt1003_muxsel(nt1003, v.channel, v.norm);
                        up(&nt1003->lock);
                        return 0;
		}
		case VIDIOCGPICT:
		{
			if (copy_to_user(arg, &nt1003->vpic,
					sizeof(nt1003->vpic)))
				return -EFAULT;
			return 0;
		}
		case VIDIOCSPICT:
		{
			struct video_picture p;
			if (copy_from_user(&p, arg, sizeof(p)))
				return -EFAULT;
			call_i2c_clients(nt1003, DECODER_SET_PICTURE, &p);
//			nt1003_adjust_picture(nt1003, p);

			return 0;
		}
		case VIDIOCSWIN:
		{
			/* PLEX: repasar */
			struct video_window vw;

			if (copy_from_user(&vw, arg, sizeof(vw)))
				return -EFAULT;
			return 0;
			if (vw.flags)
				return -EINVAL;
			if (vw.clipcount)
				return -EINVAL;
			if (vw.height != imgheight)
				return -EINVAL;
			if (vw.width != imgwidth)
				return -EINVAL;

			nt1003->compress = 0;

			return 0;
		}
		case VIDIOCGWIN:
		{
			struct video_window vw;

			vw.x = 0;
			vw.y = 0;
			vw.width = imgwidth;
			vw.height = imgheight;
			vw.chromakey = 0;
			vw.flags = nt1003_calculate_fps();

			if (copy_to_user(arg, &vw, sizeof(vw)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCGMBUF:
		{
			struct video_mbuf vm;

			memset(&vm, 0, sizeof(vm));
			vm.size = MAX_FRAME_SIZE * 2;
			vm.frames = 2;
			vm.offsets[0] = 0;
			vm.offsets[1] = MAX_FRAME_SIZE;

			if (copy_to_user((void *)arg, (void *)&vm, sizeof(vm)))
				return -EFAULT;

			return 0;
		}
		case VIDIOCMCAPTURE:
		{
			struct video_mmap vm;

			if (copy_from_user((void *)&vm, (void *)arg,sizeof(vm)))
				return -EFAULT;

			if (debug >= 1)
				printk(KERN_DEBUG
				       "frame: %d, size: %dx%d, format: %d\n",
				       vm.frame, vm.width, vm.height,
				       vm.format);

			if (vm.format != VIDEO_PALETTE_RGB24)
				return -EINVAL;

			if (vm.frame >= NT1003_NUMFRAMES)
				return -EINVAL;

			if (nt1003->frame[vm.frame].grabstate == FRAME_GRABBING)
				return -EBUSY;

			/* Don't compress if the size changed */
			if ((nt1003->frame[vm.frame].width != vm.width) ||
			    (nt1003->frame[vm.frame].height != vm.height)){
				nt1003->compress = 0;
				nt1003_set_output_size(nt1003, vm.width, 
						       vm.height);
			}

			nt1003->frame[vm.frame].width = vm.width;
			nt1003->frame[vm.frame].height = vm.height;

			/* Mark it as ready */
			nt1003->frame[vm.frame].grabstate = FRAME_READY;

			return nt1003_new_frame(nt1003, vm.frame);
		}
		case VIDIOCSYNC:
		{
			int frame;

			if (copy_from_user((void *)&frame, arg, sizeof(int)))
				return -EFAULT;

			if (debug >= 1)
				printk(KERN_DEBUG
				       "nt1003: syncing to frame %d\n", frame);

			switch (nt1003->frame[frame].grabstate) {
			case FRAME_UNUSED:
				return -EINVAL;
			case FRAME_READY:
			case FRAME_GRABBING:
			case FRAME_ERROR:
			{
				int ntries;
		redo:
				if (!NT1003_IS_OPERATIONAL(nt1003))
					return -EIO;
				ntries = 0; 
				do {
					interruptible_sleep_on(&nt1003->frame[frame].wq);
					if (signal_pending(current)) {
						if (flags & FLAGS_RETRY_VIDIOCSYNC) {
							/* Polling apps will destroy frames with that! */
							nt1003_new_frame(nt1003, frame);
							nt1003_testpattern(nt1003, 1, 0);
							nt1003->curframe = -1;
							nt1003->frame_num++;

							/* This will request another frame. */
							if (waitqueue_active(&nt1003->frame[frame].wq))
								wake_up_interruptible(&nt1003->frame[frame].wq);
							return 0;
						} else {
							/* Standard answer: not ready yet! */
							return -EINTR;
						}
					}
				} while (nt1003->frame[frame].grabstate == FRAME_GRABBING);

				if (nt1003->frame[frame].grabstate == FRAME_ERROR) {
					int ret = nt1003_new_frame(nt1003, frame);
					if (ret < 0)
						return ret;
					goto redo;
				}
			}
			case FRAME_DONE:
				nt1003->frame[frame].grabstate = FRAME_UNUSED;
				break;
			}

			nt1003->frame[frame].grabstate = FRAME_UNUSED;

			return 0;
		}
		case VIDIOCGFBUF:
		{
			struct video_buffer vb;

			memset(&vb, 0, sizeof(vb));
			vb.base = NULL;	/*frame buffer not supported, not used*/

			if (copy_to_user((void *)arg, (void *)&vb, sizeof(vb)))
				return -EFAULT;

 			return 0;
 		}
		case VIDIOCKEY:
		{
			return 0;
		}
		case VIDIOCCAPTURE:
		{
			return -EINVAL;
		}
		case VIDIOCSFBUF:
		{
			return -EINVAL;
		}
		case VIDIOCGTUNER:
		{ 
			struct video_tuner v;
		        if (copy_from_user(&v, arg,sizeof(v)))
		        	return -EFAULT;
	                
	                if (!nt1003->have_tuner || v.tuner) // Only tuner 0
	                        return -EINVAL;

	                strcpy(v.name,"Television");
	                v.rangelow  = 0;
	                v.rangehigh = ~0;
	                v.flags     = VIDEO_TUNER_PAL|VIDEO_TUNER_NTSC|
				      VIDEO_TUNER_SECAM;
	                v.mode      = nt1003->vchan.norm;
	                v.signal    = 0xFFFF; /* unknown */

			call_i2c_clients(nt1003, cmd, &v);

	                if (copy_to_user(arg,&v,sizeof(v)))
	                        return -EFAULT;

		        return 0;
		}
		case VIDIOCSTUNER:
		{
			struct video_tuner v;
		        if (copy_from_user(&v, arg, sizeof(v)))
		        	return -EFAULT;
										                         /* Only no or one tuner for now */
	                 if (!nt1003->have_tuner || v.tuner)
	                         return -EINVAL;

	                 /* and it only has certain valid modes */
	                 if( v.mode != VIDEO_MODE_PAL &&
	                     v.mode != VIDEO_MODE_NTSC &&
                             v.mode != VIDEO_MODE_SECAM)
	                         return -EOPNOTSUPP;

	                 return nt1003_muxsel(nt1003, v.tuner, v.mode);

		}
		case VIDIOCGFREQ:
		{
			unsigned long v=nt1003->freq;
		        if(copy_to_user(arg,&v,sizeof(v)))
			        return -EFAULT;
		        return 0;
		}
		case VIDIOCSFREQ:
		{
		        unsigned long v;
		        if(copy_from_user(&v, arg, sizeof(v)))
		                return -EFAULT;
		        nt1003->freq=v;
		        call_i2c_clients(nt1003,cmd,&v);
		        return 0;
		}

		/* TODO audio support */
		case VIDIOCGAUDIO:
		case VIDIOCSAUDIO:
			return 0;
		/* END TODO */

		/* Should I do CGCAPTURE and CSCAPTURE */

		default:
			return -ENOIOCTLCMD;
	}
	return 0;
}

static long 
nt1003_read(struct video_device *dev, char *buf, unsigned long count,
	    int noblock)
{
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *)dev;
	int frmx = -1;
	volatile struct nt1003_frame *frame;

	if (debug >= 1)
		printk(KERN_DEBUG "nt1003_read: %ld bytes, noblock=%d\n",
				count, noblock);

	if (!NT1003_IS_OPERATIONAL(nt1003) || (buf == NULL))
		return -EFAULT;

	/* See if a frame is completed, then use it. */
	if (nt1003->frame[0].grabstate >= FRAME_DONE)	/* _DONE or _ERROR */
		frmx = 0;
	else if (nt1003->frame[1].grabstate >= FRAME_DONE)/* _DONE or _ERROR */
		frmx = 1;

	if (noblock && (frmx == -1))
		return -EAGAIN;

	/* If no FRAME_DONE, look for a FRAME_GRABBING state. */
	/* See if a frame is in process (grabbing), then use it. */
	if (frmx == -1) {
		if (nt1003->frame[0].grabstate == FRAME_GRABBING)
			frmx = 0;
		else if (nt1003->frame[1].grabstate == FRAME_GRABBING)
			frmx = 1;
	}

	/* If no frame is active, start one. */
	if (frmx == -1)
		nt1003_new_frame(nt1003, frmx = 0);

	frame = &nt1003->frame[frmx];

restart:
	if (!NT1003_IS_OPERATIONAL(nt1003))
		return -EIO;
	while (frame->grabstate == FRAME_GRABBING) {
		interruptible_sleep_on((void *)&frame->wq);
		if (signal_pending(current))
			return -EINTR;
	}

	if (frame->grabstate == FRAME_ERROR) {
		frame->bytes_read = 0;
		if (nt1003_new_frame(nt1003, frmx))
			printk(KERN_ERR "nt1003_read: nt1003_new_frame error\n");
		goto restart;
	}

	if (debug >= 1)
		printk(KERN_DEBUG "nt1003_read: frmx=%d, bytes_read=%ld, scanlength=%ld\n",
			frmx, frame->bytes_read, frame->scanlength);

	/* copy bytes to user space; we allow for partials reads */
	if ((count + frame->bytes_read) > frame->scanlength)
		count = frame->scanlength - frame->bytes_read;

	if (copy_to_user(buf, frame->data + frame->bytes_read, count))
		return -EFAULT;

	frame->bytes_read += count;
	if (debug >= 1)
		printk(KERN_DEBUG "nt1003_read: {copy} count used=%ld, new bytes_read=%ld\n",
			count, frame->bytes_read);

	if (frame->bytes_read >= frame->scanlength) { /*All data has been read*/
		frame->bytes_read = 0;

		/* Mark it as available to be used again. */
		nt1003->frame[frmx].grabstate = FRAME_UNUSED;
		if (nt1003_new_frame(nt1003, frmx ? 0 : 1))
			printk(KERN_ERR "nt1003_read: nt1003_new_frame returned error\n");
	}

	return count;
}

static int
nt1003_mmap(struct video_device *dev, const char *adr, unsigned long size)
{
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *)dev;
	unsigned long start = (unsigned long)adr;
	unsigned long page, pos;

	if (!NT1003_IS_OPERATIONAL(nt1003))
		return -EFAULT;

	if (size > (((2 * MAX_FRAME_SIZE) + PAGE_SIZE - 1) & ~(PAGE_SIZE - 1)))
		return -EINVAL;

	pos = (unsigned long)nt1003->fbuf;
	while (size > 0) {
		page = kvirt_to_pa(pos);
		if (remap_page_range(start, page, PAGE_SIZE, PAGE_SHARED))
			return -EAGAIN;

		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return 0;
}

static struct video_device nt1003_template = {
	"NT1003 USB Video",
	VID_TYPE_TUNER|VID_TYPE_CAPTURE,
	VID_HARDWARE_CPIA,	/* FIXME */
	nt1003_open,
	nt1003_close,
	nt1003_read,
	nt1003_write,
	NULL,
	nt1003_ioctl,
	nt1003_mmap,
	nt1003_init_done,
	NULL,
	0,
	0
};

static void
nt1003_configure_video(struct usb_nt1003 *nt1003)
{
	if (nt1003 == NULL)
		return;

	RESTRICT_TO_RANGE(init_brightness, 0, 255);
	RESTRICT_TO_RANGE(init_contrast, 0, 255);
	RESTRICT_TO_RANGE(init_color, 0, 255);
	RESTRICT_TO_RANGE(init_hue, 0, 255);
	RESTRICT_TO_RANGE(hue_correction, 0, 255);

	memset(&nt1003->vpic, 0, sizeof(nt1003->vpic));

	nt1003->vpic.colour = init_color << 8;
	nt1003->vpic.hue = init_hue << 8;
	nt1003->vpic.brightness = init_brightness << 8;
	nt1003->vpic.contrast = init_contrast << 8;
	nt1003->vpic.whiteness = 105 << 8; /* This one isn't used */
	nt1003->vpic.depth = 24;
	nt1003->vpic.palette = VIDEO_PALETTE_RGB24;

	memset(&nt1003->vcap, 0, sizeof(nt1003->vcap));
	strcpy(nt1003->vcap.name, "NT1003 USB Video");
	nt1003->vcap.type = VID_TYPE_CAPTURE | VID_TYPE_SCALES | 
			    VID_TYPE_SUBCAPTURE |
			    (nt1003->have_tuner ? VID_TYPE_TUNER : 0);;
	nt1003->vcap.channels = 3; /* FIXME: Is this true */
	nt1003->vcap.audios = 0; /* Still no audio support */
	nt1003->vcap.maxwidth = max_imgwidth;
	nt1003->vcap.maxheight = max_imgheight;
	nt1003->vcap.minwidth = min_imgwidth;
	nt1003->vcap.minheight = min_imgheight;

	memset(&nt1003->vchan, 0, sizeof(nt1003->vchan));
	nt1003->vchan.flags = 0;  //TODO: VIDIO_VC_AUDIO
	nt1003->vchan.tuners = 0;
	nt1003->vchan.channel = 0;
	nt1003->vchan.type = VIDEO_TYPE_CAMERA;
	nt1003->vchan.norm = norm;
	nt1003->video_inputs = 3;		/* FIXME: model dependant */

	/* This should be here to i2c clients to work */
	nt1003_power_on(nt1003);
	nt1003_init_i2c(nt1003);
}

/*
 * nt1003_find_struct()
 *
 * This code searches the array of preallocated (static) structures
 * and returns index of the first one that isn't in use. Returns -1
 * if there are no free structures.
 *
 */
static int
nt1003_find_struct(void)
{
	int i, u;

	for (u = 0; u < MAX_NT1003; u++) {
		struct usb_nt1003 *nt1003 = &vdevs[u];
		if (!nt1003->nt1003_used) /* This one is free */
		{
			nt1003->nt1003_used = 1;	/* In use now */
			for (i=0; i < NT1003_NUMFRAMES; i++)
				init_waitqueue_head(&nt1003->frame[i].wq);
			init_MUTEX(&nt1003->lock);	/* to 1 == available */
			nt1003->dev = NULL;
			memcpy(&nt1003->vdev, &nt1003_template,
			       sizeof(nt1003_template));
			return u;
		}
	}
	return -1;
}

/*
 * nt1003_probe()
 *
 * This procedure queries device descriptor and accepts the interface
 * if it looks like IBM C-it camera.
 *
 */
static void *
nt1003_probe(struct usb_device *dev, unsigned int ifnum)
{
	struct usb_nt1003 *nt1003 = NULL;
	const struct usb_interface_descriptor *interface;
	const struct usb_endpoint_descriptor *endpoint;
	int devnum, model=0;

	if (debug >= 1)
		printk(KERN_DEBUG "nt1003_probe(%p,%u.)\n", dev, ifnum);

	/* We don't handle multi-config */
	if (dev->descriptor.bNumConfigurations != 1)
		return NULL;

	/* Is it an NT1003 video dev */
	if ((dev->descriptor.idVendor == 0x2304) &&
	    (dev->descriptor.idProduct == 0x0111)){
		model = NT1003_MIRO;
	        printk(KERN_INFO "nt1003: MIRO PCTV USB found\n");
	}
	else if ((dev->descriptor.idVendor == 0x0573) &&
	        (dev->descriptor.idProduct == 0x04d11)){
		model = NT1003_HAUPPAUGE;
	        printk(KERN_INFO "nt1003: HAUPPAUGE WINTV-USB found\n");
	}
	else
		return NULL;

	interface = &dev->actconfig->interface[ifnum].altsetting[0];
	if (interface->bNumEndpoints != 2) {
		printk(KERN_ERR "NT1003: interface %d. has %u. endpoints!\n",
		       ifnum, (unsigned)(interface->bNumEndpoints));
		return NULL;
	}
	endpoint = &interface->endpoint[1];
	if ((endpoint->bmAttributes & 0x03) != 0x01) {
		printk(KERN_ERR "NT1003: interface %d. has non-ISO endpoint!\n"
				, ifnum);
		return NULL;
	}
	if ((endpoint->bEndpointAddress & 0x80) == 0) {
		printk(KERN_ERR "NT1003: interface %d. has ISO OUT endpoint!\n"
				, ifnum);
		return NULL;
	}

	/* Validate options */
//	if ( (model == NT1003_MIRO) || (model == NT1003_HAUPPAUGE) ){
//		RESTRICT_TO_RANGE(lighting, 0, 2);
//		RESTRICT_TO_RANGE(videosize, VIDEOSIZE_128x96,
//				  VIDEOSIZE_352x288);
//	} 

	devnum = nt1003_find_struct();
	if (devnum == -1) {
		printk(KERN_INFO "NT1003 USB driver: Too many devices!\n");
		return NULL;
	}
	nt1003 = &vdevs[devnum];
	nt1003->nr = devnum;
	if( (model == NT1003_HAUPPAUGE) || (model == NT1003_MIRO) ){
		nt1003->have_tuner= 1;
		nt1003->tuner_type= tuner;
		nt1003->have_videodecoder= 1;
	}

	down(&nt1003->lock);
	nt1003->vdev_model = model;
	nt1003->remove_pending = 0;
	nt1003->last_error = 0;
	nt1003->dev = dev;
	nt1003->iface = ifnum;
	nt1003->ifaceAltInactive = 0;
	nt1003->ifaceAltActive = 1;
	nt1003->video_endp = endpoint->bEndpointAddress;
	nt1003->iso_packet_len = 959; //1023; // 1014 ????? 959 ??????
	nt1003->compress = 0;
	nt1003->user=0; 

	nt1003_configure_video(nt1003);
	up (&nt1003->lock);

#if NT1003_LOCKS_DRIVER_WHILE_DEVICE_IS_PLUGGED
	MOD_INC_USE_COUNT;
#endif

	if (video_register_device(&nt1003->vdev, VFL_TYPE_GRABBER) == -1) {
		printk(KERN_ERR "video_register_device failed\n");
		return NULL;
	}
	if (debug > 1)
		printk(KERN_DEBUG "video_register_device() successful\n");

	return nt1003;
}

/*
 * nt1003_release()
 *
 * This code does final release of struct usb_nt1003. This happens
 * after the device is disconnected -and- all clients closed their files.
 *
 */
static void 
nt1003_release(struct usb_nt1003 *nt1003)
{
	video_unregister_device(&nt1003->vdev);

	i2c_usb_del_bus(&nt1003->i2c_adap);

	nt1003_power_off(nt1003);

	if (debug > 0)
		printk(KERN_DEBUG "usb_nt1003_release: Video unregistered.\n");
	nt1003->nt1003_used = 0;
}

/*
 * nt1003_disconnect()
 *
 * This procedure stops all driver activity, deallocates interface-private
 * structure (pointed by 'ptr') and after that driver should be removable
 * with no ill consequences.
 *
 */
static void 
nt1003_disconnect(struct usb_device *dev, void *ptr)
{
	static const char proc[] = "nt1003_disconnect";
	struct usb_nt1003 *nt1003 = (struct usb_nt1003 *) ptr;

	if (debug > 0)
		printk(KERN_DEBUG "%s(%p,%p.)\n", proc, dev, ptr);

	down(&nt1003->lock);
	nt1003->remove_pending = 1; /* Now all ISO data will be ignored */

	/* At this time we ask to cancel outstanding URBs */
	nt1003_stop_isoc(nt1003);

	nt1003->dev = NULL;    	    /* USB device is no more */

#if NT1003_LOCKS_DRIVER_WHILE_DEVICE_IS_PLUGGED
	MOD_DEC_USE_COUNT;
#endif
	if (nt1003->user)
		printk(KERN_INFO "%s: In use, disconnect pending.\n", proc);
	else
		nt1003_release(nt1003);
	up(&nt1003->lock);

	printk(KERN_INFO "NT1003 USB video disconnected.\n");
}

static struct usb_driver nt1003_driver = {
	"nt1003",
	nt1003_probe,
	nt1003_disconnect,
	{ NULL, NULL }
};

/*
 * nt1003_init()
 *
 * This code is run to initialize the driver.
 *
 */
int
nt1003_init(void)
{
	unsigned u;

	/* Initialize struct */
	for (u = 0; u < MAX_NT1003; u++) {
		struct usb_nt1003 *nt1003 = &vdevs[u];
		memset (nt1003, 0, sizeof(struct usb_nt1003));
	}
	return usb_register(&nt1003_driver);
}

void nt1003_cleanup(void)
{
	usb_deregister(&nt1003_driver);
}

#ifdef MODULE
int init_module(void)
{
	return nt1003_init();
}

void cleanup_module(void)
{
	nt1003_cleanup();
}
#endif
