#ifndef __LINUX_NT1003_H
#define __LINUX_NT1003_H

#include <linux/list.h>

#define NT1003_PWR_REG				0x00
#define NT1003_CONFIG_REG			0x01
#define NT1003_ADRS_REG				0x02

#define NT1003_STATUS_REG			0x05
#define NT1003_IOPIN_REG			0x06
#define NT1003_SER_MODE				0x07
#define NT1003_SER_ADRS				0x08
#define NT1003_SER_CONT				0x09
#define NT1003_SER_DAT1				0x0A
#define NT1003_SER_DAT2				0x0B
#define NT1003_SER_DAT3				0x0C
#define NT1003_SER_DAT4				0x0D
#define NT1003_EE_DATA				0x0E
#define NT1003_EE_LSBAD				0x0F
#define NT1003_EE_CONT				0x10

#define NT1003_DRM_CONT				0x12
#define NT1003_DRM_PRM1				0x13
#define NT1003_DRM_PRM2				0x14
#define NT1003_DRM_PRM3				0x15
#define NT1003_DRM_PRM4				0x16
#define NT1003_DRM_PRM5				0x17
#define NT1003_DRM_PRM6				0x18
#define NT1003_DRM_PRM7				0x19
#define NT1003_DRM_PRM8				0x1A
#define NT1003_VIN_REG1				0x1B
#define NT1003_VIN_REG2				0x1C

#define NT1003_LXSIZE_I				0x1D
#define NT1003_MXSIZE_I				0x1E
#define NT1003_LYSIZE_I				0x1F
#define NT1003_MYSIZE_I				0x20
#define NT1003_LX_OFFST				0x21
#define NT1003_MX_OFFST				0x22
#define NT1003_LY_OFFST				0x23
#define NT1003_MY_OFFST				0x24
#define NT1003_FRM_RATE				0x25
#define NT1003_LXSIZE_O				0x26
#define NT1003_MXSIZE_O				0x27
#define NT1003_LYSIZE_O				0x28
#define NT1003_MYSIZE_O				0x29
#define NT1003_FILT_CONT			0x2A
#define NT1003_VO_MODE				0x2B
#define NT1003_INTRA_CYC			0x2C
#define NT1003_STRIP_SZ				0x2D
#define NT1003_FORCE_INTRA			0x2E
#define NT1003_FORCE_UP				0x2F
#define NT1003_BUF_THR				0x30
#define NT1003_DVI_YUV				0x31

#define NT1003_PCM_THR1				0x38
#define NT1003_PCM_THR2				0x39
#define NT1003_DIST_THR_L			0x3A
#define NT1003_DIST_THR_H			0x3B
#define NT1003_MAX_DIST_L			0x3C
#define NT1003_MAX_DIST_H			0x3D




#define USES_NT1003_PUTPIXEL    0       /* 0=Fast/oops 1=Slow/secure */

/* Video Size 384 x 288 x 3 bytes for RGB */
/* 384 because xawtv tries to grab 384 even though we tell it 352 is our max */
#define V4L_FRAME_WIDTH         384
#define V4L_FRAME_WIDTH_USED	352
#define V4L_FRAME_HEIGHT        288
#define V4L_BYTES_PER_PIXEL     3
#define MAX_FRAME_WIDTH		480
#define MAX_FRAME_HEIGHT	640
#define MAX_FRAME_SIZE          (MAX_FRAME_WIDTH * MAX_FRAME_HEIGHT * \
		                 V4L_BYTES_PER_PIXEL)

/* Camera capabilities (maximum) */
#define CAMERA_IMAGE_WIDTH      640
#define CAMERA_IMAGE_HEIGHT     480
#define CAMERA_IMAGE_LINE_SZ    ((CAMERA_IMAGE_WIDTH * 3) / 2) /* Bytes */
#define CAMERA_URB_FRAMES       128
#define CAMERA_MAX_ISO_PACKET   959 // Doc say 1023 ????

#define NT1003_NUMFRAMES	2
#define NT1003_NUMSBUF		2

#define FRAMES_PER_DESC		(CAMERA_URB_FRAMES)
#define FRAME_SIZE_PER_DESC	(CAMERA_MAX_ISO_PACKET)

/* This macro restricts an int variable to an inclusive range */
#define RESTRICT_TO_RANGE(v,mi,ma) { if ((v) < (mi)) (v) = (mi); else if ((v) > (ma)) (v) = (ma); }

/*
 * This macro performs bounds checking - use it when working with
 * new formats, or else you may get oopses all over the place.
 * If pixel falls out of bounds then it gets shoved back (as close
 * to place of offence as possible) and is painted bright red.
 */
#define NT1003_PUTPIXEL(fr, ix, iy, vr, vg, vb) { \
	register unsigned char *pf; \
	int limiter = 0, mx, my; \
	mx = ix; \
	my = iy; \
	if (mx < 0) { \
		mx=0; \
		limiter++; \
	} else if (mx >= 352) { \
		mx=351; \
		limiter++; \
	} \
	if (my < 0) { \
		my = 0; \
		limiter++; \
	} else if (my >= V4L_FRAME_HEIGHT) { \
		my = V4L_FRAME_HEIGHT - 1; \
		limiter++; \
	} \
	pf = (fr)->data + V4L_BYTES_PER_PIXEL*((iy)*352 + (ix)); \
	if (limiter) { \
		*pf++ = 0; \
		*pf++ = 0; \
		*pf++ = 0xFF; \
	} else { \
		*pf++ = (vb); \
		*pf++ = (vg); \
		*pf++ = (vr); \
	} \
}

/*
 * We use macros to do YUV -> RGB conversion because this is
 * very important for speed and totally unimportant for size.
 *
 * YUV -> RGB Conversion
 * ---------------------
 *
 * B = 1.164*(Y-16)		    + 2.018*(V-128)
 * G = 1.164*(Y-16) - 0.813*(U-128) - 0.391*(V-128)
 * R = 1.164*(Y-16) + 1.596*(U-128)
 *
 * If you fancy integer arithmetics (as you should), hear this:
 *
 * 65536*B = 76284*(Y-16)		  + 132252*(V-128)
 * 65536*G = 76284*(Y-16) -  53281*(U-128) -  25625*(V-128)
 * 65536*R = 76284*(Y-16) + 104595*(U-128)
 *
 * Make sure the output values are within [0..255] range.
 */
#define LIMIT_RGB(x) (((x) < 0) ? 0 : (((x) > 255) ? 255 : (x)))
#define YUV_TO_RGB_BY_THE_BOOK(my,mu,mv,mr,mg,mb) { \
    int mm_y, mm_yc, mm_u, mm_v, mm_r, mm_g, mm_b; \
    mm_y = (my) - 16;  \
    mm_u = (mu) - 128; \
    mm_v = (mv) - 128; \
    mm_yc= mm_y * 76284; \
    mm_b = (mm_yc		+ 132252*mm_v	) >> 16; \
    mm_g = (mm_yc -  53281*mm_u -  25625*mm_v	) >> 16; \
    mm_r = (mm_yc + 104595*mm_u			) >> 16; \
    mb = LIMIT_RGB(mm_b); \
    mg = LIMIT_RGB(mm_g); \
    mr = LIMIT_RGB(mm_r); \
}

/* Debugging aid */
#define NT1003_SAY_AND_WAIT(what) { \
	wait_queue_head_t wq; \
	init_waitqueue_head(&wq); \
	printk(KERN_INFO "Say: %s\n", what); \
	interruptible_sleep_on_timeout (&wq, HZ*3); \
}

/*
 * This macro checks if nt1003 is still operational. The 'nt1003'
 * pointer must be valid, nt1003->dev must be valid, we are not
 * removing the device and the device has not erred on us.
 */
#define NT1003_IS_OPERATIONAL(udevice) (\
	(udevice != NULL) && \
	((udevice)->dev != NULL) && \
	((udevice)->last_error == 0) && \
	(!(udevice)->remove_pending))

enum {
	STATE_SCANNING,		/* Scanning for header */
	STATE_LINES,		/* Parsing lines */
};

enum {
	FRAME_UNUSED,		/* Unused (no MCAPTURE) */
	FRAME_READY,		/* Ready to start grabbing */
	FRAME_GRABBING,		/* In the process of being grabbed into */
	FRAME_DONE,		/* Finished grabbing, but not been synced yet */
	FRAME_ERROR,		/* Something bad happened while processing */
};

struct usb_device;

struct nt1003_sbuf {
	char *data;
	urb_t *urb;
};

#define NT1003_MAGIC      0xAA55
struct nt1003_frame_header {
        unsigned char vid_frm_patt[2];     /* 0 - 1  magic */
        unsigned char header_length;    /* 2 */
        unsigned char frame_numb;       /* 3 */
        unsigned char frame_phase;      /* 4 */
        unsigned char frame_latency;    /* 5 */
        unsigned char data_format;      /* 6 */
        unsigned char format_param;     /* 7 */
        __u16 frame_width;   /* 8 - 9 */
        __u16 frame_height;  /* 10 - 11 */
};

struct nt1003_regvals
{
	unsigned char reg;
	unsigned char value;
};

struct nt1003_frame {
	char *data;		/* Frame buffer */
	struct nt1003_frame_header header;	/* Header from stream */
	int order_uv;		/* True=UV False=VU */
	int order_yc;		/* True=Yc False=cY ('c'=either U or V) */
	unsigned char hdr_sig;	/* "00 FF 00 ??" where 'hdr_sig' is '??' */

	int width;		/* Width application is expecting */
	int height;		/* Height */

	int frmwidth;		/* Width the frame actually is */
	int frmheight;		/* Height */

	volatile int grabstate;	/* State of grabbing */
	int scanstate;		/* State of scanning */

	int curline;		/* Line of frame we're working on */

	long scanlength;	/* uncompressed, raw data length of frame */
	long bytes_read;	/* amount of scanlength that has been read 
				   from *data */

	wait_queue_head_t wq;	/* Processes waiting */
};

#define	NT1003_MIRO		1	/* MIRO PCTV USB */
#define NT1003_HAUPPAUGE	2	/* HAUPPAUGE WINTV-USB */
#define I2C_CLIENTS_MAX		8

struct usb_nt1003 {
	struct video_device vdev;
//	struct video_devide rdev;		/* Radio dev */
	struct video_audio audio_dev;		/* Current audio params */

	/* i2c */
	struct i2c_adapter         i2c_adap;
	struct i2c_algo_usb_data   i2c_algo;
	struct i2c_client          i2c_client;
	int                        i2c_state, i2c_ok;
	struct i2c_client         *i2c_clients[I2C_CLIENTS_MAX];

	int have_videodecoder;
	int have_tuner;
	int tuner_type;
	int channel;
	int video_inputs;	/* # of inputs */
	unsigned long freq;

	unsigned int nr;	/* Number of the device < MAX_NT1003 */

	/* Device structure */
	struct usb_device *dev;

	unsigned char iface;                  /* Video interface number */
	unsigned char ifaceAltActive, ifaceAltInactive; /* Alt settings */

	struct semaphore lock;
	int user;		/* user count for exclusive use */

	int nt1003_used;        /* Is this structure in use? */
	int initialized;	/* Had we already sent init sequence? */
	int vdev_model;		/* What type of NT1003 video dev we got? */
	int streaming;		/* Are we streaming Isochronous? */
	int grabbing;		/* Are we grabbing? */
	int last_error;		/* What calamity struck us? */

	int compress;		/* Should the next frame be compressed? */

	char *fbuf;		/* Videodev buffer area */
	int fbuf_size;		/* Videodev buffer size */

	int curframe;
	struct nt1003_frame frame[NT1003_NUMFRAMES];	/* Double buffering */

	int cursbuf;		/* Current receiving sbuf */
	struct nt1003_sbuf sbuf[NT1003_NUMSBUF];	/* Double buffering */
	volatile int remove_pending;	/* If set then about to exit */

        /*
	 * Scratch space from the Isochronous pipe.
	 * Scratch buffer should contain at least one pair of lines
	 * (CAMERA_IMAGE_LINE_SZ). We set it to two pairs here.
	 * This will be approximately 2 KB. HOWEVER in reality this
	 * buffer must be as large as hundred of KB because otherwise
	 * you'll get lots of overflows because V4L client may request
	 * frames not as uniformly as USB sources them.
	 */
	unsigned char *scratch;
	int scratchlen;

	struct video_picture vpic;	/* Picture settings */
	struct video_capability vcap;	/* Video capabilities */
	struct video_channel vchan;	/* May be used for tuner support */
	unsigned char video_endp;	/* 0x82 for NT1003 devices based */
        int has_hdr;
        int frame_num;
	int iso_packet_len;	/* Videomode-dependent, saves bus bandwidth */

	/* Statistics that can be overlayed on screen */
        unsigned long urb_count;        /* How many URBs we received so far */
        unsigned long urb_length;       /* Length of last URB */
        unsigned long data_count;       /* How many bytes we received */
        unsigned long header_count;     /* How many frame headers we found */
	unsigned long scratch_ovf_count;/* How many times we overflowed
					   scratch */
	unsigned long iso_skip_count;	/* How many empty ISO packets
					   received */
	unsigned long iso_err_count;	/* How many bad ISO packets received */
};

#endif /* __LINUX_NT1003_H */
