/*
 * AOX se401 Camera-to-USB Bridge Driver
 * Copyright (c) 2000 Jeroen B. Vreeken
 *
 * Based on the Linux CPiA and OV511 driver.
 * 
 * Released under GPL v.2 license.
 *
 * Important keywords in comments:
 *    CAMERA SPECIFIC - Camera specific code; may not work with other cameras.
 *    DEBUG - Debugging code.
 *    FIXME - Something that is broken or needs improvement.
 *
 */

/*
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __NO_VERSION__

/* Handle mangled (versioned) external symbols */

#include <linux/config.h>   /* retrieve the CONFIG_* macros */
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#	define MODVERSIONS  /* force it on */
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/list.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/smp_lock.h>
#include <linux/videodev.h>
#include <linux/vmalloc.h>
#include <linux/wrapper.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <asm/io.h>

#include <usb.h>
#include "se401.h"

/* Video Size 640 x 480 x 3 bytes for RGB */
#define MAX_FRAME_SIZE (640 * 480 * 3)
#define MAX_DATA_SIZE (MAX_FRAME_SIZE + sizeof(struct timeval))

// FIXME - Should find a better way to do this.
#define DEFAULT_WIDTH 640
#define DEFAULT_HEIGHT 480

char kernel_version[] = UTS_RELEASE;

/*******************************/
/* Memory management functions */
/*******************************/

#define MDEBUG(x)	do { } while(0)		/* Debug memory management */

static struct usb_driver se401_driver;

/* Given PGD from the address space's page table, return the kernel
 * virtual mapping of the physical memory mapped at ADR.
 */
static inline unsigned long uvirt_to_kva(pgd_t *pgd, unsigned long adr)
{
	unsigned long ret = 0UL;
	pmd_t *pmd;
	pte_t *ptep, pte;

	if (!pgd_none(*pgd)) {
		pmd = pmd_offset(pgd, adr);
		if (!pmd_none(*pmd)) {
			ptep = pte_offset(pmd, adr);
			pte = *ptep;
			if (pte_present(pte))
				ret = page_address(pte_page(pte)) | (adr & (PAGE_SIZE-1));
		}
	}
	MDEBUG(printk("uv2kva(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long uvirt_to_bus(unsigned long adr)
{
	unsigned long kva, ret;

	kva = uvirt_to_kva(pgd_offset(current->mm, adr), adr);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("uv2b(%lx-->%lx)", adr, ret));
	return ret;
}

static inline unsigned long kvirt_to_bus(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = virt_to_bus((void *)kva);
	MDEBUG(printk("kv2b(%lx-->%lx)", adr, ret));
	return ret;
}

/* Here we want the physical address of the memory.
 * This is used when initializing the contents of the
 * area and marking the pages as reserved.
 */
static inline unsigned long kvirt_to_pa(unsigned long adr)
{
	unsigned long va, kva, ret;

	va = VMALLOC_VMADDR(adr);
	kva = uvirt_to_kva(pgd_offset_k(va), va);
	ret = __pa(kva);
	MDEBUG(printk("kv2pa(%lx-->%lx)", adr, ret));
	return ret;
}

static void *rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr, page;

	/* Round it off to PAGE_SIZE */
	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	mem = vmalloc(size);
	if (!mem)
		return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_reserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	return mem;
}

static void rvfree(void *mem, unsigned long size)
{
	unsigned long adr, page;

	if (!mem)
		return;

	size += (PAGE_SIZE - 1);
	size &= ~(PAGE_SIZE - 1);

	adr=(unsigned long) mem;
	while (size > 0) {
		page = kvirt_to_pa(adr);
		mem_map_unreserve(MAP_NR(__va(page)));
		adr += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}
	vfree(mem);
}

static void* se401_probe(struct usb_device *dev, unsigned int ifnum)
{
	struct usb_interface_descriptor *interface;
	struct usb_se401 *se401;
	purb_t urb_ptr=NULL;
	char *tbuffer;
	char *camera_name=NULL;
	int rc;

	PDEBUG("probing for device...");

	/* We don't handle multi-config cameras */
	if (dev->descriptor.bNumConfigurations != 1)
		return NULL;

	interface = &dev->actconfig->interface[ifnum].altsetting[0];

	/* Is it an se401? */
	if (dev->descriptor.idVendor == 0x03e8 &&
	    dev->descriptor.idProduct == 0x0004) {//FIXME: check this value!
		camera_name="Aox USB camera";
	} else if (dev->descriptor.idVendor == 0x0471 &&
	    dev->descriptor.idProduct == 0x030b) {
		camera_name="Philips PCVC665K USB VGA Camera";
	} else return NULL;
	printk(KERN_INFO "se401: %s found\n", camera_name);

	/* Checking vendor/product should be enough, but what the hell */
	if (interface->bInterfaceClass != 0x00) 
		return NULL;
	if (interface->bInterfaceSubClass != 0x00)
		return NULL;

	/* We found one */
	printk(KERN_INFO "se401: USB se401-based camera found\n");

	if ((se401 = kmalloc(sizeof(*se401), GFP_KERNEL)) == NULL) {
		err("couldn't kmalloc se401 struct");
		return NULL;
	}

	memset(se401, 0, sizeof(*se401));
	
	se401->dev = dev;
	se401->iface = interface->bInterfaceNumber;
	se401->camera_name = camera_name;



//	rc=usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
//	    0x57, 0, 1, 0, 0, 0, HZ);
//	rc=usb_control_msg(dev, usb_rcvctrlpipe(dev, 0),
//	    0x06, 3, 0, 0, buffer, 0x12, HZ);
//	printk("%02x %02x %02x %02x\n", buffer[0], buffer[1], buffer[2], buffer[3]);

//    	urb_ptr = usb_alloc_urb(0);
//	if (!urb_ptr) {
//		printk("se401: no urb pointer\n");
//		return NULL;
//	}
//	
//	tbuffer=kmalloc(0x40, GFP_KERNEL);
//	
//	FILL_CONTROL_URB(urb_ptr, dev, "\0xc1\0x4c\0x6f\0x0c", "\0x40\0x57\0x01\0x00\0x00\0x00\0x00\0x00",
//	    tbuffer, 0x40, NULL/*complete*/, NULL/*context*/);
//	if (usb_submit_urb(urb_ptr)) {
//		printk("se401: urb_submit error\n");
//		usb_free_urb (urb_ptr);
//		kfree(tbuffer);
//		return NULL;
//	}
//	while (urb_ptr->status!=0);
//	printk ("se401: yahoo!!!!!!!!!\n");
//	kfree(tbuffer);
//	usb_free_urb (urb_ptr);

     	return se401;
}

static void se401_disconnect(struct usb_device *dev, void *ptr)
{

	struct usb_se401 *se401 = (struct usb_se401 *) ptr;

////	video_unregister_device(&se401->vdev);

//	/* We don't want people trying to open up the device */
//	if (!se401->user)
//		video_unregister_device(&se401->vdev);
//
	usb_driver_release_interface(&se401_driver,
		&se401->dev->actconfig->interface[se401->iface]);

	se401->dev = NULL;
	se401->frame[0].grabstate = FRAME_ERROR;
	se401->frame[1].grabstate = FRAME_ERROR;
	se401->curframe = -1;

	/* This will cause the process to request another frame */
	if (waitqueue_active(&se401->frame[0].wq))
		wake_up_interruptible(&se401->frame[0].wq);
	if (waitqueue_active(&se401->frame[1].wq))
		wake_up_interruptible(&se401->frame[1].wq);
	if (waitqueue_active(&se401->wq))
		wake_up_interruptible(&se401->wq);

	se401->streaming = 0;

	/* Unschedule all of the iso td's */
	if (se401->sbuf[1].urb) {
		se401->sbuf[1].urb->next = NULL;
		usb_unlink_urb(se401->sbuf[1].urb);
		usb_free_urb(se401->sbuf[1].urb);
		se401->sbuf[1].urb = NULL;
	}
	if (se401->sbuf[0].urb) {
		se401->sbuf[0].urb->next = NULL;
		usb_unlink_urb(se401->sbuf[0].urb);
		usb_free_urb(se401->sbuf[0].urb);
		se401->sbuf[0].urb = NULL;
	}	
	printk("se401: %s disconnected\n", se401->camera_name);

	/* Free the memory */
	if (!se401->user) {
		kfree(se401);
		se401 = NULL;
	}
}

static struct usb_driver se401_driver = {
	"se401",
	se401_probe,
	se401_disconnect,
	{ NULL, NULL }
};

int usb_se401_init(void)
{
	PDEBUG("usb_se401_init()");
	
	EXPORT_NO_SYMBOLS;
	
	return usb_register(&se401_driver);
}

void usb_se401_cleanup(void)
{
	usb_deregister(&se401_driver);
}

#ifdef MODULE
int init_module(void)
{
	return usb_se401_init();
}

void cleanup_module(void)
{
	usb_se401_cleanup();
	
	PDEBUG("Module unloaded");
}
#endif

