/**********
This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the
Free Software Foundation; either version 3 of the License, or (at your
option) any later version. (See <http://www.gnu.org/copyleft/lesser.html>.)

This library is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
more details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
**********/
// Copyright (c) 1996-2023, Live Networks, Inc.  All rights reserved
// A common framework, used for the "openRTSP" and "playSIP" applications
// Implementation
//
// NOTE: If you want to develop your own RTSP client application (or embed RTSP client functionality into your own application),
// then we don't recommend using this code as a model, because it is too complex (with many options).
// Instead, we recommend using the "testRTSPClient" application code as a model.

#include "triggerRace.hh"
#include "BasicUsageEnvironment.hh"
#include "GroupsockHelper.hh"

#if defined(__WIN32__) || defined(_WIN32)
#define snprintf _snprintf
#else
#include <signal.h>
#define USE_SIGNALS 1
#endif

#include <iostream>
#include <thread>
#include <chrono>
#include <cstdlib>

// Forward function definitions:

char const* progName;
UsageEnvironment* env;

uint32_t trigger1Signalled; // Counter of how many time T1 has been triggered
uint32_t trigger2Signalled; // Counter of how many time T2 has been triggered
uint32_t trigger1Handled; // Counter of how many time T1 handler has been called
uint32_t trigger2Handled; // Counter of how many time T2 handler has been called
uint32_t statusCounter; // Counter to control how often to output status

struct timeval startTime;

void usage() {
  *env << "Usage: " << progName
<< "\n";
//  shutdown();
}

void printStats()
{
  *env << trigger1Signalled << ":" << trigger1Handled << " - " << trigger2Signalled << ":" << trigger2Handled << "\n";
}

void trigger1Handler(void *clientData)
{
//  *env << "Trigger1 handler\n";
  ++trigger1Handled;
}

void trigger2Handler(void *clientData)
{
//  *env << "Trigger2 handler\n";
  ++trigger2Handled;
}

void non_live_thread()
{
  std::chrono::microseconds duration1us(1);
  std::chrono::milliseconds duration20ms(20);
  std::this_thread::sleep_for(duration20ms); // Let event loop start
  
  
  *env << "Starting non-LIVE thread\n";
  EventTriggerId trigger1 = env->taskScheduler().createEventTrigger(trigger1Handler);
  *env << "Event trigger 1 ID: " << trigger1 << "\n";
  EventTriggerId trigger2 = env->taskScheduler().createEventTrigger(trigger2Handler);
  *env << "Event trigger 2 ID: " << trigger2 << "\n";
  do
  {
    int waitTime_T1_to_T2 = rand() % 1000 + 9500; // Random time between T1 and T2 around 10ms

    std::chrono::microseconds T1_to_T2(waitTime_T1_to_T2);
    
    // Signal T1 and T2 with some delay in between and update counters
    env->taskScheduler().triggerEvent(trigger1);
    ++trigger1Signalled;
    std::this_thread::sleep_for(T1_to_T2);
    env->taskScheduler().triggerEvent(trigger2);
    ++trigger2Signalled;
    
    int checkForEventCount = 0;
    do
    {
      // Keep checking that we receive the events that have been signalled and also
      // make sure we do not trigger new events before the previous have been handled
      if( (trigger1Signalled == trigger1Handled) && (trigger2Signalled == trigger2Handled) )
      {
//        printStats();
        break;
      }
      std::this_thread::sleep_for(duration1us); // Wait a bit before next check
      ++checkForEventCount;
      if(checkForEventCount == 20000)
      { // If we keep waiting for a trigger event, it must have disappeared - print out some info
        printStats();
        *env << "Time between trigger1 and trigger2 that resulted in race condition: " << waitTime_T1_to_T2 << "\n";
      }
    } while(true);
    
    std::this_thread::sleep_for(duration20ms); // Wait 20 ms before next events
    --statusCounter;
    if(statusCounter == 0)
    {
      statusCounter = 50;
      printStats();
    }
  } while(true);
}

void startNonLiveThread()
{
  std::thread nonLiveThread(non_live_thread);
  nonLiveThread.detach();
}

int main(int argc, char** argv) {
  // Begin by setting up our usage environment:
  TaskScheduler* scheduler = BasicTaskScheduler::createNew();
  env = BasicUsageEnvironment::createNew(*scheduler);
  trigger1Signalled = 0;
  trigger2Signalled = 0;
  trigger1Handled = 0;
  trigger2Handled = 0;
  statusCounter = 50;

  progName = argv[0];

  gettimeofday(&startTime, NULL);

  startNonLiveThread();

  *env << "Start event loop\n";
  // All subsequent activity takes place within the event loop:
  env->taskScheduler().doEventLoop(); // does not return

  return 0; // only to prevent compiler warning
}


