/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software
 * License version 1.1, a copy of which has been included with this
 * distribution in the LICENSE.txt file.  */



package org.apache.log4j;

import java.io.IOException;
import java.io.Writer;
import java.io.File;
import org.apache.log4j.helpers.OptionConverter;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.helpers.CountingQuietWriter;
import org.apache.log4j.spi.LoggingEvent;
import java.lang.NumberFormatException;
import java.lang.IllegalArgumentException;


/**
   RollingFileAppender extends FileAppender to backup the log files when
   they reach a certain size.

   @author Heinz Richter
   @author Ceki G&uuml;lc&uuml;

*/
public class RollingFileAppender extends FileAppender {

  /**
     The default maximum file size is 10MB.
  */
  protected long maxFileSize = 10*1024*1024;

  /**
     There is one backup file by default.
   */
  protected int  maxBackupIndex  = 1;

/**
   The default vaue is null. If the user just uses RollingFileappender, it will be
    null, otherwise DatePattern will be set by the subclass DailyRollingFileAppender 
    
   Modified  by : Radha S.
   Email : radhas@siptech.co.in
   Company : SIP Technologies
 */
    private String datedFilename;
 
  /**
     The default constructor simply calls its {@link
     FileAppender#FileAppender parents constructor}.  */
  public
  RollingFileAppender() {
    super();
  }

  /**
    Instantiate a RollingFileAppender and open the file designated by
    <code>filename</code>. The opened filename will become the ouput
    destination for this appender.

    <p>If the <code>append</code> parameter is true, the file will be
    appended to. Otherwise, the file desginated by
    <code>filename</code> will be truncated before being opened.
  */
  public
  RollingFileAppender(Layout layout, String filename, boolean append)
                                      throws IOException {
    super(layout, filename, append);
  }
  
  
  /**
     Instantiate a FileAppender and open the file designated by
    <code>filename</code>. The opened filename will become the output
    destination for this appender.

    <p>The file will be appended to.  */
  public
  RollingFileAppender(Layout layout, String filename) throws IOException {
    super(layout, filename);
  }

  /**
     Returns the value of the <b>MaxBackupIndex</b> option.
   */
  public
  int getMaxBackupIndex() {
    return maxBackupIndex;
  }

 /**
    Get the maximum size that the output file is allowed to reach
    before being rolled over to backup files.

    @since 1.1
 */
  public
  long getMaximumFileSize() {
    return maxFileSize;
  }

  /**
     Implements the usual roll over behaviour.

     <p>If <code>MaxBackupIndex</code> is positive, then files
     {<code>File.1</code>, ..., <code>File.MaxBackupIndex -1</code>}
     are renamed to {<code>File.2</code>, ...,
     <code>File.MaxBackupIndex</code>}. Moreover, <code>File</code> is
     renamed <code>File.1</code> and closed. A new <code>File</code> is
     created to receive further log output.

     <p>If <code>MaxBackupIndex</code> is equal to zero, then the
     <code>File</code> is truncated with no backup files created.

   */
  public // synchronization not necessary since doAppend is alreasy synched
  void rollOver() throws IOException {
    File target;
    File file;
   
    /* Checking whether date pattern has been set or not. If not, then this is an instance
       of RollingFileAppender. So, just assign the filename .
      
      Changed by Radha S
      EMail  : radhas@siptech.co.in
      Company : SIP Technologies
       
    */   
    if (datedFilename == null) {
      datedFilename=fileName;
    }  

    LogLog.debug("rolling over count=" + ((CountingQuietWriter) qw).getCount());
    LogLog.debug("maxBackupIndex="+maxBackupIndex);

    // If maxBackups <= 0, then there is no file renaming to be done.
    if(maxBackupIndex > 0) {
      // Delete the oldest file, to keep Windows happy.
      file = new File(datedFilename + '.' + maxBackupIndex);
      if (file.exists())
       file.delete();

      // Map {(maxBackupIndex - 1), ..., 2, 1} to {maxBackupIndex, ..., 3, 2}
      for (int i = maxBackupIndex - 1; i >= 1; i--) {
	file = new File(datedFilename + "." + i);
	if (file.exists()) {
	  target = new File(datedFilename + '.' + (i + 1));
	  LogLog.debug("Renaming file " + file + " to " + target);
	  file.renameTo(target);
	}
      }

      // Rename fileName to fileName.1
      target = new File(datedFilename + "." + 1);

      this.closeFile(); // keep windows happy.

      file = new File(fileName);
      LogLog.debug("Renaming file " + file + " to " + target);
      file.renameTo(target);
    }

    try {
      // This will also close the file. This is OK since multiple
      // close operations are safe.
      this.setFile(fileName, false, bufferedIO, bufferSize);
    }
    catch(IOException e) {
      LogLog.error("setFile("+fileName+", false) call failed.", e);
    }
    datedFilename=null;
  }

  public
  synchronized
  void setFile(String fileName, boolean append, boolean bufferedIO, int bufferSize)
                                                                 throws IOException {
    super.setFile(fileName, append, this.bufferedIO, this.bufferSize);
    if(append) {
      File f = new File(fileName);
      ((CountingQuietWriter) qw).setCount(f.length());
    }
  }


  /**
     Set the maximum number of backup files to keep around.

     <p>The <b>MaxBackupIndex</b> option determines how many backup
     files are kept before the oldest is erased. This option takes
     a positive integer value. If set to zero, then there will be no
     backup files and the log file will be truncated when it reaches
     <code>MaxFileSize</code>.
   */
   
   
  
  public
  void setMaxBackupIndex(int maxBackups) {
  	
   /* Checking conditions with 0 and -ve backups is done by Radha S
      EMail  : radhas@siptech.co.in
      Company : SIP Technologies
   */
   
  // if maxBackups is 0, then default value will be taken
     if (maxBackups != 0) {
         this.maxBackupIndex = maxBackups;    
      }
    try {
    if (this.maxBackupIndex < 0) {
      throw new IllegalArgumentException();
    }
    }  catch (IllegalArgumentException e) {
    	LogLog.error("maxBackupIndex should not be -ve",e);
    	System.exit(0);
       }	  
  }

  /**
     Set the maximum size that the output file is allowed to reach
     before being rolled over to backup files.

     <p>This method is equivalent to {@link #setMaxFileSize} except
     that it is required for differentiating the setter taking a
     <code>long</code> argument from the setter taking a
     <code>String</code> argument by the JavaBeans {@link
     java.beans.Introspector Introspector}.

     @see #setMaxFileSize(String)
 */
 
 
  public
  void setMaximumFileSize(long maxFileSize) {
    
    /* Checking conditions with 0 and -ve file size is done by Radha S
      EMail  : radhas@siptech.co.in
      Company : SIP Technologies
    */
   
    // if maxFileSize is 0, then default value will be taken
    
    if (maxFileSize != 0) {
    	  this.maxFileSize = maxFileSize;    
    }
    
    try   {
    if (this.maxFileSize < 0)  {
      throw new IllegalArgumentException();
    }
    }  catch (IllegalArgumentException e)   {
    	LogLog.error("maxFileSize should not be -ve",e);
       	System.exit(0);
       }
  }


  /**
     Set the maximum size that the output file is allowed to reach
     before being rolled over to backup files.

     <p>In configuration files, the <b>MaxFileSize</b> option takes an
     long integer in the range 0 - 2^63. You can specify the value
     with the suffixes "KB", "MB" or "GB" so that the integer is
     interpreted being expressed respectively in kilobytes, megabytes
     or gigabytes. For example, the value "10KB" will be interpreted
     as 10240.
   */
       
  public
  void setMaxFileSize(String value) {
  	
    long maxFileSize = 0;
    
    maxFileSize=OptionConverter.toFileSize(value, maxFileSize + 1);    
    
    /* Checking conditions with 0 and -ve file size is done by Radha S
      EMail  : radhas@siptech.co.in
      Company : SIP Technologies
    */
    
    // if maxFileSize is 0, then default value will be taken
    
    if (maxFileSize != 0) {
         this.maxFileSize = maxFileSize;    
     }  
    try  {
    if (this.maxFileSize < 0)  {
      throw new IllegalArgumentException();
    }
    }  catch (IllegalArgumentException e)  {
    	LogLog.error("maxFileSize should not be -ve",e);
    	System.exit(0);
       }
  }

  protected
  void setQWForFiles(Writer writer) {
     this.qw = new CountingQuietWriter(writer, errorHandler);
  }

  
  /** This method is  to set the datepattern and especially called by the subclass Daily
  RollingFileAppender 
  
  Modified by : Radha S
  Email : radhas@siptech.co.in
  Company : SIP Technologies */
  
  public void setDatedFilename(String sFilename)  {
  	datedFilename=sFilename;
  }	


  /**
     This method differentiates RollingFileAppender from its super
     class.

     @since 0.9.0
  */
    
  protected
  void subAppend(LoggingEvent event) {
    super.subAppend(event);
    
    if((fileName != null) &&
                     ((CountingQuietWriter) qw).getCount() >= maxFileSize)
    try
    {                 
      this.rollOver();
    }
    catch (IOException e)
    {
    	LogLog.error("setFile("+fileName+", false) call failed.", e);
    }	  
   }
}
