/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software
 * License version 1.1, a copy of which has been included with this
 * distribution in the LICENSE.APL file.  */

package org.apache.log4j;

import java.text.SimpleDateFormat;
import java.io.File;
import java.io.IOException;
import java.util.Calendar;

/**
 * Represents the filename of the log file, including a date.
 * Handles calendar logic, including the date formatting and
 * determining the next rollover time.
 * Calls to getDatedFilename should not be made by multiple threads
 * (because SimpleDateFormat isn't thread-safe) unless synchronized by
 * the caller.
 *
 * @author Jon Skeet <skeet@pobox.com>
 * @author Jim Doyle <jdoyle@iso-ne.com>
 */
public class DatedFile
{
    /** Formatter to use for dates in filename; generated from datePattern */
    private SimpleDateFormat dateFormat;
    
    /** Base of filename (ie everything before the extension) */
    private String filenameBody=null;
	
    /** File extension to add after date (includes .) */
    private String extension=null;


    /**
     * Constructs dated file with filename and explicit date pattern
     * @param filename Base filename, pulled apart and used to construct
     *                 a dated filename.  Includes directories and any extension.
     * @param datePattern Date pattern string, as expected by
     *                    {@link SimpleDateFormat}, like "yyyyMMdd"
     * @exception IOException if filesystem queries needed to create
     *                        filename canonical path fail
     */
    public DatedFile (String filename, String datePattern) throws IOException
    {
        parseFilename(filename);
        createFormat(datePattern);
    }
    
    /**
     * Constructs a dated filename, based on the date in the given calendar.
     * A filename of <code>body.ext</code> is translated to
     * <code>body-date.ext</code>.  Expects <code>setFile</code> to have
     * been called to set the body and extension.  If
     * <code>setDatePattern</code> has not been called, the default
     * "yyyyMMdd" will be used.
     * @param date Calendar set to date to apply to this file.
     *             May have any time values:
     *             hour, minutes, seconds, and millis don't have to be cleared.
     *             The given Calendar will be modified.
     * @return filename including original path, base name, date, and extension
     */
    public String getDatedFilename (Calendar date)
    {
        return filenameBody+"-"+dateFormat.format (date.getTime())+extension;
    }

    /**
     * Determines the next time the filename will change, using
     * the date in the given calendar as the current time.
     * Since the filename changes at midnight, this will be the midnight
     * following the given time.
     * @param date Calendar set to date containing current time.
     * @return Milliseconds for time of next filename change
     */
    public long nextFilenameChange (Calendar date)
    {
        Calendar nextDayMidnight = Calendar.getInstance();
        nextDayMidnight.clear();
        nextDayMidnight.set(date.get(Calendar.YEAR),
                            date.get(Calendar.MONTH),
                            date.get(Calendar.DATE));
        nextDayMidnight.add(Calendar.DATE, 1);
        return nextDayMidnight.getTime().getTime();
    }

    
    /**
     * Pulls apart the base filename for this file.
     * Sets the <code>filenameBody</code>
     * and <code>extension</code>, which get used when a dated filename
     * is created.
     * @param filename Base filename, including directories and any extension
     * @exception IOException if filesystem queries needed to create
     *                        canonical path fail
     */
    private void parseFilename (String filename) throws IOException
    {
        filename = filename.trim();
        
        File f = new File (filename);
        
        // Separate the name from the parent directory structure.
        // This is needed in case someone specifies a file like:
        // /var/log/foo.bar/asd in which case we want to create
        // /var/log/foo.bar/asd-xxxxxx.log, not
        // /var/log/foo-xxxxxx.bar/asd
        String parent = f.getParent();
        String name = f.getName();
        
        int extIndex = name.lastIndexOf (".");
        // Just make the code simple...
        if (extIndex==-1)
        {
            extIndex=name.length();
            name=name+".log";
        }
        extension = name.substring (extIndex);
        if (extension.equals ("."))
            extension=".log";
        name = name.substring (0, extIndex);
        if (parent==null)
            this.filenameBody = new File (name).getCanonicalPath();
        else
            this.filenameBody = new File (parent, name).getCanonicalPath();
    }

    /**
     * Creates a date format based on the given date pattern.
     * The format determines the date strings included in the file name.
     * @param pattern Date format string, as expected by SimpleDateFormat
     */
    private void createFormat (String pattern) 
    {
        dateFormat = new SimpleDateFormat (pattern);
    }
}
