/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software
 * License version 1.1, a copy of which has been included with this
 * distribution in the LICENSE.APL file.  */

package org.apache.log4j;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;

import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.log4j.spi.ErrorCode;

/**
 * Appender that rolls over automatically, always writing to a log file 
 * dated with today's date before the extension. Unlike 
 * DailyRollingFileAppender, this class does not rely on the application
 * being running at rollover time, as the "current" filename is always
 * unambiguous.
 * 
 * @author Jon Skeet <skeet@pobox.com>
 * @author Jim Doyle <jdoyle@iso-ne.com>
 */
public class DatedFileAppender extends FileAppender
{
    /** Date pattern to use for formatting dates, as used in SimpleDateFormat */
    private String datePattern;
    
    /** 
     * Default date pattern - allows for easy sorting and is less likely to cause
     * confusion than ddMMyyyy or MMddyyyy
     */
    public static final String DEFAULT_DATE_PATTERN = "yyyyMMdd";

    /**
     * Cached DatedFile object that computes file names for us.
     * Gets initialized after options are set, when activateOptions
     * is called.
     */
    private DatedFile datedFile;
    
    /** When to roll the log */
    private long nextMidnight=0;

    /**
     * The only supported constructor - any calls to other superconstructors 
     * could try to open the file before we necessarily have the right
     * date pattern.
     */
    public DatedFileAppender ()
    {
        super();
        setDatePattern (DEFAULT_DATE_PATTERN);
    }
    
    /**
     * Overrides FileAppender.activateOptions to open a dated log file.
     * The dated filename is based on the
     * <b>File</b> and <b>DatePattern</b> options, and the file is opened
     * with the <b>Append</b> option.  Caches a dated file object containing
     * the filename parts and the date pattern.
     */
    public void activateOptions ()
    {
        if (fileName == null)
        {
            LogLog.warn("File option not set for appender ["+name+"].");
            return;
        }
        if (datePattern == null)
        {
            LogLog.warn("Date pattern option not set for appender ["+name+"].");
            return;
        }

        try
        {
            datedFile = new DatedFile(fileName, datePattern);
            setFile(datedFile, fileAppend);
        }
        catch (IOException e)
        {
            errorHandler.error("setFile() call failed.",
                               e, ErrorCode.FILE_OPEN_FAILURE);
        }
    }
     
    /**
     * Closes the current opened file, and opens a new one with a dated
     * filename, using the FileAppender superclass.
     * Updates the <code>nextMidnight</code>
     * time to the next time the dated filename will rollover.
     * <p>
     * This is called after <code>activateOptions</code>, and is
     * expected to actually open the file.  The filename, append flag,
     * and date pattern options will all have been set previously.
     * Note that the filename option will be overwritten with
     * the dated file name currently being used.  This is also called
     * during <code>subAppend</code>, when a rollover is detected.
     * @exception IOException if closing and opening files fails
     */
    public void setFile (DatedFile datedFile, boolean append) throws IOException
    {
        Calendar now = Calendar.getInstance();
        String datedFilename = datedFile.getDatedFilename(now);
        setFile (datedFilename, append, bufferedIO, bufferSize);
        nextMidnight = datedFile.nextFilenameChange(now);
        LogLog.debug ("Opened "+datedFilename);
        LogLog.debug ("Next rollover: "+new Date(nextMidnight));
    }

    /**
     * The <b>DatePattern</b> option takes a string in the same format as
     * expected by {@link SimpleDateFormat}. 
     */
    public void setDatePattern(String pattern) 
    {
        datePattern = pattern;
    }
  
    /** Returns the value of the <b>DatePattern</b> option. */
    public String getDatePattern() 
    {
        return datePattern;
    }
    
    /**
     * Overrides FileAppender.subAppend() to check for rollover.
     * Will re-open the file with a new dated filename if rollover is
     * necessary.
     * <p>
     * Assumes all options have been set, so we can open a correct
     * dated filename.  If <code>setFile</code> has not yet been called,
     * then we will call it now, because the <code>nextMidnight</code>
     * value will be zero and cause a rollover.
     */
    protected synchronized void subAppend (LoggingEvent event)
    {
        long now = System.currentTimeMillis();
        if (now > nextMidnight)
        {
            try
            {
                setFile(datedFile, fileAppend);
            }
            catch (IOException e)
            {
                LogLog.error("setFile() failed.", e);
            }
        }
        super.subAppend (event);
    }
}
