/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software
 * License version 1.1, a copy of which has been included with this
 * distribution in the LICENSE.APL file.  */

package org.apache.log4j;

import java.io.File;
import java.util.Calendar;
import java.util.Date;
import junit.framework.TestCase;

/**
 * JUnit test cases for DatedFile.
 * @author Jim Doyle <jdoyle@iso-ne.com>
 */
public class DatedFileTest extends TestCase
{
    /** Date shared by tests: June 1, 2001 */
    private Calendar june1_2001;

    /** Date shared by tests: June 2, 2001 */
    private Calendar june2_2001;

   
    /**
     * Constructs a test case that will run the named test method
     * @param name Name of test method
     */
    public DatedFileTest (String name)
    {
        super(name);
    }
    
    /**
     * Should create correct filename for
     * a normal case like /var/log/foo/bar/asd.log.
     */
    public void testRegularName () throws Exception
    {
        String expectedPath =
            new File("/var/log/foo/bar/asd-20010601.log").getCanonicalPath();
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(expectedPath, file.getDatedFilename(june1_2001));
    }
    
    /**
     * Should still insert date at end when there's a dot in
     * one of the directories, like /var/log/foo.bar/asd.log.
     */
    public void testDotInDirectory () throws Exception
    {
        String expectedPath =
            new File("/var/log/foo.bar/asd-20010601.log").getCanonicalPath();
        DatedFile file = new DatedFile("/var/log/foo.bar/asd.log", "yyyyMMdd");
        assertEquals(expectedPath, file.getDatedFilename(june1_2001));
    }
    
    /**
     * Should tack on ".log" as extension when there is no extension.
     */
    public void testExtensionForced () throws Exception
    {
        String expectedPath =
            new File("/var/log/foo/bar/asd-20010601.log").getCanonicalPath();
        DatedFile file = new DatedFile("/var/log/foo/bar/asd", "yyyyMMdd");
        assertEquals(expectedPath, file.getDatedFilename(june1_2001));
    }
    
    /**
     * Should handle different date patterns
     */
    public void testDifferentDatePattern () throws Exception
    {
        String expectedPath =
            new File("/var/log/foo/bar/asd-06012001.log").getCanonicalPath();
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "MMddyyyy");
        assertEquals(expectedPath, file.getDatedFilename(june1_2001));
    }
    
    /**
     * Should get date even when time is in middle of day
     */
    public void testDateFromMiddleOfDay () throws Exception
    {
        june1_2001.add(Calendar.HOUR_OF_DAY, -12);
        String expectedPath =
            new File("/var/log/foo/bar/asd-20010531.log").getCanonicalPath();
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(expectedPath, file.getDatedFilename(june1_2001));
    }
    
    /**
     * Should get next rollover date when time is exactly at midnight.
     */
    public void testRolloverAtMidnight () throws Exception
    {
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(june2_2001.getTime(),
                     new Date(file.nextFilenameChange(june1_2001)));
    }
    
    /**
     * Should get next rollover date when time is a few seconds past midnight.
     */
    public void testRolloverFewSecondsLate () throws Exception
    {
        june1_2001.add(Calendar.SECOND, 10);
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(june2_2001.getTime(),
                     new Date(file.nextFilenameChange(june1_2001)));
    }

    /**
     * Should get next rollover date when time is hours after midnight.
     */
    public void testRolloverHoursLate () throws Exception
    {
        june1_2001.add(Calendar.HOUR_OF_DAY, 12);
        june1_2001.add(Calendar.SECOND, 10);
        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(june2_2001.getTime(),
                     new Date(file.nextFilenameChange(june1_2001)));
    }

    /**
     * Should get next rollover date on short (EST-EDT) day.
     */
    public void testRolloverEstEdt () throws Exception
    {
        Calendar apr1_2001 = Calendar.getInstance();
        apr1_2001.clear();
        apr1_2001.set(2001, Calendar.APRIL, 1);
        apr1_2001.add(Calendar.MINUTE, 30);
        
        Calendar apr2_2001 = Calendar.getInstance();
        apr2_2001.clear();
        apr2_2001.set(2001, Calendar.APRIL, 2);

        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(apr2_2001.getTime(),
                     new Date(file.nextFilenameChange(apr1_2001)));
    }

    /**
     * Should get next rollover date on long (EDT-EST) day.
     */
    public void testRolloverEdtEst () throws Exception
    {
        Calendar oct28_2001 = Calendar.getInstance();
        oct28_2001.clear();
        oct28_2001.set(2001, Calendar.OCTOBER, 28);
        oct28_2001.add(Calendar.MINUTE, 30);
        
        Calendar oct29_2001 = Calendar.getInstance();
        oct29_2001.clear();
        oct29_2001.set(2001, Calendar.OCTOBER, 29);

        DatedFile file = new DatedFile("/var/log/foo/bar/asd.log", "yyyyMMdd");
        assertEquals(oct29_2001.getTime(),
                     new Date(file.nextFilenameChange(oct28_2001)));
    }

    /**
     * Sets up test dates
     */
    protected void setUp ()
    {
        june1_2001 = Calendar.getInstance();
        june1_2001.clear();
        june1_2001.set(2001, Calendar.JUNE, 1);

        june2_2001 = Calendar.getInstance();
        june2_2001.clear();
        june2_2001.set(2001, Calendar.JUNE, 2);
    }
}
