/*
 * Created: 10-Jan-2005
 * Author : swoodward
 */
package com.peregrine.discovery.logserver;

import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

/**
 * @author swoodward
 */
public class Configurator
{
  /**
   * @param props list of properties used to configure the log server.
   * @return
   * @throws InstantiationException
   * @throws IllegalAccessException
   */
  public static Vector configure(Properties props)
      throws InstantiationException, IllegalAccessException
  {
    Vector ret = new Vector();
    cat.debug("Starting configuration.");
    HashSet sections = getSectionList(props);
    cat.debug(MessageFormat.format("Got section list: {0}",
        new Object[] { sections.toString() }));
    Iterator it = sections.iterator();
    while (it.hasNext())
    {
      String section = (String) it.next();
      cat.info(MessageFormat.format("Handling config section {0}",
          new Object[] { section }));
      // check that both attribs are there (port, parser class)
      if (section == null || !props.keySet().contains(section + "parserclass")
          || !props.keySet().contains(section + "port"))
      {
        cat.error(MessageFormat.format(
            "Section {0} is missing either a parserclass or port attribute.",
            new Object[] { section }));
        continue;
      }

      // Ensure that the parserclass specified is indeed an EventParser implementation
      Class parserClass = null;
      try
      {
        parserClass = Class.forName(props.getProperty(section + "parserclass"));
      }
      catch (ClassNotFoundException e)
      {
        cat.error(MessageFormat.format(
            "Specified parser class could not be found in section {0}: {1}",
            new Object[] { section, props.getProperty(section + "parserclass") }));
        continue;
      }

      if (!EventParser.class.isAssignableFrom(parserClass))
      {
        cat.error(MessageFormat.format(
            "Specified parser class is not inherited from EventParser in section {0}: {1}",
            new Object[] { section, props.getProperty(section + "parserclass") }));
        continue;
      }

      // create and start a new thread.
      cat.debug(MessageFormat.format(
          "Creating a log server thread with EventParser type {0}",
          new Object[] { parserClass.getName() }));

      LogServerRunnable runnable = new LogServerRunnable(Integer.parseInt(
          props.getProperty(section + "port"), 10), parserClass, getAttribs(
          section, props));
      new Thread(runnable).start();
      cat.debug("Created LogServerRunnable.");
      ret.add(runnable);
    }

    return ret;
  }

  /**
   * Validates that the <code>string<code> parameter is a 
   * legal property name and returns the section name portion 
   * of it. 
   * 
   * @param key string to extract the section name from
   * @return the section name, or <code>null</code> if <code>key</code> is not a legal property.
   */
  protected static String getSectionName(String key)
  {
    Matcher matcher = sectionPattern.matcher(key);
    if (matcher.matches())
      return matcher.group(1);
    else
      return null;
  }

  protected static String getAttribName(String key)
  {
    if (key.lastIndexOf(".") == -1)
      return "";
    else
      return key.substring(key.lastIndexOf(".") + 1, key.length());
  }

  protected static Properties getAttribs(String section, Properties props)
  {
    Properties ret = new Properties();

    Enumeration it = props.keys();
    while (it.hasMoreElements())
    {
      String key = (String) it.nextElement();
      if (key.startsWith(section)
          && (!getAttribName(key).equals("parserclass"))
          && (!getAttribName(key).equals("port")))
        ret.put(getAttribName(key), props.get(key));
    }
    return ret;
  }

  protected static boolean isA(Class inherited, Class superClass)
  {
    boolean ret = false;
    while (!ret && inherited.getSuperclass() != null)
    {
      inherited = inherited.getSuperclass();
      if (inherited == superClass)
        ret = true;
    }

    return ret;
  }

  /**
   * @param props
   * @return
   */
  protected static HashSet getSectionList(Properties props)
  {
    Enumeration enum = props.keys();
    HashSet ret = new HashSet();
    while (enum.hasMoreElements())
    {
      String name = getSectionName((String) enum.nextElement());
      if (name != null)
        ret.add(name);
    }
    return ret;
  }

  static final String SET_PATTERN = "(logserver\\.serverthread\\.\\w*\\.).*";
  static Pattern sectionPattern = Pattern.compile(SET_PATTERN);
  protected static Logger cat = Logger.getLogger(Configurator.class);
}