/*
 * Created: 10-Jan-2005
 * Author : swoodward
 */
package com.peregrine.discovery.logserver;

import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.Properties;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.config.PropertySetter;

/**
 * One <code>LogServerRunnable</code> is created for every configured 
 * server port in the LogServer.
 * Each LogServerRunnable listens on its port and invokes a <code>ClientHandler</code>
 * for every client that connects.
 * 
 * @author swoodward
 */
public class LogServerRunnable implements Runnable
{
  public LogServerRunnable(int port, Class eventParserClass,
      Properties parserProps)
  {
    this.port = port;
    this.eventParserClass = eventParserClass;
    this.parserProps = parserProps;
  }

  /* (non-Javadoc)
   * @see java.lang.Runnable#run()
   */
  public void run()
  {
    try
    {
      cat.info("Listening on port " + port);
      ServerSocket serverSocket = new ServerSocket(port);
      while (true)
      {
        if (Thread.currentThread().isInterrupted())
        {
          cat.info(MessageFormat.format(
              "Thread interruped; exiting thread listening on {0}",
              new Object[] { new Integer(port) }));
        }
        cat.info("Waiting to accept a new client.");
        Socket socket = serverSocket.accept();
        cat.info("Connected to client at " + socket.getInetAddress());
        cat.info("Starting new socket node.");
        cat.debug(MessageFormat.format("Creating parser of type {0} with props {1}.",
            new Object[]{eventParserClass.getName(), parserProps.toString()}));
        new Thread(new ClientHandler(socket, createParser(eventParserClass,
            parserProps, socket.getInputStream()), LogManager.getLoggerRepository())).start();
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  protected static EventParser createParser(Class eventParserClass,
      Properties parserProps, InputStream strm)
  {
    EventParser parser = null;
    try
    {
      parser = (EventParser) eventParserClass.newInstance();
    }
    catch (InstantiationException e)
    {
      cat.error("unable to instantiate parser " + eventParserClass.getName(), e);
    }
    catch (IllegalAccessException e)
    {
      cat.error("unable to instantiate parser " + eventParserClass.getName(), e);
    }

    if (parser == null)
      return parser;

    parser.setInputStream(strm);
    PropertySetter ps = new PropertySetter(parser);
    Enumeration enum = parserProps.keys();
    while (enum.hasMoreElements())
    {
      String attrib = (String) enum.nextElement();
      cat.debug(MessageFormat.format("Setting attribute {0} to \"{0}\"",
          new Object[] { attrib, parserProps.getProperty(attrib) }));
      ps.setProperty(attrib, parserProps.getProperty(attrib));
    }
    return parser;
  }

  protected static Logger cat = Logger.getLogger(LogServerRunnable.class);
  protected int port;
  private Class eventParserClass;
  private Properties parserProps;
}