/*
 * Created: 10-Jan-2005
 * Author : swoodward
 */
package com.peregrine.discovery.logserver.tests;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.Format;
import java.text.MessageFormat;
import java.util.HashSet;
import java.util.Properties;

import junit.framework.TestCase;

import com.peregrine.discovery.logserver.Configurator;

/*
 * Class to provide access to the internals of Configurator
 */
class PublicConfigurator extends Configurator
{
  public static String getSectionName(String key)
  {
    return Configurator.getSectionName(key);
  }

  public static String getAttribName(String key)
  {
    return Configurator.getAttribName(key);
  }

  public static Properties getAttribs(String section, Properties props)
  {
    return Configurator.getAttribs(section, props);
  }
  
  public static boolean isA(Class inherited, Class superClass)
  {
    return Configurator.isA(inherited, superClass);
  }
  
  public static HashSet getSectionList(Properties props)
  {
    return Configurator.getSectionList(props);
  }
}

/**
 * @author swoodward
 */

public class ConfiguratorTest extends TestCase
{

  /*
   * @see TestCase#setUp()
   */
  protected void setUp() throws Exception
  {
    super.setUp();
  }

  /*
   * @see TestCase#tearDown()
   */
  protected void tearDown() throws Exception
  {
    super.tearDown();
  }

  public void testGetSectionName()
  {
    assertEquals(
        "Section name unexpected with parserclass attribute.",
        PublicConfigurator.getSectionName("logserver.serverthread.1.parserclass"),
        "logserver.serverthread.1.");

    assertEquals("Section name unexpected with someattrib attribute.",
        PublicConfigurator.getSectionName("logserver.serverthread.1.someattrib"), "logserver.serverthread.1.");

    assertEquals("Section name unexpected with erroneous section name.",
        PublicConfigurator.getSectionName("nosection.serverthread.1.someattrib"), null);

    assertEquals("Section name unexpected with no section name.",
        PublicConfigurator.getSectionName("someattrib"), null);
  }

  public void testGetAttribName()
  {  
    assertEquals("Unexpected attribute using parserclass attrib",
        "parserclass",
        PublicConfigurator.getAttribName("logserver.serverthread.1.parserclass"));
    assertEquals("Unexpected attribute using myattrib attrib",
        "myattrib",
        PublicConfigurator.getAttribName("logserver.serverthread.1.myattrib"));
    assertEquals("Unexpected attribute using erroneous section name",
        "port",
        PublicConfigurator.getAttribName("erroneous.serverthread.1.port"));
    assertEquals("Unexpected attribute using no attrib",
        "",
        PublicConfigurator.getAttribName("logserver.serverthread.1."));
    assertEquals("Unexpected attribute using no dots",
        "",
        PublicConfigurator.getAttribName("string"));
    assertEquals("Unexpected attribute using empty string",
        "",
        PublicConfigurator.getAttribName("string"));    
  } 

  public void testGetAttribs() throws IOException
  {
    String propsString = "logserver.serverthread.1.parserclass\n" +
        "logserver.serverthread.1.port=1000\n" +
        "logserver.serverthread.1.attriba=value1a\n" +
        "logserver.serverthread.1.attribb=value1b\n" +
        "error.serverthread.2.attriba=value2a\n" +
        "error.serverthread.2.attribb=value2b\n" +
        "error.serverthread.3.attriba=value3a\n" +
        "logserver.serverthread.3.attribb=value3b\n";
    Properties props = new Properties();
    props.load(new ByteArrayInputStream(propsString.getBytes()));
          
    Properties result = PublicConfigurator.getAttribs("logserver.serverthread.1.", props);
    assertTrue("unexpected values of properties for section 1",
        result.get("attriba").equals("value1a") &&
        result.get("attribb").equals("value1b") &&
        result.size() == 2);

    result = PublicConfigurator.getAttribs("error.serverthread.2.", props);
    assertTrue("unexpected values of properties for section 2",
        result.get("attriba").equals("value2a") &&
        result.get("attribb").equals("value2b") &&
        result.size() == 2);
    
    result = PublicConfigurator.getAttribs("logserver.serverthread.3.", props);
    assertTrue("unexpected values of properties for section 3",
        result.get("attribb").equals("value3b") &&
        result.size() == 1);    
  }
  
  public void testIsA()
  {
    assertTrue("One level", PublicConfigurator.isA(MessageFormat.class, Format.class));
    assertFalse("Unrelated", PublicConfigurator.isA(MessageFormat.class, String.class));
    assertFalse("Reverse", PublicConfigurator.isA(Format.class, MessageFormat.class));
  }
  
  public void testGetSectionList() throws IOException
  {
    String propsString = 
      "logserver.serverthread.1.parserclass\n" +
      "logserver.serverthread.1.port=1000\n" +
      "logserver.serverthread.1.attriba=value1a\n" +
      "logserver.serverthread.1.attribb=value1b\n" +
      "error.serverthread.2.attriba=value2a\n" +
      "error.serverthread.2.attribb=value2b\n" +
      "error.serverthread.3.attriba=value3a\n" +
      "logserver.serverthread.3.attribb=value3b\n" +
      "logserver.serverthread.test.attribb=value3b\n";
    Properties props = new Properties();
    props.load(new ByteArrayInputStream(propsString.getBytes()));
    
    HashSet set = PublicConfigurator.getSectionList(props);
    assertTrue(set.contains("logserver.serverthread.1.") &&
      set.contains("logserver.serverthread.3.") &&
      set.contains("logserver.serverthread.test.") &&
      set.size() == 3);
  }
}