#
# MinHeap implementation
#
# $Id: minheap.rb 1336 2009-07-07 23:29:41Z packet $

module Ruffman
  class MinHeap
    attr_reader :size

    def initialize(*pairs)
      @tree = Array.new
      @size = 0

      bulk_insert(*pairs)
    end

    def empty?
      @size > 0 ? false : true
    end

    def insert(key, value)
      index = @size
      new_node = HeapNode.new(key, value, index)

      @size += 1
      @tree[index] = new_node

      heapify(index)

      new_node
    end

    def remove(node)
      if (node.index >= 0 and node.index < @size and
            @tree[node.index] === node)
        remove_index(node.index)
      end
    end

    def extract_min
      remove_index(0)
    end

    def change_key(node, key)
      if (node.index >= 0 and node.index < @size and
            @tree[node.index] === node)
        @tree[node.index].key = key

        heapify(node.index)

        node
      end
    end

    def bulk_insert(*pairs)
      unless pairs.empty?
        self.capacity = @size + pairs.size

        nodes = pairs.map do |pair|
          HeapNode.new(*pair)
        end

        nodes.each do |node|
          node.index = @size
          @tree[@size] = node
          @size += 1
        end

        (@size-1).step(0, -2) do |index|
          bubble_down(MinHeap.parent(index))
        end

        nodes
      end
    end

    def capacity
      @tree.size
    end

    def capacity=(new_capacity)
      @tree[new_capacity-1] = nil if new_capacity > @size
    end

    private

    attr_writer :size

    def self.left(index)
      2*index+1
    end

    def self.right(index)
      2*index+2
    end

    def self.parent(index)
      (index - 1)/ 2
    end

    def leaf?(index)
      MinHeap.left(index) < @size ? false : true
    end

    def swap(orig_index, new_index)
      obj1 = @tree[orig_index]
      obj2 = @tree[new_index]

      obj1.index = new_index unless obj1.nil?
      obj2.index = orig_index unless obj2.nil?
      @tree[orig_index] = obj2
      @tree[new_index] = obj1

      new_index
    end

    def heapify(index)
      index = bubble_up(index)
      index = bubble_down(index)
    end

    def bubble_up(index)
      # bubble-up
      while index > 0 and (@tree[index].key <=> @tree[MinHeap.parent(index)].key) == -1 do
        index = swap(index, MinHeap.parent(index))
      end

      index
    end

    def bubble_down(index)
      # bubble-down
      current_node = nil
      min_child_index = nil
      right_child_index = nil
      min_child = nil
      right_child = nil

      while not leaf?(index) do
        current_node = @tree[index]
        min_child_index = MinHeap.left(index)
        min_child = @tree[min_child_index]
        right_child_index = MinHeap.right(index)
        right_child = @tree[right_child_index]

        if not right_child.nil? and (right_child.key <=> min_child.key) == -1
          min_child_index = right_child_index
          min_child = right_child
        end

        if (min_child.key <=> current_node.key) == -1
          index = swap(index, min_child_index)
        else
          break
        end
      end

      index
    end

    def remove_index(index)
      if index >= 0 and index < @size
        @size -= 1
        value = @tree[index].value

        @tree[index].index = nil
        @tree[index] = nil

        # special case: index is already the last index in the tree
        unless index == @size
          swap(index, @size)
          heapify(index)
        end

        value
      end
    end

    def check_tree
      @tree.each_index do |index|
        unless leaf?(index)
          left = MinHeap.left(index)
          if (@tree[left].key <=> @tree[index].key) == -1
            p "Heap property violated, parent key: #{@tree[index].key}, child key: #{@tree[left].key}"
            return false
          end

          right = MinHeap.right(index)
          if not @tree[right].nil? and (@tree[right].key <=> @tree[index].key) == -1
            p "Heap property violated, parent key: #{@tree[index].key}, child key: #{@tree[right].key}"
            return false
          end
        end
      end

      return true
    end

    class HeapNode

      def initialize(key, value, index = nil)
        @key = key
        @value = value
        @index = index
      end

      attr_accessor :key, :value, :index
    end
  end
end
