# Performance-Test for MinHeap
#
# $Id: minheap_performance_test.rb 1337 2009-07-09 23:40:07Z packet $

$:.unshift File.join(File.dirname(__FILE__),'..','lib')

require 'test/unit'
require 'benchmark'
require 'minheap'
require 'generic_wrapper'
require 'immutable_wrapper'

class MinHeapPerformanceTest < Test::Unit::TestCase
  def setup
    srand()
    random_seed = srand(0)
    srand(random_seed)

    @array = Array.new(100000) do rand(1000000) end
    @pairs = @array.map { |key| [key, key] }
    @expected = @array.dup.sort!

    @frozen_array = @array.map do |i| Ruffman::ImmutableWrapper.new(i).freeze end.freeze
    @frozen_pairs = @frozen_array.map { |obj| [obj.value, obj] }.freeze
    @frozen_expected = @frozen_array.dup.sort!.freeze

    @obj_array = @array.map do |i| Ruffman::GenericWrapper.new(i) end
    @obj_pairs = @obj_array.map { |obj| [obj.value, obj] }
    @obj_expected = @obj_array.dup.sort!
  end

  def test_minheap_integer_performance
    Benchmark.bmbm do |job|
      job.report("MinHeap integer insert") { minheap_insert_performance(@array, @pairs, @expected) }
      job.report("MinHeap integer bulk insert") { minheap_bulk_insert_performance(@array, @pairs, @expected) }
      job.report("Integer Array sort") { array_insert_performance(@array, @pairs, @expected) }
    end
  end

  def test_minheap_object_performance
    Benchmark.bmbm do |job|
      job.report("MinHeap object insert") { minheap_insert_performance(@obj_array, @obj_pairs, @obj_expected) }
      job.report("MinHeap bulk object insert") { minheap_bulk_insert_performance(@obj_array, @obj_pairs, @obj_expected) }
      job.report("Object Array sort") { array_insert_performance(@obj_array, @obj_pairs, @obj_expected) }
    end
  end

  def test_minheap_frozen_performance
    Benchmark.bmbm do |job|
      job.report("MinHeap object insert") { minheap_insert_performance(@frozen_array, @frozen_pairs, @frozen_expected) }
      job.report("MinHeap bulk object insert") { minheap_bulk_insert_performance(@frozen_array, @frozen_pairs, @frozen_expected) }
      job.report("Object Array sort") { array_insert_performance(@frozen_array, @frozen_pairs, @frozen_expected) }
    end
  end

  private

  def minheap_insert_performance(ary, pairs, expected)
    mh = Ruffman::MinHeap.new
    mh.capacity = pairs.size
    pairs.each { |p| mh.insert(*p) }

    result = Array.new(mh.size) { mh.extract_min }

    assert_equal(expected, result, "MinHeap property violated")
  end

  def minheap_bulk_insert_performance(ary, pairs, expected)
    mh = Ruffman::MinHeap.new
    mh.capacity = pairs.size
    mh.bulk_insert(*pairs)

    result = Array.new(mh.size) { mh.extract_min }

    assert_equal(expected, result, "MinHeap property violated")
  end

  def array_insert_performance(ary, pairs, expected)
    result = Array.new(ary)
    result.sort!

    assert_equal(expected, result, "Array sort is wrong")
  end
end
