# Testcase for MinHeap
#
# $Id: minheap_test.rb 1336 2009-07-07 23:29:41Z packet $

$:.unshift File.join(File.dirname(__FILE__),'..','lib')

require 'test/unit'
require 'minheap'

class MinHeapTest < Test::Unit::TestCase
  def setup
    srand()
    random_seed = srand(0)
    srand(random_seed)

    @test_array1 = Array.new(1000) do rand(1000000) end
    @pairs1 = @test_array1.map { |key| [key, key] }
    @expected1 = @test_array1.sort

    @test_array2 = Array.new(1000) do rand(1000000) end
    @pairs2 = @test_array2.map { |key| [key, key] }
    @expected12 = (@test_array1 + @test_array2).sort
  end

  def test_minheap_insert
    assert_nothing_raised("Exception raised during insert test") do
      mh = Ruffman::MinHeap.new

      @pairs1.each do |p|
        assert_not_nil(mh.insert(*p), "Got no handle from insert")
        assert(mh.send(:check_tree), "Heap property violated")
      end

      @pairs2.each do |p|
        assert_not_nil(mh.insert(*p), "Got no handle from insert")
        assert(mh.send(:check_tree), "Heap property violated")
      end

      result = Array.new(mh.size) do
        value = mh.extract_min
        assert(mh.send(:check_tree), "Heap property violated")
        assert_not_nil(value, "Value obtained from heap is nil but shouldn't")
        value
      end

      assert_nil(mh.extract_min, "heap should be empty")
      assert(mh.empty?, "heap should be empty")
      assert_equal(0, mh.size, "heap should be empty")
      assert_equal(@expected12.size, result.size, "Size does not match")
      assert_equal(@expected12, result, "MinHeap property violated")
    end
  end

  def test_minheap_bulk_insert
    assert_nothing_raised("Exception raised during bulk_insert test") do
      mh = Ruffman::MinHeap.new(*@pairs1)
      mh.bulk_insert(*@pairs2)

      result = Array.new(mh.size) do
        value = mh.extract_min
        assert_not_nil(value, "Value obtained from heap is nil but shouldn't")
        assert(mh.send(:check_tree), "Heap property violated")
        value
      end

      assert_nil(mh.extract_min, "heap should be empty")
      assert(mh.empty?, "heap should be empty")
      assert_equal(0, mh.size, "heap should be empty")
      assert_equal(@expected12.size, result.size, "Size does not match")
      assert_equal(@expected12, result, "MinHeap property violated")
    end
  end

  def test_minheap_indexing
    assert_nothing_raised("Exception raised during minheap indexing test") do

      @test_array1.each do |index|
        # no two of these should equal
        indices = [ index, Ruffman::MinHeap.send(:parent, index),
          Ruffman::MinHeap.send(:left, index), Ruffman::MinHeap.send(:right, index) ]

        indices.each_index do |i|
          0.upto(i-1) do |j|
            assert_not_equal(indices[i], indices[j], "any of the index operations is broken")
          end
        end
      end

      @test_array1.each do |index|
        result = Ruffman::MinHeap.send(:parent, Ruffman::MinHeap.send(:left, index))
        assert_equal(index, result, "parent of left child of index broken")
      end

      @test_array1.each do |index|
        result = Ruffman::MinHeap.send(:parent, Ruffman::MinHeap.send(:right, index))
        assert_equal(index, result, "parent of right child of index broken")
      end
    end
  end

  def test_minheap_remove
    assert_nothing_raised("Exception raised during remove test") do
      mh = Ruffman::MinHeap.new
      rm_handles = mh.bulk_insert(*@pairs1)
      expected = Array.new(@test_array1)

      # shuffle rm_handles and expected
      @pairs1.size.downto(2) do |i|
        j = rand(i)

        rm_handles[j], rm_handles[i-1] = rm_handles[i-1], rm_handles[j]
        expected[j], expected[i-1] = expected[i-1], expected[j]
      end

      num_remove = expected.size/10+1
      rm_handles = rm_handles[0, num_remove]
      expected_values = expected[0, num_remove]
      expected = expected[num_remove..-1].sort

      rm_handles.each_index do |index|
        value = mh.remove(rm_handles[index])

        assert_equal(expected_values[index], value, "remove does not return correct value")
      end

      result = Array.new(mh.size) { mh.extract_min }

      assert_equal(expected, result, "heap property violated after remove")
    end
  end

  def test_minheap_remove_regression
    ary = (0..5).to_a
    regression_pairs = ary.map { |key| [key, key] }
    expected = ary[0..-2].sort

    assert_nothing_raised("Exception raised in remove regression test") do
      mh = Ruffman::MinHeap.new
      handles = mh.bulk_insert(*regression_pairs)

      mh.remove(handles.last)
      result = Array.new(mh.size) { mh.extract_min }

      assert_equal(expected, result, "heap property violated after remove")
    end
  end
end
