#######################################################################
# $Id: MakeMaker.pm,v 1.4 2010-11-29 00:39:15 dpchrist Exp $
#######################################################################
# package:
#----------------------------------------------------------------------

package Dpchrist::ExtUtils::MakeMaker;

use strict;
use warnings;

our $VERSION  = sprintf "%d.%03d", q$Revision: 1.4 $ =~ /(\d+)/g;

#######################################################################
# uses:
#----------------------------------------------------------------------

use Carp;
use Data::Dumper;
use File::Basename;
use File::Spec::Functions;

#######################################################################
# constants:
#----------------------------------------------------------------------

use constant RX_UNDEFINED_POSTAMBLE =>
    qr/Can't call method "SUPER::postamble" on an undefined value/;

#######################################################################
# package variables:
#----------------------------------------------------------------------

our %import_args;

#######################################################################

=head1 NAME

Dpchrist::ExtUtils::MakeMaker - additional Makefile targets and rules


=head1 SYNOPSIS

    eval {
	require Dpchrist::ExtUtils::MakeMaker;
	import  Dpchrist::ExtUtils::MakeMaker (
	    mcpani   => $ENV{CPAN_AUTHORID},
	    pod2html => 'lib/Dpchrist/ExtUtils/MakeMaker.pm',
	    readme   => 'lib/Dpchrist/ExtUtils/MakeMaker.pm',
	    release  => $ENV{RELEASE_ROOT},
	);
    };
    warn $@ if $@;

eval(), 'require', and import() are preferred to 'use'
so that 'perl Makefile.PL' will work on systems that don't have
this module installed.


=head1 DESCRIPTION

This documentation describes module revision $Revision: 1.4 $.


This is alpha test level software
and may change or disappear at any time.


This module contains override routines for ExtUtils::MakeMaker
which add optional rules and/or targets to the Makefile
generated by WriteMakefile().

=cut

#######################################################################

=head2 CLASS METHODS

=cut

#----------------------------------------------------------------------

=head3 import

    # Makefile.PL
    eval {
	require Dpchrist::ExtUtils::MakeMaker;
	import Dpchrist::ExtUtils::MakeMaker (
	    [TARGET => ARG,]...
	);
    };
    warn $@ if $@;

Daisy-chains this module's MY::postamble() override into
ExtUtils::MakeMaker for the given Make TARGET.
One or more TARGET => ARG pairs may be given.

Available Make targets are as follows:

=cut

sub import
{
    my $class = shift;

    confess join(' ',
	'Import arguments must be key => value pairs',
	Data::Dumper->Dump([$class, \@_], [qw(class *@)]),
    ) if @_ % 2 != 0;

    %import_args = @_;

    ### The next statement seems to create a symbol table entry for
    ### MY::postamble even if none previously existed, and
    ### $old_postamble seems to be defined in any case:

    my $old_postamble = \&MY::postamble;

    no warnings 'redefine';

    *MY::postamble = sub
    {
	### Call previous MY::postamble:
	my $retval = eval { &$old_postamble() };

	### Warn for all errors other than 'Can't call method
	### "SUPER::postamble" on an undefined value':
	warn $@ if $@ && $@ !~ RX_UNDEFINED_POSTAMBLE;

	$retval .= join('',

	    $import_args{mcpani}	? _mcpani(@_)	: (),

	    $import_args{pod2html}	? _pod2html(@_)	: (),

	    $import_args{readme}	? _readme(@_)	: (),

	    $import_args{release}	? _release(@_)	: (),
	);

	return $retval;
    }
}

=head4 mcpani

    mcpani => AUTHORID,

Adds a Make target 'mcpani'
to the Makefile generated by ExtUtils::MakeMaker::WriteMakefile()
which adds the distribution tarball
to the MCPAN working directory (repository)
and pushes it to the MCPAN local directory
when the following commands are issued:

    $ make dist
    $ make mcpani

Note that you need to run 'make dist'
to create the distribution tarball
before running 'make mcpani'.

AUTHORID is used for the --authorid
parameter to 'mcpani'.
Default is 'NONE'.
I put my CPAN author id (DPCHRIST)
into an environment variable CPAN_AUTHORID in my .bash_profile:

    # .bash_profile
    export CPAN_AUTHORID=DPCHRIST

I then use this environment variable in Makefile.PL:

    # Makefile.PL
    mcpani => $ENV{CPAN_AUTHORID},

You will need a working CPAN::Module::Inject installation
before running 'make mcpani'.  See the following for details:

    perldoc mcpani
    http://www.ddj.com/web-development/184416190
    http://www.stonehenge.com/merlyn/LinuxMag/col42.html

I set an environment variable in .bash_profile that points to my
mcpani configuration file:

    # .bash_profile
    export MCPANI_CONFIG=$HOME/.mcpanirc

Here is my mcpani configuration file:

    # .mcpanirc
    local: /mnt/z/mirror/MCPAN
    remote: ftp://ftp.cpan.org/pub/CPAN ftp://ftp.kernel.org/pub/CPAN
    repository: /home/dpchrist/.mcpani
    passive: yes
    dirmode: 0755

My staging directory is ~/.mcpani.

/mnt/z/mirror/MCPAN is directory on my web server
that is served as http://mirror.holgerdanske.com/MCPAN/.

I can then run cpan on my machines
and have them use the web mirror to fetch my modules
(I only needed to do this once):

    $ sudo cpan
    cpan[1]> o conf urllist http://mirror.holgerdanske.com/MCPAN/
    cpan[2]> o conf commit
    cpan[3]> reload index

Whenever I inject a new or updated module,
I need to reload the cpan index
before I install the module:

    $ sudo cpan
    cpan[1]> reload index
    cpan[2]> install MyModule

=cut

sub _mcpani
{
    my $authorid = $import_args{mcpani} || 'NONE';

    confess join(' ',
	"Bad CPAN author ID for mcpani option",
	Data::Dumper->Dump([$authorid, \%import_args],
			 [qw(authorid   *import_args)]),
    ) unless eval {
	ref \$authorid eq "SCALAR"
	&& $authorid =~ /^[A-Z]/
    };

    return <<EOF;

mcpani ::

	mcpani --add \\
	--module \$(NAME) \\
	--authorid $authorid \\
	--modversion \$(VERSION) \\
	--file \$(DISTVNAME).tar.gz

	mcpani --inject -v
EOF

}

#----------------------------------------------------------------------

=head4 pod2html

    pod2html => [ LIST ],

Adds a rule to the Make target 'all'
in the Makefile generated by ExtUtils::MakeMaker::WriteMakefile()
which will run 'pod2html' against the files in LIST
(e.g. Perl modules and scripts)
using the commands:

    pod2html FILE > PACKAGE-VERSION.html
    rm -f pod2htm?.tmp

PACKAGE and VERSION are determined by reading FILE:

* The namespace of the first 'package' decalaration found
is used for PACKAGE.
If no 'package' declaration is found,
File::Basename::basename(FILE) is used for PACKAGE.

* The argument of the first '$VERSION' variable assignment found
is evaluated and used for VERSION.

HTML files will be generated or updated
whenever the following commands are issued:

    $ make

Or,

    $ make all

If there is only one FILE,
it may be given as the argument to import():

    pod2html => FILE,

=cut

sub _pod2html
{
    my $arg = $import_args{pod2html};

    my @files = (   ref $arg eq "ARRAY"
		    ? @$arg
		    : ($arg)
		);

    my $frag;

    foreach my $file (@files) {

	confess join(' ',
	    'Bad file name for pod2html option',
	    Data::Dumper->Dump([$file, \%import_args],
			     [qw(file   *import_args)]),
    	) unless eval {
    	    ref \$file eq "SCALAR"
    	    && -e $file
	};

	my $package;
	my $version;
	open(F, $file)
	    or confess join(' ',
		"Failed to open file '$file': $!",
	    );
	my $inpod = 0;
	while (<F>) {
    	    $inpod = 1 if $_ =~ /^=\w/;
	    $inpod = 0 if $_ =~ /^=cut/;
	    next if $inpod;

	    $package = $1
		if $_ =~ /^package\s+([\w\:]+);/;
	    $version = eval $1
		if $_ =~ /\$VERSION\s+=\s+(.+)/;
	    last if $package && $version;
	}
	close F
	    or confess "Failed to close file '$file': $!";

	$package = basename($file) unless $package;

	confess join(' ',
	    "Unable to find package name and/or version",
	    "for file '$file'",
	    Data::Dumper->Dump([$package, $version],
			     [qw(package   version)]),
	) unless $package && $version;

	$package =~ s/\:\:/-/g;

	my $html = $package . '-' . $version . '.html';

    	$frag .= <<EOF;

all :: $html

$html :: $file
	pod2html \$< > $html
	rm -f pod2htm?.tmp
EOF

    }

    return $frag;
}

#----------------------------------------------------------------------

=head4 readme

    readme => FILE,

Adds a rule to the Make target 'all'
in the Makefile generated by ExtUtils::MakeMaker::WriteMakefile()
which will run 'pod2text' against FILE
(e.g. Perl module)
using the command:

    pod2text FILE > README

The README file will be generated or updated
whenever the following commands are issued:

    $ make

Or,

    $ make all

=cut

sub _readme
{
    my $file = $import_args{readme};

    confess join(' ',
	'Bad file name for readme option',
	Data::Dumper->Dump([$file, \%import_args],
			 [qw(file   *import_args)]),
    ) unless eval {
	ref \$file eq "SCALAR"
	&& -e $file
    };

    my $frag = <<EOF;

all :: README

README :: $file
	pod2text \$< > README
EOF

    return $frag;
}

#----------------------------------------------------------------------

=head4 release

    release => RELEASE_ROOT,

Adds a Make target 'release'
to the Makefile generated by ExtUtils::MakeMaker::WriteMakefile()
which copies all *.tar.gz and *.html files
to a subdirectory under RELEASE_ROOT
that is named after the module
(changing double colons to a single dash)
when the following commands are issued:

    $ make dist
    $ make release

Note that you should run 'make dist'
to create the distribution tarball before running 'make mcpani'.

I set an environment variable in my .bash_profile:

    # .bash_profile
    export RELEASE_ROOT=/mnt/z/data/release

and use this environment variable in Makefile.PL:

    # Makefile.PL
    release => $ENV{RELEASE_ROOT},

=cut

sub _release
{
    my $root = $import_args{release};

    confess join(' ',
	"Bad directory name for release option",
	Data::Dumper->Dump([$root, \%import_args],
			 [qw(root   *import_args)]),
    ) unless eval {
	ref \$root eq "SCALAR"
    };

    return <<EOF;

release ::
	mkdir -p $root/\$(DISTNAME)
	cp -i *.tar.gz *.html $root/\$(DISTNAME)
EOF
}

#######################################################################
# end of code:
#----------------------------------------------------------------------

1;

__END__

#######################################################################

=head2 EXPORT

None.


=head1 INSTALLATION

Old school:

    $ perl Makefile.PL
    $ make
    $ make test
    $ make install

Minimal:

    $ cpan Dpchrist::ExtUtils::MakeMaker

Complete:

    $ cpan Bundle::Dpchrist


The following warnings should not prevent installation:

    Can't locate Dpchrist/ExtUtils/MakeMaker.pm in @INC (@INC contains: 
    /etc/perl /usr/local/lib/perl/5.10.0 /usr/local/share/perl/5.10.0 /u
    sr/lib/perl5 /usr/share/perl5 /usr/lib/perl/5.10 /usr/share/perl/5.1
    0 /usr/local/lib/site_perl .) at Makefile.PL line XX.

    Can't locate Test/Manifest.pm in @INC (@INC contains: /etc/perl /usr
    /local/lib/perl/5.10.0 /usr/local/share/perl/5.10.0 /usr/lib/perl5 /
    usr/share/perl5 /usr/lib/perl/5.10 /usr/share/perl/5.10 /usr/local/l
    ib/site_perl .) at (eval X) line XX.
    BEGIN failed--compilation aborted at (eval X) line XX.


=head2 PREREQUISITES

See Makefile.PL in the source distribution root directory.


=head1 SEE ALSO

    mcpani
    pod2text
    pod2html
    ExtUtils::MakeMaker
    ExtUtils::MM_Unix
    Programming Perl, 3 e., Ch. 29 "use" (pp. 822-823).
    Mastering Perl, Ch 10 "Replacing Module Parts" (pp. 160-162).


=head1 AUTHOR

David Paul Christensen dpchrist@holgerdanske.com


=head1 COPYRIGHT AND LICENSE

Copyright 2010 by David Paul Christensen dpchrist@holgerdanske.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; version 2.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,
USA.

=cut

#######################################################################
