/***********************************************************
 * YUVFPS with motion estimator for the mjpegtools         *
 * ------------------------------------------------------- *
 * (c) 2005, Jerome Cornet <jcornet@free.fr>               *
 *                                                         *
 * This was shamelessy inspired from the yuvdeinterlace    *
 * (C) 2001-2004 Stefan Fendt                              *
 * Thanks man, I took all the good things from you ;-)     *
 *                                                         *
 * Licensed and protected by the GNU-General-Public-       *
 * License version 2 or if you prefer any later version of *
 * that license). See the file LICENSE for detailed infor- *
 * mation.                                                 *
 *                                                         *
 * FILE: main.c                                            *
 *                                                         *
 ***********************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "yuv4mpeg.h"
#include "motionsearch_deint.h"

int search_radius = 32;
int verbose = 0;
int fast_mode = 0;
int width = 0;
int height = 0;
int input_chroma_subsampling = 0;
int output_chroma_subsampling = 0;

uint8_t *inframe[3];
uint8_t *outframe[3];
uint8_t *frame1[3];
uint8_t *reconstructed[3];

int buff_offset;
int buff_size;

int block_size = 8;
int search_path_radius = 8;
float percent_frame;
int scene_change_threshold = 16;

/* Helper function that add numers represented by fractions */
y4m_ratio_t
add_ratio (y4m_ratio_t a, y4m_ratio_t b)
{
  y4m_ratio_t sum;

  sum.n = a.n * b.d + b.n * a.d;
  sum.d = a.d * b.d;
  y4m_ratio_reduce (&sum);
  return sum;
}

y4m_ratio_t
ratio_minus_1 (y4m_ratio_t a)
{
  y4m_ratio_t sum;
  sum.n = a.n - a.d;
  sum.d = a.d;
  y4m_ratio_reduce (&sum);
  return sum;
}

/***********************************************************
 * Main Loop                                               *
 ***********************************************************/


int
main (int argc, char *argv[])
{
  extern char *optarg;
  char c;
  int fd_in = 0;
  int fd_out = 1;
  int errno = 0;
  int have_framerate = 0;
  int force_interlacing = 0;
  y4m_frame_info_t iframeinfo;
  y4m_stream_info_t istreaminfo;
  y4m_frame_info_t oframeinfo;
  y4m_stream_info_t ostreaminfo;
  int output_frame_number = 0;
  int input_frame_number = 0;
  y4m_ratio_t output_frame_rate, input_frame_rate, frame_rate_ratio;
  float ratio = 0;		// input/output, output should be > input )
  int scene_change;
  y4m_ratio_t ratio_percent_frame;
  float percent_threshold = 0.02;

/* percent_threshold is there to avoid interpolating frames when the output frame
 * is very close to an input frame
 */

  mjpeg_info ("-------------------------------------------------");
  mjpeg_info ("   Motion-Compensating-Frame-Rate-Converter     ");
  mjpeg_info ("-------------------------------------------------");

  while ((c = getopt (argc, argv, "hvb:p:r:t:s:f")) != -1)
    {
      switch (c)
	{
	case 'h':
	  {
	    mjpeg_info ("Usage ");
	    mjpeg_info ("-------------------------");
	    mjpeg_info ("  This program converts frame rates");
	    mjpeg_info (
		       "with a smart algorithm that estimates the motion of the elements");
	    mjpeg_info (
		       "to smooth the motion, rather than duplicating frames.");
	    mjpeg_info (
		       "  It's way smoother, but introduces a bit of blocking and/or");
	    mjpeg_info (
		       " maybe blurryness when things move too fast.");
	    mjpeg_info (" ");
	    mjpeg_info (
		       " -r Frame rate for the resulting stream (in X:Y fractional form)");
	    mjpeg_info (
		       " -b block size (default = 8, will be rounded to even number )");
	    mjpeg_info (
		       " -p search path radius (default = 8, do not use high values ~ > 20)");
	    mjpeg_info (
		       "-t frame approximation threshold (default=50, higher=better)");
	    mjpeg_info (
		       "-s scene change threshold (default=8, 0=disable scene change detection)");
	    mjpeg_info (
		       "-r Frame rate for the resulting stream (in X:Y fractional form)");
	    mjpeg_info (
		       " -f force processing interlaced input (don't know what it does)");

	    mjpeg_info (" -v verbose/debug");

	    exit (0);
	    break;
	  }
	case 'v':
	  {
	    verbose = 1;
	    break;
	  }
	case 'f':
	  {
	    force_interlacing = 1;
	    break;
	  }
	case 'b':
	  {
	    block_size = strtol (optarg, (char **) NULL, 10);
	    /* we only want even block sizes */
	    if (block_size % 1 != 0)
	      {
		block_size = block_size + 1;
		mjpeg_warn ("Block size changed to %d", block_size);
	      }
	    else
	      mjpeg_info ("Block size: %d", block_size);
	    break;
	  }
	case 'p':
	  {
	    search_path_radius = strtol (optarg, (char **) NULL, 10);	/* safer atoi */
	    mjpeg_info ("Search radius %d", search_path_radius);

	    break;
	  }
	case 'r':
	  {
	    if (Y4M_OK != y4m_parse_ratio (&output_frame_rate, optarg))
	      mjpeg_error_exit1
		("Syntax for frame rate should be Numerator:Denominator");


	    mjpeg_info ("New Frame rate %d:%d",
		       output_frame_rate.n, output_frame_rate.d);
	    have_framerate = 1;
	    break;
	  }
	case 't':
	  {
	    percent_threshold = strtol (optarg, (char **) NULL, 10);
	    if ((percent_threshold > 1) && (percent_threshold <= 1024))
	      percent_threshold = 1.0 / percent_threshold;
	    else
	      mjpeg_error_exit1 ("Threshold should be between 2 and 1024");

	    mjpeg_info ("Approximation threshold %d",
		       (int) ((float) 1.0 / percent_threshold));
	    break;

	  }
	case 's':
	  {
	    scene_change_threshold = strtol (optarg, (char **) NULL, 10);
	    if (scene_change_threshold == 0)
	      mjpeg_info ("Scene change detection disabled");
	    else
	      mjpeg_info ("Scene change threshold: %d00 percent",
			 scene_change_threshold);
	    break;

	  }
	}
    }

  if (!have_framerate)
      mjpeg_error_exit1 ("Please specify a frame rate; yuvmotionfps -h for more info");

  /* initialize motion_library */
  init_motion_search ();

  /* initialize stream-information */
  y4m_accept_extensions (1);
  y4m_init_stream_info (&istreaminfo);
  y4m_init_frame_info (&iframeinfo);
  y4m_init_stream_info (&ostreaminfo);
  y4m_init_frame_info (&oframeinfo);

  /* open input stream */
  if ((errno = y4m_read_stream_header (fd_in, &istreaminfo)) != Y4M_OK)
      mjpeg_error_exit1("Couldn't read YUV4MPEG header: %s!", y4m_strerr (errno));

  /* get format information */
  width = y4m_si_get_width (&istreaminfo);
  height = y4m_si_get_height (&istreaminfo);
  input_chroma_subsampling = y4m_si_get_chroma (&istreaminfo);
  mjpeg_info ("Y4M-Stream is %ix%i(%s)",
	     width,
	     height,
	     input_chroma_subsampling ==
	     Y4M_CHROMA_420JPEG ? "4:2:0 MPEG1" : input_chroma_subsampling
	     ==
	     Y4M_CHROMA_420MPEG2 ? "4:2:0 MPEG2" :
	     input_chroma_subsampling ==
	     Y4M_CHROMA_420PALDV ? "4:2:0 PAL-DV" :
	     input_chroma_subsampling ==
	     Y4M_CHROMA_444 ? "4:4:4" : input_chroma_subsampling ==
	     Y4M_CHROMA_422 ? "4:2:2" : input_chroma_subsampling ==
	     Y4M_CHROMA_411 ? "4:1:1 NTSC-DV" : input_chroma_subsampling
	     ==
	     Y4M_CHROMA_MONO ? "MONOCHROME" : input_chroma_subsampling ==
	     Y4M_CHROMA_444ALPHA ? "4:4:4:4 ALPHA" : "unknown");

  /* if chroma-subsampling isn't supported bail out ... */
  switch (input_chroma_subsampling)
    {
    case Y4M_CHROMA_420JPEG:
      break;
    case Y4M_CHROMA_420PALDV:
    case Y4M_CHROMA_420MPEG2:
    case Y4M_CHROMA_411:
      mjpeg_warn ( "This chroma subsampling mode has not been thoroughly tested");
      break;
    default:

      mjpeg_error_exit1
	("Y4M-Stream is not 4:2:0. Other chroma-modes currently not allowed. Sorry.");
    }

  /* the output is progressive 4:2:0 MPEG 1 */
  y4m_si_set_interlace (&ostreaminfo, Y4M_ILACE_NONE);
  y4m_si_set_chroma (&ostreaminfo, Y4M_CHROMA_420JPEG);
  y4m_si_set_width (&ostreaminfo, width);
  y4m_si_set_height (&ostreaminfo, height);
  y4m_si_set_sampleaspect (&ostreaminfo,
			   y4m_si_get_sampleaspect (&istreaminfo));

  input_frame_rate = y4m_si_get_framerate (&istreaminfo);

  y4m_si_set_framerate (&ostreaminfo, output_frame_rate);

  if (width % block_size != 0)
    {
      mjpeg_warn ("Warning, stream width(%d) is not a multiple of block_size (%d)",
		 width, block_size);
      mjpeg_warn ( "The right side of the image might not be what you want");
    }
  if (height % block_size != 0)
    {
      mjpeg_warn ("Warning, stream height(%d) is not a multiple of block_size (%d)",
		 height, block_size);
      mjpeg_warn ( "The lower side of the image might not be what you want");
    }



  /* Calculate the different ratios:
   * ratio is (input framerate / output framerate)
   * ratio_percent_frame is the fractional representation of percent frame
   */
  frame_rate_ratio.n = input_frame_rate.n * output_frame_rate.d;
  frame_rate_ratio.d = input_frame_rate.d * output_frame_rate.n;
  y4m_ratio_reduce (&frame_rate_ratio);
  ratio = (float) frame_rate_ratio.n / frame_rate_ratio.d;

  ratio_percent_frame.d = 1;
  ratio_percent_frame.n = 0;

  if (ratio == 0)
    mjpeg_error_exit1 ("Cannot have ratio =0 ");
  else if (ratio > 128)
    mjpeg_error_exit1 ("Cannot have ratio >128  ");


  if ((y4m_si_get_interlace (&istreaminfo) != Y4M_ILACE_NONE) && (!force_interlacing))
      mjpeg_error_exit1 ("Sorry, can only convert progressive streams");

  /* write the outstream header */
  y4m_write_stream_header (fd_out, &ostreaminfo);

    /* calculate the memory offset needed to allow the processing
     * functions to overshot. The biggest overshot is needed for the
     * MC-functions, so we'll use 8*width...
     */
    buff_offset = width * 8;
    buff_size = buff_offset * 2 + width * height;

    inframe[0] = (uint8_t *) bufalloc (buff_offset + buff_size);
    inframe[1] = (uint8_t *) bufalloc (buff_offset + buff_size);
    inframe[2] = (uint8_t *) bufalloc (buff_offset + buff_size);

    reconstructed[0] = (uint8_t *) bufalloc (buff_offset + buff_size);
    reconstructed[1] = (uint8_t *) bufalloc (buff_offset + buff_size);
    reconstructed[2] = (uint8_t *) bufalloc (buff_offset + buff_size);

    frame1[0] = (uint8_t *) bufalloc (buff_offset + buff_size);
    frame1[1] = (uint8_t *) bufalloc (buff_offset + buff_size);
    frame1[2] = (uint8_t *) bufalloc (buff_offset + buff_size);

    mjpeg_info ("Buffers allocated.");

  /* initialize motion-search-pattern */
  init_search_pattern ();

  errno = y4m_read_frame (fd_in, &istreaminfo, &iframeinfo, frame1);
  if (errno != Y4M_OK)
    goto The_end;

  /* read every frame until the end of the input stream and process it */
  while (Y4M_OK == (errno = y4m_read_frame (fd_in,
					    &istreaminfo,
					    &iframeinfo, inframe)))
    {
/* frame1 contains the previous input frame
 * inframe contains the current input frame
 * reconstructed contains the current output frame
 * percent_frame is the amount of time after which the output frame is sent 
 * 	in percent of the time between input frames
 *
 * Input:
 * frame1 . . . . . . . . . . . . . . . . . . inframe
 * Output: 
 * . . . . . . . . . . .reconstructed. . . . . . . 
 * |<- - percent_frame - - - ->|
 * |< - - - - - - - - - -100% - - - - - - - - - >|
 *
 * The variable ratio_percent_frame is the fractional representation of
 * percent_frame; it is there to avoid rounding errors 
 */
      input_frame_number++;

      if (verbose)
	  mjpeg_info ("Input frame number %d", input_frame_number);

      while (percent_frame < (1.0 - percent_threshold))
	{
	  output_frame_number++;
	  if (verbose)
	      mjpeg_info ("Output frame number %d", output_frame_number);

#define ABS(value) ((value)<0)?-(value):(value)

	  if (ABS (percent_frame) <= percent_threshold)
	    {
	      /* I put a threshold here to avoid wasting time */
	      /* The output frame coincides with the input frame
	       * so there is no need to do any processing 
	       * just copy the input frame as is */
	      y4m_write_frame (fd_out, &ostreaminfo, &oframeinfo, frame1);
	      if (verbose)
		mjpeg_info ("Percent %f rounded to next frame",
			   percent_frame);
	    }
	  else
	    {
	      /* We have to interpolate the frame (between the current inframe
	       * and the previous frame1 
	       * if there is a scene change, motion_compensate_field will
	       * return 1 and we use the previous frame */

	      if (verbose)
		mjpeg_info ("Percent %f", percent_frame);

	      scene_change = motion_compensate_field ();
	      if (scene_change)
		{
		  mjpeg_info ("Scene change at frame %d",
			     input_frame_number);
		  y4m_write_frame (fd_out, &ostreaminfo, &oframeinfo, frame1);
		}
	      else
		{
		  y4m_write_frame (fd_out, &ostreaminfo, &oframeinfo,
				   reconstructed);
		}
	    }
	  ratio_percent_frame =
	    add_ratio (ratio_percent_frame, frame_rate_ratio);
	  percent_frame = Y4M_RATIO_DBL (ratio_percent_frame);

	}

      /* Skip input frames if downsampling  (ratio > 1)
       * when upsampling, ratio < 1
       *    so we have ( 1< percent_frame < 2) at this point 
       *    hence we don't go in in the loop */
      while (percent_frame >= 2)
	{
	  percent_frame = percent_frame - 1;
	  ratio_percent_frame = ratio_minus_1 (ratio_percent_frame);
	  if (Y4M_OK !=
	      (errno =
	       y4m_read_frame (fd_in, &istreaminfo, &iframeinfo, inframe)))
	    goto The_end;
	}
      ratio_percent_frame = ratio_minus_1 (ratio_percent_frame);
      percent_frame = percent_frame - 1;

      /* store the previous frame */
      memcpy (frame1[0], inframe[0], width * height);
      memcpy (frame1[1], inframe[1], width * height / 4);
      memcpy (frame1[2], inframe[2], width * height / 4);

    }

The_end:

  /* free allocated buffers */
  {
    free (inframe[0]);
    free (inframe[1]);
    free (inframe[2]);

    free (reconstructed[0]);
    free (reconstructed[1]);
    free (reconstructed[2]);

    free (frame1[0]);
    free (frame1[1]);
    free (frame1[2]);

    mjpeg_info ("Buffers freed.");
  }

  /* did stream end unexpectedly ? */
  if (errno != Y4M_ERR_EOF)
    mjpeg_error_exit1 ("%s", y4m_strerr (errno));

  /* Exit gently */
  return (0);
}
