/***********************************************************
 * YUVFPS with motion estimator for the mjpegtools         *
 * ------------------------------------------------------- *
 * (c) 2005, Jerome Cornet <jcornet@free.fr>               *
 *                                                         *
 * This was shamelessy inspired from the yuvdeinterlace    *
 * (C) 2001-2004 Stefan Fendt                              *
 * Thanks man, I took all the good things from you ;-)     *
 *                                                         *
 * Licensed and protected by the GNU-General-Public-       *
 * License version 2 or if you prefer any later version of *
 * that license). See the file LICENSE for detailed infor- *
 * mation.                                                 *
 *                                                         *
 * FILE: motionsearch_deint.c                              *
 *                                                         *
 ***********************************************************/

#include <string.h>
#include "yuv4mpeg.h"
#include "motionsearch_deint.h"
#include "transform_block.h"
#include "vector.h"

/*
 * from mjpegtools motionsearch.h  - if this isn't declared here then
 * systems with different linking scope rules will get a strangely initialized
 * version from the shared library.
*/
extern int (*psumsq) (uint8_t *blk1, uint8_t *blk2,
                                    int rowstride, int hx, int hy, int h);

extern int width;
extern int height;
extern uint8_t *inframe[3];
extern uint8_t *outframe[3];
extern uint8_t *frame1[3];
extern uint8_t *reconstructed[3];
extern float percent_frame;
extern int block_size;
extern int search_path_radius;
extern int scene_change_threshold;

uint32_t mean_SAD = 0;


#define PATTERN_SIZE 1255
struct
{
  int x;
  int y;
} pattern[PATTERN_SIZE];

int pattern_length;

void
init_search_pattern (void)
{
  int x, y, r, cnt, i;
  int already_in_path;

  cnt = 0;
  for (r = 0; r <= (search_path_radius * search_path_radius); r++)
    {
      for (y = -2 * search_path_radius; y <= 2 * search_path_radius; y += 2)
	for (x = -2 * search_path_radius; x <= 2 * search_path_radius; x++)
	  {
	    if (r >= (x * x + y * y))
	      {
		/* possible candidate */
		/* check if it's already in the path */
		already_in_path = 0;
		for (i = 0; i <= cnt; i++)
		  {
		    if (pattern[i].x == x && pattern[i].y == y)
		      {
			already_in_path = 1;
		      }
		  }
		/* if it's not already in the path, store it */
		if (already_in_path == 0)
		  {
		    pattern[cnt].x = x;
		    pattern[cnt].y = y;
		    cnt++;
		    if (cnt >= PATTERN_SIZE)
		      mjpeg_error_exit1 ("Search radius too big");
		  }
	      }
	  }
    }
  pattern_length = cnt + 1;
  mjpeg_info ("Search-Path-Length = %i", cnt + 1);
}

/* TODO: fix this; psad_00 calculates SAD for a block of 16 by 2*blocksize 
so the block size is actually not really what the motion is calculated on*/

struct vector
search_backward_vector (int x, int y)
{
  uint32_t min, SAD;
  int dx, dy;
  struct vector v;
  int i;

  x -= 4;
  y -= 4;

  min = 100 * psumsq(inframe[0] + (x) + (y) * width, 
                     frame1[0] + (x) + (y) * width,
                     width,
                     0, 0,
                     2 * block_size);
  v.x = v.y = 0;
  min /= 350;

  if (min > mean_SAD)
    for (i = 1; i <= pattern_length; i++)
      {
	dx = pattern[i].x;
	dy = pattern[i].y;

	SAD = psumsq(inframe[0] + (x) + (y) * width,
                     frame1[0] + (x + dx) + (y + dy) * width,
                     width,
                     0, 0,
                     2 * block_size);

	if (SAD < min)
	  {
	    min = SAD * 100;
	    min /= 100;
	    v.x = dx;
	    v.y = dy;
	  }
	if (min <= mean_SAD)
	  {
	    break;
	  }
      }

  v.min = min;

  return v;			/* return the found vector */
}
static float average_difference = 0;

/* returns 0 if everything went well, 1 if there is a scene change */
int
motion_compensate_field (void)
{
  int x, y;
  struct vector fv;
  struct vector bv;
  float thisframe_difference = 0;
  long number_of_blocks = 0;

  for (y = 0; y < height - block_size + 1; y += block_size)
    {
      for (x = 0; x < width - block_size + 1; x += block_size)
	{
	  /* search the forward and backward motion-vector using
	   * an early terminating circular full-search. Please do
	   * not fiddle with the motion-search itself, as that one
	   * used was the only I could find which met the following
	   * conditions:
	   *
	   * a) consistent vectors
	   * b) shortest possible vectors
	   * c) near alias-free vectors
	   *
	   */

	  fv = bv = search_backward_vector (x, y);

	  bv.x = (signed int) ((float) bv.x * percent_frame);
	  bv.y = (signed int) ((float) bv.y * percent_frame);
	  fv.x = (signed int) (fv.x - bv.x);
	  fv.y = (signed int) (fv.y - bv.y);	/* I do this to avoid rounding errors */

	  /* Search for out of bounds indexes */
	  if (x - fv.x < 0)
	    fv.x = x;
	  if (y - fv.y < 0)
	    fv.y = y;
	  if (x - fv.x >= width)
	    fv.x = width - x - 1;
	  if (y - fv.y >= height)
	    fv.y = height - y - 1;

	  if (x + bv.x < 0)
	    bv.x = x;
	  if (y + bv.y < 0)
	    bv.y = y;
	  if (x + bv.x >= width)
	    bv.x = width - x - 1;
	  if (y + bv.y >= height)
	    bv.y = height - y - 1;

	  /* finally transform the macroblock using a blended variant of
	   * forward and backward block
	   */
	  transform_block
	    (reconstructed[0] + x + y * width,
	     inframe[0] + (x - fv.x) + (y - fv.y) * width,
	     frame1[0] + (x + bv.x) + (y + bv.y) * width, width,
	     percent_frame);

	  transform_block_chroma
	    (reconstructed[1] + (x / 2) + (y / 2) * width / 2,
	     inframe[1] + ((x - fv.x) / 2) + ((y - fv.y) / 2) * width / 2,
	     frame1[1] + ((x + bv.x) / 2) + ((y + bv.y) / 2) * width / 2,
	     width / 2, percent_frame);

	  transform_block_chroma
	    (reconstructed[2] + (x / 2) + (y / 2) * width / 2,
	     inframe[2] + ((x - fv.x) / 2) + ((y - fv.y) / 2) * width / 2,
	     frame1[2] + ((x + bv.x) / 2) + ((y + bv.y) / 2) * width / 2,
	     width / 2, percent_frame);


	  thisframe_difference += fv.min;
	  number_of_blocks += 1;
	}
      /* Finish off the last rows if the width is not multiple of block_size */
      for (; x < width; x++)	/* No need to initialize x, it's already the good value */
	{
	  int index = x + y * width;
	  reconstructed[0][index] = (int) ((float)
					   inframe[0][index] * percent_frame +
					   frame1[0][index] * (1 -
							       percent_frame));
	  index = x / 2 + y / 2 * width / 2;
	  reconstructed[1][index] = (int) ((float)
					   inframe[1][index] * percent_frame +
					   frame1[1][index] * (1 -
							       percent_frame));
	  reconstructed[2][index] =
	    (int) ((float) inframe[2][index] * percent_frame +
		   frame1[2][index] * (1 - percent_frame));

	}
    }
  /* Finish off the last lines if the height is not multiple of block_size */

  for (; y < height; y++)	/* No need to initialize y, it's already the good value */
    {
      for (x = 0; x < width; x++)
	{
	  int index = x + y * width;
	  reconstructed[0][index] = (int) ((float)
					   inframe[0][index] * percent_frame +
					   frame1[0][index] * (1 -
							       percent_frame));
	  index = x / 2 + y / 2 * width / 2;
	  reconstructed[1][index] = (int) ((float)
					   inframe[1][index] * percent_frame +
					   frame1[1][index] * (1 -
							       percent_frame));
	  reconstructed[2][index] =
	    (int) ((float) inframe[2][index] * percent_frame +
		   frame1[2][index] * (1 - percent_frame));
	}
    }
/* if scene_change_treshold !=0, calculate if the scene has changed */
  if ((scene_change_threshold)
      && (scene_change_threshold * average_difference <
	  thisframe_difference / number_of_blocks))
    {
      /* There is a scene change */
      average_difference = thisframe_difference / number_of_blocks;
      return 1;
    }

  average_difference = thisframe_difference / number_of_blocks;
  /* mjpeg_info("Average difference: %f",average_difference); */

/* everything went well, return */
  return 0;
}
