/*
 * Canvas.cs - Graphic canvas objects.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using XWindows.Types;

/// <summary>
/// <para>The <see cref="T:XWindows.Canvas"/> class manages a graphic
/// context for a <see cref="T:XWindows.Drawable"/> instance for
/// drawing to an X display server.</para>
/// </summary>
///
/// <remarks>
/// <para>Graphics canvases should not be created and used for long
/// periods of time.  This is in opposition to normal X practice.</para>
///
/// <para>Most X widget systems initialize graphical contexts at startup
/// time and then use them over and over after that.  However, this can make
/// it difficult to handle changes to the user's preferred color theme.</para>
///
/// <para>Note: although <see cref="T:XWindows.Canvas"/> objects should not
/// be used for long periods of time, the underlying Xlib graphics contexts
/// are reused to avoid over-allocation of X display server resources.
/// The color theme will be properly tracked when Xlib contexts are reused.
/// This behaviour is transparent to the user of the class.</para>
///
/// <para>There are two ways to obtain a canvas.  The first is via the
/// <c>Paint</c> event on a widget.  You should always use the supplied
/// canvas in this case, because it has already been initialized with the
/// correct clipping region to perform the paint operation.</para>
///
/// <para>The second way to obtain a canvas is by directly constructing
/// it from a widget or pixmap, and then disposing it when drawing
/// operations are complete.</para>
/// </remarks>
public sealed class Canvas : IDisposable
{
	// Internal state.
	private Display dpy;
	private Drawable drawable;
	private Xlib.Drawable drawableHandle;
	private IntPtr gc;
	private Color foreground;
	private Color background;
	private Pixmap tile;
	private Bitmap stipple;

	/// <summary>
	/// <para>Constructs a new <see cref="T:XWindows.Canvas"/> object and
	/// attaches it to a <see cref="T:XWindows.Drawable"/> instance.</para>
	/// </summary>
	///
	/// <param name="drawable">
	/// <para>The drawable to attach this graphics context to.  If the
	/// drawable is a widget, the foreground and background colors of the
	/// canvas will be initially set to the widget's standard foreground and
	/// background colors.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>Raised if <paramref name="drawable"/> is <see langword="null"/>.
	/// </para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XInvalidOperationException">
	/// <para>Raised if <paramref name="drawable"/> does not support
	/// output, is disposed, or is the root window.</para>
	/// </exception>
	public Canvas(Drawable drawable)
			{
				if(drawable == null)
				{
					throw new ArgumentNullException("drawable");
				}
				else if(drawable.Kind == DrawableKind.InputOnlyWidget)
				{
					throw new XInvalidOperationException
						(S._("X_CanvasIsOutputOnly"));
				}
				else if(drawable is RootWindow)
				{
					throw new XInvalidOperationException
						(S._("X_NonRootOperation"));
				}
				dpy = drawable.dpy;
				this.drawable = drawable;
				/* eve include = new XGCValues() */
				XGCValues gcValues = new XGCValues();
				InputOutputWidget widget = (drawable as InputOutputWidget);
				Bitmap bitmap = (drawable as Bitmap);
				if(widget != null)
				{
					foreground = widget.Foreground;
					background = widget.Background;
				}
				else if(bitmap != null)
				{
					foreground = new Color(0x00, 0x00, 0x00);
					background = new Color(0xFF, 0xFF, 0xFF);
				}
				else
				{
					foreground = new Color (StandardColor.Foreground);
					background = new Color (StandardColor.Background);
				}
				gcValues.foreground = drawable.ToPixel(foreground);
				gcValues.background = drawable.ToPixel(background);
				try
				{
					IntPtr display = dpy.Lock();
					drawableHandle = drawable.GetGCHandle();
					gc = drawable.screen.GetGC(bitmap != null);
					if(gc == IntPtr.Zero)
					{
						// Create a new GC because the cache is empty.
						gc = Xlib.XCreateGC(display, drawableHandle,
											(uint)(GCValueMask.GCForeground |
												   GCValueMask.GCBackground),
											ref gcValues);
						if(gc == IntPtr.Zero)
						{
							Display.OutOfMemory();
						}
					}
					else
					{
						// Reset the cached GC back to the default settings.
						// Xlib will take care of stripping the list down
						// to just the changes that need to be applied.
						gcValues.function = XWindows.Function.GXcopy;
						gcValues.plane_mask = (Xlib.Pixel)(~((ulong)0));
						gcValues.line_width = 0;
						gcValues.line_style = XWindows.LineStyle.LineSolid;
						gcValues.cap_style = XWindows.CapStyle.CapButt;
						gcValues.join_style = XWindows.JoinStyle.JoinMiter;
						gcValues.fill_style = XWindows.FillStyle.FillSolid;
						gcValues.fill_rule = XWindows.FillRule.EvenOddRule;
						gcValues.arc_mode = XWindows.ArcMode.ArcPieSlice;
						gcValues.ts_x_origin = 0;
						gcValues.ts_y_origin = 0;
						gcValues.subwindow_mode =
							XWindows.SubwindowMode.ClipByChildren;
						gcValues.graphics_exposures = true;
						gcValues.clip_x_origin = 0;
						gcValues.clip_y_origin = 0;
						gcValues.clip_mask = Xlib.Pixmap.Zero;
						gcValues.dash_offset = 0;
						gcValues.dashes = (sbyte)4;
						Xlib.XChangeGC(display, gc,
									   (uint)(GCValueMask.GCFunction |
											  GCValueMask.GCPlaneMask |
											  GCValueMask.GCForeground |
											  GCValueMask.GCBackground |
											  GCValueMask.GCLineWidth |
											  GCValueMask.GCLineStyle |
											  GCValueMask.GCCapStyle |
											  GCValueMask.GCJoinStyle |
											  GCValueMask.GCFillStyle |
											  GCValueMask.GCFillRule |
											  GCValueMask.GCTileStipXOrigin |
											  GCValueMask.GCTileStipYOrigin |
											  GCValueMask.GCSubwindowMode |
											  GCValueMask.GCGraphicsExposures |
											  GCValueMask.GCClipXOrigin |
											  GCValueMask.GCClipYOrigin |
											  GCValueMask.GCClipMask |
											  GCValueMask.GCDashOffset |
											  GCValueMask.GCDashList |
											  GCValueMask.GCArcMode),
									   ref gcValues);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Dispose this graphics context object.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method implements the <see cref="T:System.IDisposable"/>
	/// interface.</para>
	/// </remarks>
	public void Dispose()
			{
				try
				{
					IntPtr display = dpy.Lock();
					if(gc != IntPtr.Zero)
					{
						// Release the GC back to the screen's cache so
						// that we can reuse it the next time we need a GC.
						drawable.screen.ReleaseGC(gc, (drawable is Bitmap));
						gc = IntPtr.Zero;
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	// Lock this canvas's display and validate the parameters.
	// Returns the display pointer.
	private IntPtr Lock()
			{
				IntPtr display = dpy.Lock();
				if(drawable.handle != Xlib.Drawable.Zero && gc != IntPtr.Zero)
				{
					// All of the relevant handles are still valid.
					return display;
				}
				else
				{
					// Validate the drawable handle to see if it is disposed.
					drawable.GetGCHandle();

					// The canvas must have been destroyed.
					throw new XInvalidOperationException
								(S._("X_CanvasDestroyed"));
				}
			}

	/// <summary>
	/// <para>Get or set the foreground drawing color.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The foreground <see cref="T:XWindows.Color"/> value.</para>
	/// </value>
	public Color Foreground
			{
				get
				{
					return foreground;
				}
				set
				{
					if(foreground != value)
					{
						try
						{
							IntPtr display = Lock();
							foreground = value;
							Xlib.XSetForeground
								(display, gc, drawable.ToPixel(value));
						}
						finally
						{
							dpy.Unlock();
						}
					}
				}
			}

	/// <summary>
	/// <para>Get or set the background drawing color.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The background <see cref="T:XWindows.Color"/> value.</para>
	/// </value>
	public Color Background
			{
				get
				{
					return background;
				}
				set
				{
					if(background != value)
					{
						try
						{
							IntPtr display = Lock();
							background = value;
							Xlib.XSetBackground
								(display, gc, drawable.ToPixel(value));
						}
						finally
						{
							dpy.Unlock();
						}
					}
				}
			}

	/// <summary>
	/// <para>Get or set the drawing function mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Function"/> value for the mode.</para>
	/// </value>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if set to an invalid value.</para>
	/// </exception>
	public XWindows.Function Function
			{
				get
				{
					try
					{
						IntPtr display = Lock();
						XGCValues values;
						Xlib.XGetGCValues(display, gc,
										  (uint)(GCValueMask.GCFunction),
										  out values);
						return values.function;
					}
					finally
					{
						dpy.Unlock();
					}
				}
				set
				{
					if(value < XWindows.Function.GXclear ||
					   value > XWindows.Function.GXset)
					{
						throw new XException
							(String.Format(S._("X_Function"), (int)value));
					}
					try
					{
						IntPtr display = Lock();
						Xlib.XSetFunction(display, gc, (int)value);
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get or set the line style mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.LineStyle"/> value for the mode.</para>
	/// </value>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if set to an invalid value.</para>
	/// </exception>
	public XWindows.LineStyle LineStyle
			{
				get
				{
					try
					{
						IntPtr display = Lock();
						XGCValues values;
						Xlib.XGetGCValues(display, gc,
										  (uint)(GCValueMask.GCLineStyle),
										  out values);
						return values.line_style;
					}
					finally
					{
						dpy.Unlock();
					}
				}
				set
				{
					if(value < XWindows.LineStyle.LineSolid ||
					   value > XWindows.LineStyle.LineDoubleDash)
					{
						throw new XException
							(String.Format(S._("X_LineStyle"), (int)value));
					}
					try
					{
						IntPtr display = Lock();
						/* eve include  = new XGCValues() */
						XGCValues values = new XGCValues();
						values.line_style = value;
						Xlib.XChangeGC(display, gc,
									   (uint)(GCValueMask.GCLineStyle),
									   ref values);
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get or set the line capping style mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.CapStyle"/> value for the mode.</para>
	/// </value>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if set to an invalid value.</para>
	/// </exception>
	public XWindows.CapStyle CapStyle
			{
				get
				{
					try
					{
						IntPtr display = Lock();
						XGCValues values;
						Xlib.XGetGCValues(display, gc,
										  (uint)(GCValueMask.GCCapStyle),
										  out values);
						return values.cap_style;
					}
					finally
					{
						dpy.Unlock();
					}
				}
				set
				{
					if(value < XWindows.CapStyle.CapNotLast ||
					   value > XWindows.CapStyle.CapProjecting)
					{
						throw new XException
							(String.Format(S._("X_CapStyle"), (int)value));
					}
					try
					{
						IntPtr display = Lock();
						/* eve include  = new XGCValues() */
						XGCValues values = new XGCValues();
						values.cap_style = value;
						Xlib.XChangeGC(display, gc,
									   (uint)(GCValueMask.GCCapStyle),
									   ref values);
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get or set the line join style mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.JoinStyle"/> value for the mode.</para>
	/// </value>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>Raised if set to an invalid value.</para>
	/// </exception>
	public XWindows.JoinStyle JoinStyle
			{
				get
				{
					try
					{
						IntPtr display = Lock();
						XGCValues values;
						Xlib.XGetGCValues(display, gc,
										  (uint)(GCValueMask.GCJoinStyle),
										  out values);
						return values.join_style;
					}
					finally
					{
						dpy.Unlock();
					}
				}
				set
				{
					if(value < XWindows.JoinStyle.JoinMiter ||
					   value > XWindows.JoinStyle.JoinBevel)
					{
						throw new XException
							(String.Format(S._("X_JoinStyle"), (int)value));
					}
					try
					{
						IntPtr display = Lock();
						/* eve include  = new XGCValues() */
						XGCValues values = new XGCValues();
						values.join_style = value;
						Xlib.XChangeGC(display, gc,
									   (uint)(GCValueMask.GCJoinStyle),
									   ref values);
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get the fill style mode.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.FillStyle"/> value for the mode.</para>
	/// </value>
	public XWindows.FillStyle FillStyle
			{
				get
				{
					try
					{
						IntPtr display = Lock();
						XGCValues values;
						Xlib.XGetGCValues(display, gc,
										  (uint)(GCValueMask.GCFillStyle),
										  out values);
						return values.fill_style;
					}
					finally
					{
						dpy.Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get the fill style tiling pixmap.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Pixmap"/> value for the pixmap,
	/// or <see langword="null"/> if the fill style is not
	/// <c>FillTiled</c>.</para>
	/// </value>
	public Pixmap Tile
			{
				get
				{
					return tile;
				}
			}

	/// <summary>
	/// <para>Get the fill style stippling bitmap.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The <see cref="T:XWindows.Bitmap"/> value for the bitmap,
	/// or <see langword="null"/> if the fill style is not
	/// <c>FillStippled</c> or <c>FillOpaqueStippled</c>.</para>
	/// </value>
	public Bitmap Stipple
			{
				get
				{
					return stipple;
				}
			}

	/// <summary>
	/// <para>Set the fill style mode to "solid".</para>
	/// </summary>
	public void SetFillSolid()
			{
				try
				{
					IntPtr display = Lock();
					Xlib.XSetFillStyle
						(display, gc, (int)(XWindows.FillStyle.FillSolid));
					if(tile != null)
					{
						Xlib.XSetTSOrigin(display, gc, 0, 0);
						tile = null;
					}
					else if(stipple != null)
					{
						Xlib.XSetTSOrigin(display, gc, 0, 0);
						stipple = null;
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the fill style mode to "tiled", with a specific
	/// tiling pixmap.</para>
	/// </summary>
	///
	/// <param name="tile">
	/// <para>The tiling pixmap to use.</para>
	/// </param>
	///
	/// <param name="xorigin">
	/// <para>The X co-ordinate of the tiling origin.</para>
	/// </param>
	///
	/// <param name="yorigin">
	/// <para>The Y co-ordinate of the tiling origin.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="tile"/> value is
	/// <see langword="null"/>.</para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="xorigin"/> or <paramref name="yorigin"/>
	/// value is out of range.</para>
	/// </exception>
	///
	/// <exception cref="T:System.XInvalidOperationException">
	/// <para>The <paramref name="tile"/> value is disposed.</para>
	/// </exception>
	public void SetFillTiled(Pixmap tile, int xorigin, int yorigin)
			{
				if(tile == null)
				{
					throw new ArgumentNullException("tile");
				}
				if(xorigin < -32768 || xorigin > 32767 ||
				   yorigin < -32768 || yorigin > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetTile(display, gc, tile.GetPixmapHandle());
					Xlib.XSetFillStyle
						(display, gc, (int)(XWindows.FillStyle.FillTiled));
					Xlib.XSetTSOrigin(display, gc, xorigin, yorigin);
					this.tile = tile;
					stipple = null;
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the fill style mode to "stippled", with a specific
	/// stippling bitmap.</para>
	/// </summary>
	///
	/// <param name="stipple">
	/// <para>The stippling bitmap to use.</para>
	/// </param>
	///
	/// <param name="xorigin">
	/// <para>The X co-ordinate of the stippling origin.</para>
	/// </param>
	///
	/// <param name="yorigin">
	/// <para>The Y co-ordinate of the stippling origin.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="stipple"/> value is
	/// <see langword="null"/>.</para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="xorigin"/> or <paramref name="yorigin"/>
	/// value is out of range.</para>
	/// </exception>
	///
	/// <exception cref="T:System.XInvalidOperationException">
	/// <para>The <paramref name="tile"/> value is disposed.</para>
	/// </exception>
	public void SetFillStippled(Bitmap stipple, int xorigin, int yorigin)
			{
				if(stipple == null)
				{
					throw new ArgumentNullException("stipple");
				}
				if(xorigin < -32768 || xorigin > 32767 ||
				   yorigin < -32768 || yorigin > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetStipple(display, gc, stipple.GetPixmapHandle());
					Xlib.XSetFillStyle
						(display, gc, (int)(XWindows.FillStyle.FillStippled));
					Xlib.XSetTSOrigin(display, gc, xorigin, yorigin);
					this.stipple = stipple;
					tile = null;
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the fill style mode to "opaque stippled", with a specific
	/// stippling bitmap.</para>
	/// </summary>
	///
	/// <param name="stipple">
	/// <para>The stippling bitmap to use.</para>
	/// </param>
	///
	/// <param name="xorigin">
	/// <para>The X co-ordinate of the stippling origin.</para>
	/// </param>
	///
	/// <param name="yorigin">
	/// <para>The Y co-ordinate of the stippling origin.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="stipple"/> value is
	/// <see langword="null"/>.</para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="xorigin"/> or <paramref name="yorigin"/>
	/// value is out of range.</para>
	/// </exception>
	///
	/// <exception cref="T:System.XInvalidOperationException">
	/// <para>The <paramref name="tile"/> value is disposed.</para>
	/// </exception>
	public void SetFillOpaqueStippled(Bitmap stipple, int xorigin, int yorigin)
			{
				if(stipple == null)
				{
					throw new ArgumentNullException("stipple");
				}
				if(xorigin < -32768 || xorigin > 32767 ||
				   yorigin < -32768 || yorigin > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetStipple(display, gc, stipple.GetPixmapHandle());
					Xlib.XSetFillStyle
						(display, gc,
						 (int)(XWindows.FillStyle.FillOpaqueStippled));
					Xlib.XSetTSOrigin(display, gc, xorigin, yorigin);
					this.stipple = stipple;
					tile = null;
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the clip area to a region object.</para>
	/// </summary>
	///
	/// <param name="r">
	/// <para>The clipping region to set.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="r"/> value is
	/// <see langword="null"/>.</para>
	/// </exception>
	public void SetClipRegion(Region r)
			{
				if(r == null)
				{
					throw new ArgumentNullException("r");
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetClipOrigin(display, gc, 0, 0);
					Xlib.XSetRegion(display, gc, r.GetRegion());
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the clip area to a region object, with a
	/// specified origin.</para>
	/// </summary>
	///
	/// <param name="r">
	/// <para>The clipping region to set.</para>
	/// </param>
	///
	/// <param name="xorigin">
	/// <para>The X co-ordinate of the clipping origin.</para>
	/// </param>
	///
	/// <param name="yorigin">
	/// <para>The Y co-ordinate of the clipping origin.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentNullException">
	/// <para>The <paramref name="r"/> value is
	/// <see langword="null"/>.</para>
	/// </exception>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="xorigin"/> or <paramref name="yorigin"/>
	/// value is out of range.</para>
	/// </exception>
	public void SetClipRegion(Region r, int xorigin, int yorigin)
			{
				if(r == null)
				{
					throw new ArgumentNullException("r");
				}
				if(xorigin < -32768 || xorigin > 32767 ||
				   yorigin < -32768 || yorigin > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetClipOrigin(display, gc, xorigin, yorigin);
					Xlib.XSetRegion(display, gc, r.GetRegion());
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the clip area to a bitmap mask</para>
	/// </summary>
	///
	/// <param name="mask">
	/// <para>The mask to set, or <see langword="null"/> to clear the
	/// clip mask.</para>
	/// </param>
	///
	/// <exception cref="T:System.XInvalidOperationException">
	/// <para>The <paramref name="mask"/> value is disposed.</para>
	/// </exception>
	public void SetClipMask(Bitmap mask)
			{
				try
				{
					IntPtr display = Lock();
					Xlib.XSetClipOrigin(display, gc, 0, 0);
					if(mask != null)
					{
						Xlib.XSetClipMask(display, gc, mask.GetPixmapHandle());
					}
					else
					{
						Xlib.XSetClipMask(display, gc, Xlib.Pixmap.Zero);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Set the clip area to a bitmap mask, with a specified
	/// clip origin.</para>
	/// </summary>
	///
	/// <param name="mask">
	/// <para>The mask to set, or <see langword="null"/> to clear the
	/// clip mask.</para>
	/// </param>
	///
	/// <param name="xorigin">
	/// <para>The X co-ordinate of the clipping origin.</para>
	/// </param>
	///
	/// <param name="yorigin">
	/// <para>The Y co-ordinate of the clipping origin.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>The <paramref name="xorigin"/> or <paramref name="yorigin"/>
	/// value is out of range.</para>
	/// </exception>
	///
	/// <exception cref="T:System.XInvalidOperationException">
	/// <para>The <paramref name="mask"/> value is disposed.</para>
	/// </exception>
	public void SetClipMask(Bitmap mask, int xorigin, int yorigin)
			{
				if(xorigin < -32768 || xorigin > 32767 ||
				   yorigin < -32768 || yorigin > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XSetClipOrigin(display, gc, xorigin, xorigin);
					if(mask != null)
					{
						Xlib.XSetClipMask(display, gc, mask.GetPixmapHandle());
					}
					else
					{
						Xlib.XSetClipMask(display, gc, Xlib.Pixmap.Zero);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Draw a line between two points.</para>
	/// </summary>
	///
	/// <param name="x1">
	/// <para>The X co-ordinate of the first point.</para>
	/// </param>
	///
	/// <param name="y1">
	/// <para>The Y co-ordinate of the first point.</para>
	/// </param>
	///
	/// <param name="x2">
	/// <para>The X co-ordinate of the second point.</para>
	/// </param>
	///
	/// <param name="y2">
	/// <para>The Y co-ordinate of the second point.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>One of the co-ordinate values is out of range.</para>
	/// </exception>
	public void DrawLine(int x1, int y1, int x2, int y2)
			{
				if(x1 < -32768 || x1 > 32767 ||
				   y1 < -32768 || y1 > 32767 ||
				   x2 < -32768 || x2 > 32767 ||
				   y2 < -32768 || y2 > 32767)
				{
					throw new XException(S._("X_InvalidPosition"));
				}
				try
				{
					IntPtr display = Lock();
					Xlib.XDrawLine(display, drawableHandle, gc,
								   x1, y1, x2, y2);
				}
				finally
				{
					dpy.Unlock();
				}
			}

	/// <summary>
	/// <para>Draw a line between two points.</para>
	/// </summary>
	///
	/// <param name="p1">
	/// <para>The first point.</para>
	/// </param>
	///
	/// <param name="p2">
	/// <para>The second point.</para>
	/// </param>
	///
	/// <exception cref="T:XWindows.XException">
	/// <para>One of the co-ordinate values is out of range.</para>
	/// </exception>
	public void DrawLine(Point p1, Point p2)
			{
				DrawLine(p1.x, p1.y, p2.x, p2.y);
			}

} // class Canvas

} // namespace XWindows
