/*
 * Colormap.cs - Colormap handling for X applications.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using System.Collections;
using XWindows.Types;

/// <summary>
/// <para>The <see cref="T:XWindows.Colormap"/> class manages colormaps
/// for an X display screen.</para>
/// </summary>
public sealed class Colormap
{
	// Internal state.
	private Display dpy;
	private Screen screen;
	private Xlib.Colormap colormap;
	private Hashtable cachedPixels;

	// Constructor that is called from the "Screen" class.
	internal Colormap(Display dpy, Screen screen, Xlib.Colormap colormap)
			{
				this.dpy = dpy;
				this.screen = screen;
				this.colormap = colormap;
				this.cachedPixels = new Hashtable();
			}

	// Convert an RGB color value into a pixel using this colormap.
	// This assumes that standard colors have already been expanded.
	internal Xlib.Pixel RGBToPixel(Color color)
			{
				try
				{
					// Acquire the display lock.
					IntPtr display = dpy.Lock();

					// Do we already have a cached pixel value for this color?
					Object cached = cachedPixels[color];
					if(cached != null)
					{
						return (Xlib.Pixel)(cached);
					}

					// Try to allocate the color from the X display server.
					XColor xcolor;
					int red = color.Red;
					int green = color.Green;
					int blue = color.Blue;
					xcolor.pixel = Xlib.Pixel.Zero;
					xcolor.red = (ushort)((red << 8) | red);
					xcolor.green = (ushort)((green << 8) | green);
					xcolor.blue = (ushort)((blue << 8) | blue);
					xcolor.flags =
						(sbyte)(XColor.DoRed | XColor.DoGreen | XColor.DoBlue);
					xcolor.pad = (sbyte)0;
					if(Xlib.XAllocColor(display, colormap, ref xcolor) != 0)
					{
						// Cache the value for next time.
						cachedPixels[color] = xcolor.pixel;
						return xcolor.pixel;
					}

					// TODO: do a closest color match for the color.

					// Last ditch: return black or white depending upon
					// the intensity of the color.
					if(((color.Red * 54 + color.Green * 183 +
					     color.Blue * 19) / 256) < 128)
					{
						return Xlib.XBlackPixelOfScreen(screen.screen);
					}
					else
					{
						return Xlib.XWhitePixelOfScreen(screen.screen);
					}
				}
				finally
				{
					dpy.Unlock();
				}
			}

} // class Colormap

} // namespace XWindows
