/*
 * Display.cs - Access an X display server.
 *
 * This file is part of the X# library.
 * Copyright (C) 2002  Southern Storm Software, Pty Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace XWindows
{

using System;
using System.Collections;
using System.Threading;
using XWindows.Events;

/// <summary>
/// <para>The <see cref="T:XWindows.Display"/> class manages connections
/// to X display servers.</para>
///
/// <para>The application normally obtains a <see cref="T:XWindows.Display"/>
/// instance by accessing the <c>Primary</c> property or calling
/// the <c>Open</c> method.</para>
///
/// <para>Connections to multiple X display servers can be open
/// simultaneously.</para>
/// </summary>
public sealed class Display : IDisposable
{
	// Internal state.
	internal IntPtr dpy;
	private String displayName;
	private Screen[] screens;
	private int defaultScreen;
	private bool quit;
	private bool pendingExposes;
	private InputOutputWidget exposeList;
	private bool inMainLoop;
	private Xlib.Cursor[] cursors;
	internal Xlib.Time knownEventTime;
	internal Hashtable handleMap;
	private static Display primary;
	private static Hashtable others;
	private static bool threadsInited;

	// Constructor.
	private Display(IntPtr dpy, String displayName)
			{
				// Copy parameters in from the create process.
				this.dpy = dpy;
				this.displayName = displayName;

				// Create objects for each of the display screens.
				int nscreens = (int)(Xlib.XScreenCount(dpy));
				screens = new Screen [nscreens];
				for(int scr = 0; scr < nscreens; ++scr)
				{
					screens[scr] = new Screen
						(this, scr, Xlib.XScreenOfDisplay(dpy, scr));
				}

				// Get the index of the default screen.
				defaultScreen = (int)(Xlib.XDefaultScreen(dpy));

				// Create an array to hold the standard cursors.
				cursors = new Xlib.Cursor [(int)(CursorType.XC_num_glyphs)];

				// Reset the time of the last known event.
				knownEventTime = Xlib.Time.CurrentTime;

				// Construct the window handle map if not already present.
				if(handleMap == null)
				{
					handleMap = new Hashtable();
				}
			}

	/// <summary>
	/// <para>Close the X display connection if it is currently active.</para>
	/// </summary>
	~Display()
			{
				Close();
			}

	/// <summary>
	/// <para>Close the X display connection if it is currently active.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>This method implements the <see cref="T:System.IDisposable"/>
	/// interface.</para>
	/// </remarks>
	public void Dispose()
			{
				Close();
			}

	/// <summary>
	/// <para>Get the primary X display connection.  The connection will
	/// be opened if it is not currently active.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The primary X display connection.</para>
	/// </value>
	///
	/// <remarks>
	/// <para>The primary X display is normally specified using the
	/// <c>DISPLAY</c> environment variable.</para>
	/// </remarks>
	///
	/// <exception cref="T:XWindows.XCannotConnectException">
	/// <para>A connection to the primary X display server could not
	/// be established.</para>
	/// </exception>
	public static Display Primary
			{
				get
				{
					lock(typeof(Display))
					{
						if(primary != null)
						{
							return primary;
						}
						else
						{
							primary = OpenInternal(DefaultDisplayName());
							return primary;
						}
					}
				}
			}

	// Internal version of "Open()" that is called once the
	// type lock has been acquired.
	private static Display OpenInternal(String displayName)
			{
				try
				{
					try
					{
						// Initialize Xlib thread support.
						if(!threadsInited)
						{
							threadsInited = true;
							Xlib.XInitThreads();
						}
					}
					catch(MissingMethodException)
					{
						// The "X11" library may not have support for
						// threads, which is OK(-ish) because we will
						// be locking every access to the display anyway.
					}
					IntPtr dpy = Xlib.XOpenDisplay(displayName);
					if(dpy != IntPtr.Zero)
					{
						// We have opened the display successfully.
						return new Display(dpy, displayName);
					}
					else
					{
						// We were unable to connect to the display.
						if(displayName != null)
						{
							throw new XCannotConnectException
								(String.Format(S._("X_CannotOpenTo"),
											   displayName));
						}
						else
						{
							throw new XCannotConnectException
								(S._("X_CannotOpen"));
						}
					}
				}
				catch(MissingMethodException)
				{
					// The engine was unable to locate "XOpenDisplay",
					// so we probably don't have an X library, or it
					// is not on the LD_LIBRARY_PATH.
					throw new XCannotConnectException
						(S._("X_LibraryNotPresent"));
				}
			}

	/// <summary>
	/// <para>Open the primary connection to the primary X display
	/// server if it is not currently active.  This method will have
	/// no effect if a primary connection is already available.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The primary X display is normally specified using the
	/// <c>DISPLAY</c> environment variable.</para>
	/// </remarks>
	///
	/// <exception cref="T:XWindows.XCannotConnectException">
	/// <para>A connection to the primary X display server could not
	/// be established.</para>
	/// </exception>
	public static void Open()
			{
				lock(typeof(Display))
				{
					if(primary == null)
					{
						primary = OpenInternal(DefaultDisplayName());
					}
				}
			}

	/// <summary>
	/// <para>Open a connection to a specific X display server, or retrieve
	/// an existing connection to the specified server.</para>
	/// </summary>
	///
	/// <param name="displayName">
	/// <para>The name of the X display server to open a connection to.
	/// May be <see langword="null"/> to specify the primary X display
	/// server.</para>
	/// </param>
	///
	/// <returns>
	/// <para>Returns the <see cref="T:XWindows.Display"/> instance that
	/// corresponds to <paramref name="displayName"/>.</para>
	/// </returns>
	///
	/// <exception cref="T:XWindows.XCannotConnectException">
	/// <para>A connection to <paramref name="displayName"/> could not
	/// be established.</para>
	/// </exception>
	public static Display Open(String displayName)
			{
				if(displayName == null || displayName == DefaultDisplayName())
				{
					// Return the primary display connection.
					return Primary;
				}
				else
				{
					lock(typeof(Display))
					{
						// See if we already have a connection.
						if(others == null)
						{
							others = new Hashtable();
						}
						Display display = (Display)(others[displayName]);
						if(display != null)
						{
							return display;
						}

						// Open a new connection to the specified display.
						display = OpenInternal(displayName);
						others[displayName] = display;
						return display;
					}
				}
			}

	// Get the default display name from the user's environment.
	private static String DefaultDisplayName()
			{
				try
				{
					return Xlib.XDisplayName(null);
				}
				catch(MissingMethodException)
				{
					// The engine was unable to locate "XDisplayName".
					return null;
				}
			}

	/// <summary>
	/// <para>Close the X display connection if it is currently active.</para>
	/// </summary>
	public void Close()
			{
				lock(this)
				{
					if(dpy != IntPtr.Zero)
					{
						lock(typeof(Display))
						{
							// Disassociate window handles from all windows.
							for(int scr = 0; scr < screens.Length; ++scr)
							{
								screens[scr].RootWindow.Disassociate();
							}

							// Remove the object from the global map.
							if(primary == this)
							{
								primary = null;
							}
							else
							{
								others.Remove(displayName);
							}

							// Close the connection to the X server.
							Xlib.XCloseDisplay(dpy);
							dpy = IntPtr.Zero;
						}
					}
				}
			}

	// Lock this display and get the raw display pointer.
	internal IntPtr Lock()
			{
				Monitor.Enter(this);
				if(dpy == IntPtr.Zero)
				{
					throw new XInvalidOperationException
						(S._("X_ConnectionLost"));
				}
				return dpy;
			}

	// Unlock this display.  The correct way to use "Lock" and
	// "Unlock" is as follows:
	//
	//		try
	//		{
	//			IntPtr dpy = display.Lock();
	//			... use the display pointer ...
	//		}
	//		finally
	//		{
	//			display.Unlock();
	//		}
	//
	// This sequence ensures that there can be no race conditions
	// where a raw display pointer is used when the connection is
	// closed or in use by another thread.
	//
	internal void Unlock()
			{
				Monitor.Exit(this);
			}

	/// <summary>
	/// <para>Get the default screen for this display.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The default screen instance.</para>
	/// </value>
	public Screen DefaultScreenOfDisplay
			{
				get
				{
					return screens[defaultScreen];
				}
			}

	/// <summary>
	/// <para>Get the root window for the default screen.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The default root window instance.</para>
	/// </value>
	public RootWindow DefaultRootWindow
			{
				get
				{
					return DefaultScreenOfDisplay.RootWindow;
				}
			}

	/// <summary>
	/// <para>Get the display name associated with this display.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The display name string.</para>
	/// </value>
	public String DisplayString
			{
				get
				{
					try
					{
						return Xlib.XDisplayString(Lock());
					}
					finally
					{
						Unlock();
					}
				}
			}

	/// <summary>
	/// <para>Get the number of screens that are attached to
	/// this display.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The number of screens.</para>
	/// </value>
	public int ScreenCount
			{
				get
				{
					return screens.Length;
				}
			}

	/// <summary>
	/// <para>Get the index of the default screen that is attached
	/// to this display.</para>
	/// </summary>
	///
	/// <value>
	/// <para>The default screen index.</para>
	/// </value>
	public int DefaultScreen
			{
				get
				{
					return defaultScreen;
				}
			}

	/// <summary>
	/// <para>Get a specific screen from this display, by index.</para>
	/// </summary>
	///
	/// <param name="scr">
	/// <para>The index of the screen to get.</para>
	/// </param>
	///
	/// <returns>
	/// <para>The <see cref="T:XWindows.Screen"/> instance that corresponds
	/// to <paramref name="scr"/>.</para>
	/// </returns>
	///
	/// <exception cref="T:System.IndexOutOfRangeException">
	/// <para>Raised if <paramref name="scr"/> is less than zero or
	/// greater than or equal to <c>ScreenCount</c>.</para>
	/// </exception>
	public Screen ScreenOfDisplay(int scr)
			{
				return screens[scr];
			}

	/// <summary>
	/// <para>Flush all pending requests to the X display server.</para>
	/// </summary>
	public void Flush()
			{
				try
				{
					Xlib.XFlush(Lock());
				}
				finally
				{
					Unlock();
				}
			}

	/// <summary>
	/// <para>Synchronize operations against the X display server.</para>
	/// </summary>
	public void Sync()
			{
				try
				{
					Xlib.XSync(Lock(), Xlib.Bool.False);
				}
				finally
				{
					Unlock();
				}
			}

	/// <summary>
	/// <para>Ring the X display server bell.</para>
	/// </summary>
	///
	/// <param name="percent">
	/// <para>The percentage of the base bell volume to ring at.
	/// This must be between -100 and 100 inclusive.</para>
	/// </param>
	///
	/// <exception cref="T:System.ArgumentOfRangeException">
	/// <para>The <paramref name="percent"/> value is less than
	/// -100 or greater than 100.</para>
	/// </exception>
	public void Bell(int percent)
			{
				try
				{
					IntPtr dpy = Lock();
					if(percent >= -100 && percent <= 100)
					{
						Xlib.XBell(dpy, percent);
					}
					else
					{
						throw new ArgumentOutOfRangeException
							("percent", S._("X_BellPercent"));
					}
				}
				finally
				{
					Unlock();
				}
			}

	/// <summary>
	/// <para>Ring the X display server bell at top volume.</para>
	/// </summary>
	public void Bell()
			{
				Bell(100);
			}

	/// <summary>
	/// <para>Run the main event loop on this display.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The main event loop will run until the <c>Quit</c>
	/// method is called.</para>
	/// </remarks>
	public void Run()
			{
				Console.WriteLine("wwwwwwwwwwwwww0.1");
				try
				{
					Console.WriteLine("0.1");
					IntPtr dpy = Lock();
					XEvent xevent;
					inMainLoop = true;
					Console.WriteLine("0.2");
					while(!quit)
					{
						// Do we have pending expose events to process?
						if(pendingExposes)
						{
							// If there are still events in the queue,
							// then process them before the exposes.
							Console.WriteLine("1");

							if(Xlib.XEventsQueued
									(dpy, 2 /* QueuedAfterFlush */) != 0)
							{
								// Read the next event and dispatch it.
								Xlib.XNextEvent(dpy, out xevent);
								DispatchEvent(ref xevent);
							}
							else
							{
								Console.WriteLine("2");
								// Process the pending expose events.
								InputOutputWidget widget;
								while(exposeList != null)
								{
									widget = exposeList;
									exposeList = exposeList.nextExpose;
									widget.Expose();
								}
								pendingExposes = false;
							}
						}
						else
						{
							// Wait for the next event.
							Console.WriteLine("3");
							Xlib.XNextEvent(dpy, out xevent);
							DispatchEvent(ref xevent);
						}
					}
				}
				finally
				{
					inMainLoop = false;
					Unlock();
				}
			}

	/// <summary>
	/// <para>Tell the main event loop on this display to quit.</para>
	/// </summary>
	public void Quit()
			{
				quit = true;
			}

	// Dispatch an event that occurred on this display.  We currently
	// have the display lock.
	private void DispatchEvent(ref XEvent xevent)
			{
				// Record the time at which the event occurred.  We need
				// this to process keyboard and pointer grabs correctly.
				switch(xevent.type)
				{
					case EventType.KeyPress:
					case EventType.KeyRelease:
					{
						knownEventTime = xevent.xkey.time;
					}
					break;

					case EventType.ButtonPress:
					case EventType.ButtonRelease:
					{
						knownEventTime = xevent.xbutton.time;
					}
					break;

					case EventType.MotionNotify:
					{
						knownEventTime = xevent.xmotion.time;
					}
					break;

					case EventType.EnterNotify:
					case EventType.LeaveNotify:
					{
						knownEventTime = xevent.xcrossing.time;
					}
					break;

					case EventType.PropertyNotify:
					{
						knownEventTime = xevent.xproperty.time;
					}
					break;

					case EventType.SelectionClear:
					{
						knownEventTime = xevent.xselectionclear.time;
					}
					break;

					case EventType.SelectionNotify:
					{
						knownEventTime = xevent.xselection.time;
					}
					break;

					case EventType.SelectionRequest:
					{
						knownEventTime = xevent.xselectionrequest.time;
					}
					break;

					default:
					{
						// We don't have a time value for this event.
						knownEventTime = Xlib.Time.CurrentTime;
					}
					break;
				}

				// Find the widget that should process the event.
				Widget widget = (Widget)(handleMap[(int)(xevent.window)]);

				// Dispatch the event to the widget.
				if(widget != null)
				{
					widget.DispatchEvent(ref xevent);
				}
			}

	// Simulate an out of memory exception.
	internal static void OutOfMemory()
			{
				// Try to perform a memory allocation that we know
				// the engine will reject as being too big.
				int[] x = new int [0x7FFFFFFF];
			}

	// Retrieve or create a standard cursor.  Call with the display lock.
	internal Xlib.Cursor GetCursor(CursorType type)
			{
				uint shape = (uint)type;
				if(shape >= (uint)(CursorType.XC_num_glyphs))
				{
					shape = (uint)(CursorType.XC_X_cursor);
				}
				Xlib.Cursor cursor = cursors[(int)shape];
				if(cursor != Xlib.Cursor.Zero)
				{
					return cursor;
				}
				cursor = cursors[(int)shape] =
					Xlib.XCreateFontCursor(dpy, shape);
				return cursor;
			}

	// Add an input/output widget to the pending expose list.
	internal void AddPendingExpose(InputOutputWidget widget)
			{
				widget.nextExpose = exposeList;
				exposeList = widget;
				pendingExposes = true;
			}

	// Remove an input/output widget from the pending expose list.
	internal void RemovePendingExpose(InputOutputWidget widget)
			{
				InputOutputWidget current, prev;
				current = exposeList;
				prev = null;
				while(widget != null && current != widget)
				{
					prev = current;
					current = current.nextExpose;
				}
				if(current != null)
				{
					if(prev != null)
					{
						prev.nextExpose = current.nextExpose;
					}
					else
					{
						exposeList = current.nextExpose;
					}
				}
			}

} // class Display

} // namespace XWindows
